//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GeniviDevicePackageInterface.h"
#include <CanderaPlatform/Device/Genivi/GeniviDevicePackageDescriptor.h>
#include <CanderaPlatform/Device/Genivi/GeniviDisplay.h>

#include <CanderaPlatform/Device/Genivi/GeniviWindowSurface.h>
#include <CanderaPlatform/Device/Genivi/GeniviFrameBufferObject.h>

#include <CanderaPlatform/Device/Common/Internal/GraphicDeviceUnitConfiguration.h>
#include <FeatStd/Util/PointerUtil.h>

namespace Candera
{

GeniviDisplay* GeniviDevicePackageInterface::GetDisplay(Int displayId) const
{
    if ((displayId < 0) || (displayId >= MaxDisplayCount)) {
        return 0;
    }

    return m_displays[displayId];
}

void GeniviDevicePackageInterface::DestroyDisplay(Int displayId)
{
    if ((displayId < 0) || (displayId >= MaxDisplayCount) || (m_displays[displayId] == 0)) {
        return;
    }

    m_displays[displayId]->Unload();

    CANDERA_DELETE(m_displays[displayId]);
    m_displays[displayId] = 0;
}

GeniviDevicePackageInterface& GeniviDevicePackageInterface::GetInstance()
{
    static GeniviDevicePackageInterface s_instance;
    return s_instance;
}

GeniviDevicePackageInterface::GeniviDevicePackageInterface()
{
    MemoryPlatform::Set(m_displays, 0, sizeof(m_displays));
}

GeniviDevicePackageInterface::~GeniviDevicePackageInterface()
{
    for (Int displayId = 0; displayId < MaxDisplayCount; displayId++) {
        DestroyDisplay(displayId);
    }
}

Display* DevicePackageInterface::CreateDisplay(Int displayId)
{
    if (!GeniviDevicePackageInterface::GetInstance().CreateDisplay(displayId)) {
        return 0;
    }
    return GeniviDevicePackageInterface::GetInstance().GetDisplay(displayId);
}

void DevicePackageInterface::DestroyDisplay(Display* display)
{
    FEATSTD_LINT_SYMBOL(818, display, "could be made const, accepted to maintain interface compatibility")
    if (display != 0) {
        GeniviDevicePackageInterface::GetInstance().DestroyDisplay(display->GetDisplayId());
    }
}

Display* DevicePackageInterface::GetDisplay(Int displayId)
{
    return GeniviDevicePackageInterface::GetInstance().GetDisplay(displayId);
}


FEATSTD_LINT_NEXT_EXPRESSION(1573, "the generic function is a local function thus there is no risk of unconsidered ADL lookup")
template<typename Type>
static void SetGeniviLayer(Type& dst, const Int* configuration)
{
    GraphicDeviceUnitWidthConfiguration widthConfig(configuration);
    GraphicDeviceUnitHeightConfiguration heightConfig(configuration);

    if (!widthConfig.IsDefault()) {
        dst.SetWidth(widthConfig[GduWidthAttr]);
    }
    if (!heightConfig.IsDefault()) {
        dst.SetHeight(heightConfig[GduHeightAttr]);
    }
}

FEATSTD_LINT_NEXT_EXPRESSION(1573, "the generic function is a local function thus there is no risk of unconsidered ADL lookup")
template<typename Type>
static void SetGeniviEgl(Type& dst, const Int* configuration)
{
    GraphicDeviceUnitConfiguration<static_cast<Int>(GduRedSizeAttr)> redSizeConfig(configuration);
    GraphicDeviceUnitConfiguration<static_cast<Int>(GduGreenSizeAttr)> greenSizeConfig(configuration);
    GraphicDeviceUnitConfiguration<static_cast<Int>(GduBlueSizeAttr)> blueSizeConfig(configuration);
    GraphicDeviceUnitConfiguration<static_cast<Int>(GduAlphaSizeAttr)> alphaSizeConfig(configuration);

    if (!redSizeConfig.IsDefault()) {
        dst.SetRedSize(redSizeConfig[GduRedSizeAttr]);
    }
    if (!greenSizeConfig.IsDefault()) {
        dst.SetGreenSize(greenSizeConfig[GduGreenSizeAttr]);
    }
    if (!blueSizeConfig.IsDefault()) {
        dst.SetBlueSize(blueSizeConfig[GduBlueSizeAttr]);
    }
    if (!alphaSizeConfig.IsDefault()) {
        dst.SetAlphaSize(alphaSizeConfig[GduAlphaSizeAttr]);
    }
}

GraphicDeviceUnit* DevicePackageInterface::CreateGraphicDeviceUnit(GraphicDeviceUnitTypeHandle type, const Int* configuration)
{
    GraphicDeviceUnit* result;

    GraphicDeviceUnitWidthConfiguration widthConfig(configuration);
    GraphicDeviceUnitHeightConfiguration heightConfig(configuration);

    switch (type) {
        case c_geniviWindowSurfaceType: {
            GeniviWindowSurface* gdu = CANDERA_NEW(GeniviWindowSurface);
            if (gdu != 0) {
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1013, GetSupportProperties, "False positive because GetSupportProperties does seem to be a member of GeniviWindowSurface through GduBase via inheritence and templates.");
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::SetGeniviLayer, "False positive because 1st argument is matched with a GeniviSurfaceProperties object and 2nd by const Int*.");
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1055, Candera::GetSupportProperties, "False positive because GetSupportProperties seems to be declared in GduBase.");
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::GetSupportProperties, "False positive because no Candera::GetSupportProperties is called.");
                SetGeniviLayer(gdu->GetSupportProperties(), configuration);
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1013, GetProperties, "False positive because GetProperties does seem to be a member of GeniviWindowSurface through GduBase via inheritence and templates.");
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::SetGeniviEgl, "False positive because 1st argument is matched with an object of class EglAntiAliasedWindowSurfaceProperties and 2nd by const Int*.");
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1055, Candera::GetProperties, "False positive because GetProperties seems to be a member of GeniviWindowSurface through GduBase via inheritence and templates.");
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::GetProperties, "False positive because no Candera::GetProperties is called.");
                SetGeniviEgl(gdu->GetProperties(), configuration);
            }
            result = FeatStd::Internal::PointerToPointer<GraphicDeviceUnit*>(gdu);
            break;
        }

        case c_geniviFrameBufferObjectType: {
            GeniviFrameBufferObject* gdu = CANDERA_NEW(GeniviFrameBufferObject);
            if (gdu != 0) {
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1013, GetProperties, "False positive because GetProperties does seem to be a member of GeniviFrameBufferObject through GduBase via inheritence and templates.");
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::SetGeniviLayer, "False positive because 1st argument is matched with an object of class derived from GeniviFrameBufferProperties and 2nd by const Int*.");
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::GetProperties, "False positive because no Candera::GetProperties is called.");
                SetGeniviLayer(gdu->GetProperties(), configuration);
            }
            result = FeatStd::Internal::PointerToPointer<GraphicDeviceUnit*>(gdu);
            break;
        }

        default:
            result = 0;
            break;
    }

    return result;
}

void DevicePackageInterface::DestroyGraphicDeviceUnit(GraphicDeviceUnit* object)
{
    CANDERA_DELETE(object);
}

}
