//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlInclude.h>
#include <CanderaPlatform/Device/Common/OpenGLES20/GlTypeMapper.h>

#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>

#include <Candera/Engine3D/Core/Texture.h>
#include <Candera/Engine3D/Core/TextureImage.h>


#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <Candera/Engine3D/Core/RenderBuffer.h>
#include <Candera/Engine3D/Core/ShaderParamNames.h>

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Diagnostics/VideoMemoryStatistic.h>

#include <Candera/System/Monitor/MonitorPublicIF.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/System/Monitor/GlobalExperimentPublicIF.h>

#include <Candera/EngineBase/Common/Color.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlTrace.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>
#include <CanderaPlatform/Device/Common/Base/DirectTextureImage.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>

#include <CanderaPlatform/Device/Common/EGL/EglKhrExternalTextureImage.h>
#include <CanderaPlatform/Device/Common/EGL/EglWrapper.h>

#include <CanderaPlatform/Device/Common/Internal/EGL/EglExtensions.h>

#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
#include <CanderaPlatform/Device/Common/OpenGLES/GlShaderStorageProvider.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlShaderStorageInterface.h>
#endif

#define CANDERA_RENDERDEVICE_CLEAR_EGL_ERRORS() GLuint eglerr = eglGetError(); while (eglerr != EGL_SUCCESS) eglerr = eglGetError();
#define CANDERA_RENDERDEVICE_CLEAR_GL_ERRORS() GLuint err = glGetError(); while (err != GL_NO_ERROR) err = glGetError();

#include <CanderaPlatform/OS/PerfCounterPlatform.h>
#if (defined FEATSTD_PLATFORM_OS_Posix)
#include <sys/types.h>
#include <unistd.h>
#endif

#ifdef MONITOR_RECORDING_OPENGL_ENABLED
    #define CANDERA_RENDERDEVICE_OPENGL_RECORDER(funcName) \
    CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::OpenGL, #funcName));
#else
    #define CANDERA_RENDERDEVICE_OPENGL_RECORDER(funcName)
#endif

#define CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(funcName, params) \
    { \
        CANDERA_RENDERDEVICE_OPENGL_RECORDER(funcName) \
        MONITOR_FRAME_DEBUGGER_OPENGL_CALL(funcName) \
        funcName params; \
    }

FEATSTD_LINT_MACRO(666, CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL, "Expression with side effects passed to repeated parameter 1: funcName used to obtain its stringified name, and then calls it, effectively evaluating it only once")
#define CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(value, funcName, params) \
    { \
        CANDERA_RENDERDEVICE_OPENGL_RECORDER(funcName) \
        MONITOR_FRAME_DEBUGGER_OPENGL_CALL(funcName) \
        value = funcName params; \
    }

namespace Candera {

    FEATSTD_LINT_CURRENT_SCOPE(1576, "Violates MISRA C++ 2008 Required Rule 14-7-3: specialization of function templates moved to seperate files on purpose")

    using namespace FeatStd::Internal;
    using namespace Internal;
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);

    // Default floating point sampling parameters
    const Float RenderDevice::c_defaultMaxLod = 1000.0F;
    const Float RenderDevice::c_defaultMinLod = -1000.0F;

    // Private light list
    const RenderDevice::LightList* RenderDevice::m_activeLights = 0;
    const Light* RenderDevice::m_activeLightIterator = 0;

    // Private render state cache that is set each time a render target gets activated.
    RenderStateCache* RenderDevice::m_currentRenderStateCache = 0;


    /**
     * The struct RenderStateCache minimizes render state updates in OpenGL. Thus, only new render states that differ
     * from cache are forwarded to respectively retrieved from OpenGL.
     * A RenderStateCache is related to a certain context and its belonging render targets. Therefore,
     * in the course of activating a render target, the associated render state cache is set as current in the class RenderDevice.
     */
    struct RenderStateCache
    {
        RenderMode::Culling culling;
        //Dithering
        UInt32 isDitheringEnabled;
        //Winding
        RenderMode::Winding frontFace;
        //Color
        bool isRedWriteEnabled;
        bool isGreenWriteEnabled;
        bool isBlueWriteEnabled;
        bool isAlphaWriteEnabled;
        //Depth buffer
        UInt32 isDepthWriteEnabled;
        UInt32 isDepthTestEnabled;
        RenderMode::ComparisonFunction depthComparisonFunction;
        Float depthBiasScaleFactor;
        Float depthBiasUnits;
        //Stencil buffer
        UInt32 isStencilTestEnabled;
        GLenum stencilFunctionBackFace;
        GLint stencilFunctionRefValueBackFace;
        GLuint stencilFunctionMaskBackFace;
        GLenum stencilFunctionFrontFace;
        GLint stencilFunctionRefValueFrontFace;
        GLuint stencilFunctionMaskFrontFace;
        GLenum stencilOperationStencilFailBackFace;
        GLenum stencilOperationDepthFailBackFace;
        GLenum stencilOperationDepthPassBackFace;
        GLenum stencilOperationStencilFailFrontFace;
        GLenum stencilOperationDepthFailFrontFace;
        GLenum stencilOperationDepthPassFrontFace;
        GLuint stencilWriteMaskBackFace;
        GLuint stencilWriteMaskFrontFace;
        //Scissoring
        UInt32 isScissorTestEnabled;
        //Blending
        UInt32 isBlendingEnabled; // Specifies if blending is enabled or not.
        RenderMode::BlendFactor blendSrcRGB; // Blending coefficient is used for source RGB blending factor.
        RenderMode::BlendFactor blendDstRGB; // Blending coefficient is used for destination RGB blending factor.
        RenderMode::BlendOperation blendOpRGB; // Operation is used for RGB blend equation.
        RenderMode::BlendFactor blendSrcAlpha; // Blending coefficient is used for source Alpha blending factor.
        RenderMode::BlendFactor blendDstAlpha; // Blending coefficient is used for destination Alpha blending factor.
        RenderMode::BlendOperation blendOpAlpha; // Operation is used for Alpha blend equation.
        Color blendColor; // Specifies the RGBA values for the constant blending color.
        //Multisample Antialiasing
        UInt32 isSampleCoverageMaskEnabled; // Specifies if customized sample coverage mask for multisample antialiasing is enabled or not.
        UInt32 isSampleAlphaToCoverageMaskEnabled; // Specifies if alpha value is used to determine an additional sample coverage mask.
        Float sampleCoverage; // Specifies a value in the range [0,1] that is converted into a multisample coverage mask.
        bool isSampleCoverageInverted; // Specifies if the multisample coverage mask is inverted.

        // Viewport
        GLint viewportX;
        GLint viewportY;
        GLsizei viewportWidth;
        GLsizei viewportHeight;

        // Scissor
        GLint scissorX;
        GLint scissorY;
        GLsizei scissorWidth;
        GLsizei scissorHeight;

        // Clear states
        Color clearColor;
        Float clearDepth;
        Int32 clearStencil;

        // Shaders
        Handle activeShaderHandle; // Handle of current used program handle.

        // Textures
        UInt32 activeTextureUnit; // Index of active texture unit [0..(MaxTextureUnitsSupportedByCandera-1)]
        Handle activeTextureHandle[CANDERA_MAX_TEXTURE_UNIT_COUNT][3]; // Handle of active texture in each unit and target combination.

        // VertexBuffers
        Handle activeVertexBufferHandle; // Handle of active array buffer.
        Handle activeIndexBufferHandle; // Handle of active index array buffer.

        UInt32 enabledVertexAttribs; // bitfield for enabled state (i.e. max 32 vertex attributes supported)

        /* Constructor */
        RenderStateCache() :
            culling(RenderMode::NoCulling),  // GL_CULL_FACE: GL_FALSE, GL_CULL_FACE_MODE = GL_BACK
            isDitheringEnabled(GL_TRUE),
            frontFace(RenderMode::CounterClockWise), // GL_CCW
            isRedWriteEnabled(true),     // GL_TRUE
            isGreenWriteEnabled(true),   // GL_TRUE
            isBlueWriteEnabled(true),    // GL_TRUE
            isAlphaWriteEnabled(true),   // GL_TRUE
            isDepthWriteEnabled(GL_TRUE),
            isDepthTestEnabled(GL_FALSE),
            depthComparisonFunction(RenderMode::CompareLess), //GL_LESS
            depthBiasScaleFactor(0.0F),
            depthBiasUnits(0.0F),
            isStencilTestEnabled(GL_FALSE),
            stencilFunctionBackFace(GL_ALWAYS),
            stencilFunctionRefValueBackFace(0),
            stencilFunctionMaskBackFace(~static_cast<GLuint>(0)),
            stencilFunctionFrontFace(GL_ALWAYS),
            stencilFunctionRefValueFrontFace(0),
            stencilFunctionMaskFrontFace(~static_cast<GLuint>(0)),
            stencilOperationStencilFailBackFace(GL_KEEP),
            stencilOperationDepthFailBackFace(GL_KEEP),
            stencilOperationDepthPassBackFace(GL_KEEP),
            stencilOperationStencilFailFrontFace(GL_KEEP),
            stencilOperationDepthFailFrontFace(GL_KEEP),
            stencilOperationDepthPassFrontFace(GL_KEEP),
            stencilWriteMaskBackFace(~static_cast<GLuint>(0)),
            stencilWriteMaskFrontFace(~static_cast<GLuint>(0)),
            isScissorTestEnabled(GL_FALSE),
            isBlendingEnabled(GL_FALSE),
            blendSrcRGB(RenderMode::One), //GL_ONE
            blendDstRGB(RenderMode::Zero), //GL_ZERO
            blendOpRGB(RenderMode::Add), //GL_FUNC_ADD
            blendSrcAlpha(RenderMode::One), //GL_ONE
            blendDstAlpha(RenderMode::Zero), //GL_ZERO
            blendOpAlpha(RenderMode::Add), //GL_FUNC_ADD
            blendColor(0.0F, 0.0F, 0.0F, 0.0F),
            isSampleCoverageMaskEnabled(GL_FALSE),
            isSampleAlphaToCoverageMaskEnabled(GL_FALSE),
            sampleCoverage(1.0F),
            isSampleCoverageInverted(false),
            viewportX(0),
            viewportY(0),
            viewportWidth(0),
            viewportHeight(0),
            scissorX(0),
            scissorY(0),
            scissorWidth(0),
            scissorHeight(0),
            clearColor(0.0F, 0.0F, 0.0F, 1.0F),
            clearDepth(1.0F),
            clearStencil(0),
            activeShaderHandle(0),
            activeTextureUnit(0),
            FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4351, All texture handles will be zero initialized)
            activeTextureHandle(),
            activeVertexBufferHandle(0),
            activeIndexBufferHandle(0),
            enabledVertexAttribs(0),
            m_next(0),
            m_previous(0)
        {
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1938, "accesses global data [MISRA C++ Rule 12-8-1]")
            m_next = m_head;
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1938, "accesses global data [MISRA C++ Rule 12-8-1]")
            if (0 != m_head) {
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1938, "accesses global data [MISRA C++ Rule 12-8-1]")
                m_head->m_previous = this;
            }
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1938, "accesses global data [MISRA C++ Rule 12-8-1]")
            m_head = this;
        }

        ~RenderStateCache()
        {
            if (m_head == this) {
                if (0 != m_next) {
                    m_head = m_next;
                }
                else {
                    m_head = m_previous;
                }
            }
            if (0 != m_next) {
                m_next->m_previous = m_previous;
            }
            if (0 != m_previous) {
                m_previous->m_next = m_next;
            }
            m_next = 0;
            m_previous = 0;
        }

        void ResetActiveTextureHandles(Handle textureHandle) const
        {
            RenderStateCache* current = m_head;
            while (0 != current) {
                // Reset the active texture cache entry for all contexts.
                // This is a workaround for users of the RenderDevice who fail to
                // up-/unload textures in the same context.
                current->ResetActiveTextureHandlesImpl(textureHandle, this);
                current = current->m_next;
            }
        }

    private:
        void ResetActiveTextureHandlesImpl(Handle textureHandle, const RenderStateCache* trigger)
        {
            for (UInt unit = 0; unit < sizeof(activeTextureHandle) / sizeof(activeTextureHandle[0]); unit++) {
                for (UInt target = 0; target < sizeof(activeTextureHandle[0]) / sizeof(activeTextureHandle[0][0]); target++) {
                    if (activeTextureHandle[unit][target] == textureHandle) {
                        activeTextureHandle[unit][target] = 0;
                        if (this != trigger) {
                            // Depending on the actual OpenGL implementation, in non-shared contexts this
                            // message can be a false positive. The OpenGL specification does not guarantee that
                            // texture names are unique across contexts, so the same texture name could potentially
                            // refer to a different texture.
                            FEATSTD_LOG_ERROR("Unload of texture in wrong context!");
                        }
                    }
                }
            }
        }


        static RenderStateCache* m_head;
        RenderStateCache* m_next;
        RenderStateCache* m_previous;

        // Disallow copies of RenderStateCache objects.
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::RenderStateCache::RenderStateCache, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            RenderStateCache(const RenderStateCache& /*rhs*/);
        RenderStateCache& operator=(const RenderStateCache& /*rhs*/);
    };

    RenderStateCache* RenderStateCache::m_head = 0;

    /**
     * The struct TextureStateCache minimizes texture state updates in OpenGL.
     * Texture state caching differs from render state caching by the fact that OpenGL texture states
     * are stored in the texture object itself. In Candera, each TextureImage object stores one TextureStateCache
     * for each (unshared) context.
     */
    struct TextureStateCache {
        bool isMipMappingEnabled;
        bool isTextureFilterInitialized;    //On some GPUs it is mandatory to set the texture states at least once, even if default values are used.
        bool isWrapModeInitialized;
        bool isMaxAnisotropyInitialized;
        Texture::MipMapFilter mipMapFilter;  // MipMapFilter defines filter type used for mipmapping.
        Texture::MinMagFilter minFilter;     // MinMagFilter defines filter type used for minification and magnification.
        Texture::MinMagFilter magFilter;     // MinMagFilter defines filter type used for minification and magnification.

        Texture::WrapMode wrapModeU; // Wrap mode of active texture in u-direction
        Texture::WrapMode wrapModeV; // Wrap mode of active texture in v-direction

        Float maxAnisotropy;                // The maximum degree of anisotropy for anisotropic filtering. Default value is 1.0F.

        TextureStateCache() :
            isMipMappingEnabled(false),
            isTextureFilterInitialized(false),
            isWrapModeInitialized(false),
            isMaxAnisotropyInitialized(false),
            mipMapFilter(Texture::MipMapLinear),
            minFilter(Texture::MinMagNearest),
            magFilter(Texture::MinMagLinear),
            wrapModeU(Texture::Repeat),
            wrapModeV(Texture::Repeat),
            maxAnisotropy(1.0F)
        {
        }

        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(TextureStateCache);
    };

namespace {

class ContextResourcePoolVisitor
{
public:
    void Visit()
    {
        EGLContext context = EGLWrapper::GetInstance().GetCurrentContext();
        ContextResourcePool::ContextProviderIterator contextIterator =
            ContextResourcePool::GetActive().GetContextProviderIterator();
        ContextResourcePool::ContextProviderIterator orginalContextIterator;
        while (contextIterator.IsValid()) {
            if (contextIterator->GetContextHandle() == context) {
                orginalContextIterator = contextIterator;
            }
            else {
                if (contextIterator->Activate()) {
                    VisitContext();
                }
            }
            ++contextIterator;
        }
        if (orginalContextIterator.IsValid()) {
            FEATSTD_UNUSED(orginalContextIterator->Activate());
        }
        VisitContext();
    }

protected:
    virtual void VisitContext() = 0;
};

}


RenderStateCache* RenderDevice::CreateRenderStateCache()
{
    return FEATSTD_NEW(RenderStateCache);
}

void RenderDevice::DestroyRenderStateCache(RenderStateCache* renderStateCache)
{
    if (m_currentRenderStateCache == renderStateCache) {
        m_currentRenderStateCache = 0;
    }
    FEATSTD_DELETE(renderStateCache);
}

void RenderDevice::OnContextUnload(SizeType contextIndex)
{
    bool success = Renderer::FlushTexturePool(contextIndex);
    FEATSTD_DEBUG_ASSERT(success);
    FEATSTD_UNUSED(success);
}

RenderStateCache* RenderDevice::GetCurrentRenderStateCache()
{
#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
    return m_currentRenderStateCache;
#else
    return 0;
#endif
}

void RenderDevice::SetCurrentRenderStateCache(RenderStateCache* currentRenderStateCache)
{
#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
    m_currentRenderStateCache = currentRenderStateCache;
#else
    (void) currentRenderStateCache; /* avoid warning about unused parameter */
#endif
}

bool RenderDevice::IsRenderStateCacheEnabled()
{
#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
    return true;
#else
    return false;
#endif
}

TextureStateCache* RenderDevice::CreateTextureStateCache()
{
#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
    return FEATSTD_NEW_ARRAY(TextureStateCache, CANDERA_MAX_CONTEXT_COUNT);
#else
    return 0;
#endif
}

void RenderDevice::DestroyTextureStateCache(TextureStateCache* textureStateCache)
{
    if (textureStateCache != 0) {
        FEATSTD_DELETE_ARRAY(textureStateCache);
    }
}

void RenderDevice::SetTextureStateCache(TextureImage& textureImage, TextureStateCache* textureStateCache)
{
    DestroyTextureStateCache(textureImage.GetStateCache());
    textureImage.SetStateCache(textureStateCache);
}

static TextureStateCache* GetTextureStateCache(const MemoryManagement::SharedPointer<TextureImage>& textureImage)
{
    if ((!textureImage.PointsToNull()) && (textureImage->GetStateCache() != 0)) {
        return textureImage->GetStateCache() + ContextResourcePool::GetActive().GetIndex();
    }

    return 0;
}

bool RenderDevice::IsInstancingSupported()
{
    return false;
}

RenderDevice::RenderDevice()
{
}

RenderDevice::~RenderDevice()
{
    m_activeLights = 0;
    m_currentRenderStateCache = 0;
}

bool RenderDevice::ActivateScissorRectangle(const Rectangle& rectangle, const RenderTarget3D* renderTarget)
{
    return SetScissorRectangle(rectangle, static_cast<Float>(renderTarget->GetActualWidth()), static_cast<Float>(renderTarget->GetActualHeight()));
}

bool RenderDevice::SetScissorRectangle(const Rectangle& rectangle, Float renderTargetWidth, Float renderTargetHeight)
{
    // Create absolute OpenGL pixel coordinates out of normalized rectangle.
    GLint x = static_cast<GLint>(rectangle.GetLeft() * renderTargetWidth + 0.5F);
    GLint y = static_cast<GLint>((renderTargetHeight - ((rectangle.GetTop() + rectangle.GetHeight()) * renderTargetHeight)) + 0.5F);
    GLsizei width = static_cast<GLsizei>(rectangle.GetWidth() * renderTargetWidth + 0.5F);
    GLsizei height = static_cast<GLsizei>(rectangle.GetHeight() * renderTargetHeight + 0.5F);

    if (GetCurrentRenderStateCache() != 0) {
        if ((GetCurrentRenderStateCache()->scissorX != x) || (GetCurrentRenderStateCache()->scissorY != y) ||
            (GetCurrentRenderStateCache()->scissorWidth != width) || (GetCurrentRenderStateCache()->scissorHeight != height)) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glScissor, (x, y, width, height));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glScissor failed.");

            GetCurrentRenderStateCache()->scissorX = x;
            GetCurrentRenderStateCache()->scissorY = y;
            GetCurrentRenderStateCache()->scissorWidth = width;
            GetCurrentRenderStateCache()->scissorHeight = height;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glScissor, (x, y, width, height));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glScissor failed.");
    }

    return true;
}

const Rectangle RenderDevice::GetScissorRectangle(const RenderTarget3D* renderTarget)
{
    Rectangle rectangle(0.0F, 0.0F, 0.0F, 0.0F);
    GLint box[4] = {0, 0, 0, 0};
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_SCISSOR_BOX, box));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv(GL_SCISSOR_BOX, &box) failed.")

    // Avoid division through zero.
    if ((renderTarget->GetActualWidth() != 0) && (renderTarget->GetActualHeight() != 0)) {
        Float x = static_cast<Float>(box[0]);
        Float y = static_cast<Float>(box[1]);
        Float width = static_cast<Float>(box[2]);
        Float height = static_cast<Float>(box[3]);

        // Normalize OpenGL's absolute pixel coordinates, where absolute x:0, y:0 is left, bottom.
        rectangle.SetLeft(x / static_cast<Float>(renderTarget->GetActualWidth()));
        rectangle.SetTop((static_cast<Float>(renderTarget->GetActualHeight()) - (y + height)) /
            static_cast<Float>(renderTarget->GetActualHeight()));
        rectangle.SetWidth(width / static_cast<Float>(renderTarget->GetActualWidth()));
        rectangle.SetHeight(height / static_cast<Float>(renderTarget->GetActualHeight()));
    }

    return rectangle;
}

bool RenderDevice::UploadRenderBuffer(RenderBuffer& renderBuffer)
{
    GLuint renderBufferName;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenRenderbuffers, (1, &renderBufferName));
    CANDERA_GL_CHECK_ERROR_CONDITION(renderBufferName != 0, "glGenRenderbuffers failed.")

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindRenderbuffer, (GL_RENDERBUFFER, renderBufferName));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindRenderbuffer failed.");

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glRenderbufferStorage, (GL_RENDERBUFFER, GlTypeMapper::MapRenderBufferFormat(renderBuffer.GetFormat()),
                              renderBuffer.GetWidth(), renderBuffer.GetHeight()));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glRenderbufferStorage failed.");

    renderBuffer.SetVideoMemoryHandle(renderBufferName);
    return true;
}

bool RenderDevice::UnloadRenderBuffer(RenderBuffer& renderBuffer)
{
    GLuint renderBufferName = static_cast<GLuint>(renderBuffer.GetVideoMemoryHandle());
    if (renderBufferName == 0){
        FEATSTD_LOG_ERROR("Unload render buffer failed, render buffer name invalid.");
        return false;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteRenderbuffers, (1, &renderBufferName));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteRenderbuffers failed.");
    renderBuffer.SetVideoMemoryHandle(0);
    return true;
}

bool RenderDevice::ActivateRenderBuffer(const RenderBuffer& renderBuffer)
{
    GLuint renderBufferName = static_cast<GLuint>(renderBuffer.GetVideoMemoryHandle());
    if (renderBufferName == 0){
        FEATSTD_LOG_ERROR("Activate render buffer failed, render buffer name invalid.");
        return false;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindRenderbuffer, (GL_RENDERBUFFER, renderBufferName));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindRenderbuffer failed.");
    return true;
}

//
//// Internal function used by RenderDevice::UploadFrameBufferRenderTarget
//bool UploadFrameBufferRenderTargetAttachment(const FrameBufferRenderTarget::Attachment& attachment)
//{
//    Handle videoMemoryHandle;
//    if (attachment.attachmentType == FrameBufferRenderTarget::TextureImageType) {
//        videoMemoryHandle = attachment.textureImage != 0 ? attachment.textureImage->GetVideoMemoryHandle() : 0;
//        glFramebufferTexture2D(GL_FRAMEBUFFER,
//                               GlTypeMapper::MapAttachmentPoint(attachment.attachmentPoint),
//                               GL_TEXTURE_2D,
//                               videoMemoryHandle,
//                               0 /* mipmap level must be zero */);
//    }
//    else if (attachment.attachmentType == FrameBufferRenderTarget::RenderBufferType) {
//        videoMemoryHandle = attachment.renderBuffer != 0 ? attachment.renderBuffer->GetVideoMemoryHandle() : 0;
//        glFramebufferRenderbuffer(GL_FRAMEBUFFER,
//                                  GlTypeMapper::MapAttachmentPoint(attachment.attachmentPoint),
//                                  GL_RENDERBUFFER,
//                                  videoMemoryHandle);
//    }
//    // No upload if attachment.attachmentType == FrameBufferRenderTarget::InvalidType.
//    CANDERA_GL_CHECK_ERROR_LOG_ERROR("Upload of framebuffer attachement failed.");
//    return true;
//}
//
//bool RenderDevice::UploadFrameBufferRenderTarget(FrameBufferRenderTarget& frameBufferRenderTarget)
//{
//    bool rc = true;
//    bool rcTemp = true;
//    GLuint frameBufferRenderTargetHandle;
//
//    // Generate Framebuffer handle.
//    glGenFramebuffers(1, &frameBufferRenderTargetHandle);
//    CANDERA_GL_CHECK_ERROR_CONDITION(frameBufferRenderTargetHandle != 0, "rcGenFramebuffers failed.")
//
//    // Create framebuffer object.
//    glBindFramebuffer(GL_FRAMEBUFFER, frameBufferRenderTargetHandle);
//    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindFramebuffer failed.");
//
//    // Upload color attachment.
//    rcTemp = UploadFrameBufferRenderTargetAttachment(frameBufferRenderTarget.GetAttachment(FrameBufferRenderTarget::ColorAttachment));
//    if (!rcTemp) {
//        rc = rcTemp;
//    }
//
//    // Upload depth attachment.
//    rcTemp = UploadFrameBufferRenderTargetAttachment(frameBufferRenderTarget.GetAttachment(FrameBufferRenderTarget::DepthAttachment));
//    if (!rcTemp) {
//        rc = rcTemp;
//    }
//
//    // Upload stencil attachment.
//    rcTemp = UploadFrameBufferRenderTargetAttachment(frameBufferRenderTarget.GetAttachment(FrameBufferRenderTarget::StencilAttachment));
//    if (!rcTemp) {
//        rc = rcTemp;
//    }
//
//    if ((glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE) || (!rc)) {
//        glDeleteFramebuffers(1, &frameBufferRenderTargetHandle);
//        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCheckFramebufferStatus failed.");
//    }
//
//    frameBufferRenderTarget.SetVideoMemoryHandle(frameBufferRenderTargetHandle);
//    return rc;
//}
//
//bool RenderDevice::UnloadFrameBufferRenderTarget(FrameBufferRenderTarget& frameBufferRenderTarget)
//{
//    GLuint frameBufferRenderTargetHandle = frameBufferRenderTarget.GetVideoMemoryHandle();
//    if (frameBufferRenderTargetHandle == 0) {
//        FEATSTD_LOG_ERROR("Render Buffer Name invalid.");
//        return false;
//    }
//    glDeleteFramebuffers(1, &frameBufferRenderTargetHandle);
//    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteFramebuffers failed.");
//    frameBufferRenderTarget.SetVideoMemoryHandle(frameBufferRenderTargetHandle); // Handle is zero after successfull deletion
//    return true;
//}
//
//bool RenderDevice::ActivateFrameBufferRenderTarget(const FrameBufferRenderTarget& frameBufferRenderTarget)
//{
//    GLuint frameBufferRenderTargetHandle = frameBufferRenderTarget.GetVideoMemoryHandle();
//    if (frameBufferRenderTargetHandle == 0) {
//        FEATSTD_LOG_ERROR("Render Buffer Name invalid.");
//        return false;
//    }
//    glBindFramebuffer(GL_FRAMEBUFFER, frameBufferRenderTargetHandle);
//    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindFramebuffer failed.");
//    return true;
//}

bool RenderDevice::ClearColorBuffer(const Color& color)
{
    if (ActivateClearColor(color) == false){
        return false;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClear, (GL_COLOR_BUFFER_BIT));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClear failed.");
    return true;
}

bool RenderDevice::ActivateClearColor(const Color& color)
{
    if (GetCurrentRenderStateCache() != 0) {
        if ( color != GetCurrentRenderStateCache()->clearColor ) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClearColor, (color.GetRed(), color.GetGreen(), color.GetBlue(), color.GetAlpha()));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClearColor failed.");
            GetCurrentRenderStateCache()->clearColor = color;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClearColor, (color.GetRed(), color.GetGreen(), color.GetBlue(), color.GetAlpha()));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClearColor failed.");
    }

    return true;
}

bool RenderDevice::ClearDepthBuffer(Float depth)
{
    bool result = ActivateDepthClearValue(depth);

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClear, (GL_DEPTH_BUFFER_BIT));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClear failed.");
    return result;
}

bool RenderDevice::ActivateDepthClearValue(Float depth)
{
    if (GetCurrentRenderStateCache() != 0) {
        if (!Math::FloatAlmostEqual(depth, GetCurrentRenderStateCache()->clearDepth)) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClearDepthf, (static_cast<GLclampf>(depth)));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClearDepthf failed.");
            GetCurrentRenderStateCache()->clearDepth = depth;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClearDepthf, (static_cast<GLclampf>(depth)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClearDepthf failed.");
    }

    return true;
}

bool RenderDevice::ClearStencilBuffer(Int32 value)
{
    FEATSTD_UNUSED(ActivateStencilClearValue(value));

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClear, (GL_STENCIL_BUFFER_BIT));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClear failed.");
    return true;
}

bool RenderDevice::ActivateStencilClearValue(Int32 value)
{
    if (GetCurrentRenderStateCache() != 0) {
        if (value != GetCurrentRenderStateCache()->clearStencil) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClearStencil, (static_cast<GLint>(value)));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClearStencil failed.");
            GetCurrentRenderStateCache()->clearStencil = value;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClearStencil, (static_cast<GLint>(value)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClearStencil failed.");
    }

    return true;
}

bool RenderDevice::ClearFrameBuffer(bool color, bool depth, bool stencil)
{
    GLbitfield mask = 0;

    if (color) {
        mask |= static_cast<GLbitfield>(GL_COLOR_BUFFER_BIT);
    }

    if (depth) {
        mask |= static_cast<GLbitfield>(GL_DEPTH_BUFFER_BIT);
    }

    if (stencil) {
        mask |= static_cast<GLbitfield>(GL_STENCIL_BUFFER_BIT);
    }

    if (mask != 0) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClear, (mask));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClear failed.");
    }
    return true;
}

bool RenderDevice::ClearCoverageBuffer()
{
#if GL_NV_coverage_sample
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glClear, (GL_COVERAGE_BUFFER_BIT_NV));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glClear failed for coverage buffer. GL error:. Is extension GL_NV_coverage_sample supported?")
    return true;
#else
    return false;
#endif
}

//returns required number of mipmap levels for the specified image dimensions.
UInt32 GetMipmapCount(UInt32 width, UInt32 height){
    UInt32 levels = 1;
    UInt32 size = (width | height);
    while (size > 1){
        ++levels;
        size >>= 1;
    }
    return levels;
}


bool IsMipMapChainComplete(const Candera::Bitmap* bitmap) {

    if(bitmap == 0) {
        return false;
    }

    Candera::Bitmap::PixelFormat format = bitmap->GetPixelFormat();

    Candera::MemoryManagement::SharedPointer<Candera::Bitmap> nextBitmap = bitmap->GetNext();

    UInt32 width = bitmap->GetWidth();
    UInt32 height = bitmap->GetHeight();

    while (nextBitmap != 0) {
        if (width > 1) {
            width >>= 1;
        }

        if (height > 1) {
            height >>= 1;
        }

        if (
            (nextBitmap->GetPixelFormat() != format)
            || (nextBitmap->GetWidth() != width)
            || (nextBitmap->GetHeight() != height)
            || (
                (!nextBitmap->HasNext())
                && ((nextBitmap->GetWidth() != 1) || (nextBitmap->GetHeight() != 1))
            )) {
            /*
             * Following conditions are checked for mipmap completeness:
             *  - Bitmap Format is always the same.
             *  - All MipMap levels are defined in having the previous dimensions divided by two.
             *  - Additionally, it is checked that all levels down to 1x1 are specified.
             */
            return false;
        }
        nextBitmap = nextBitmap->GetNext();
    }

    return true;
}

UInt8 RenderDevice::GetBitsPerPixel(Bitmap::PixelFormat pixelFormat)
{
    if (pixelFormat >= Bitmap::PixelFormatCount) {
        FEATSTD_LOG_ERROR("Pixel format not valid.");
        return 0;
    } else {
        return GlTypeMapper::MapBitmapPixelFormatToGLBitsPerPixel(pixelFormat);
    }
}

UInt32 RenderDevice::GetSize(const Bitmap* bitmap, TextureImage::TextureMemoryPool memoryPool)
{
    UInt32 size = 0;
    if ((0 != bitmap) && (memoryPool == TextureImage::VideoMemory)) {
        if (bitmap->GetPixelFormat() < Bitmap::PixelFormatCount) {
            size = GlTypeMapper::MapBitmapPixelFormatToGLBitsPerPixel(bitmap->GetPixelFormat());
            if (bitmap->IsCompressed()) {
                // All ETC formats store compressed data in 4x4 blocks. I.e. images smaller than 4x4 are stored as 4x4.
                // size calculation: https://www.khronos.org/opengles/sdk/docs/man3/html/glCompressedTexImage2D.xhtml where last value is bytes per block
                const UInt32 bytesPerBlock = (size * 16/*4x4 block*/) >> 3;
                const UInt16 minCompressedBlockSize = 4;
                const Float width = static_cast<Float>(bitmap->GetWidth()) / static_cast<Float>(minCompressedBlockSize);
                const Float height = static_cast<Float>(bitmap->GetHeight()) / static_cast<Float>(minCompressedBlockSize);
                size = (static_cast<UInt32>(Math::Ceil(width)) * static_cast<UInt32>(Math::Ceil(height)) * bytesPerBlock);
            }
            else {
                size *= static_cast<UInt32>(bitmap->GetWidth()) * static_cast<UInt32>(bitmap->GetHeight());
                size >>= 3; // divide by 8 to return size in bytes
            }
        }
        else {
            CANDERA_DEVICE_LOG_WARN("Size cannot be computed for unsupported bitmap format!");
        }
    }
    return size;
}

bool RenderDevice::UploadBitmapTextureImage(BitmapTextureImage& textureImage, UInt unit)
{
    return Renderer::UploadBitmapTextureImage(textureImage, unit, DeviceObject::NoHint);
}

bool RenderDevice::UploadBitmapTextureImageInternal(BitmapTextureImage& textureImage, UInt unit)
{
    const Bitmap* bitmap = textureImage.GetBitmap().GetPointerToSharedInstance();
    if ((bitmap == 0) || (bitmap->IsCustomPixelFormat())) {
        FEATSTD_LOG_ERROR("Upload texture image failed, bitmap is not valid.");
        return false;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glPixelStorei, (GL_UNPACK_ALIGNMENT, static_cast<GLint>(bitmap->GetPackAlignment())));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glPixelStorei failed.");

    Handle handle = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenTextures, (1, FeatStd::Internal::PointerToPointer<GLuint*>(&handle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGenTextures failed.");

    textureImage.SetVideoMemoryHandle(handle);

    //On Upload always freshly activate texture unit and bind it to new handle.
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glActiveTexture, (GlTypeMapper::MapTextureUnit(unit)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glActiveTexture failed.");

    TextureImage::TextureTargetType target = textureImage.GetTextureTargetType();
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindTexture, (GlTypeMapper::MapTextureTargetType(target), static_cast<GLuint>(handle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindTexture failed.");

    if (GetCurrentRenderStateCache() != 0) {
        //Always update the current render state cache on upload.
        GetCurrentRenderStateCache()->activeTextureUnit = unit;
        GetCurrentRenderStateCache()->activeTextureHandle[unit][target] =  handle;
    }

    {
        Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
        // This is where the real upload happens.
        if (!bitmap->IsCompressed()) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexImage2D, (GL_TEXTURE_2D, 0,
                GLint(GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(bitmap->GetPixelFormat())),
                GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()), 0,
                GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
                GlTypeMapper::MapBitmapPixelFormatToGLType(bitmap->GetPixelFormat()),
                pixelsResource.GetData()));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexImage2D failed.");
        }
        else {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompressedTexImage2D, (GL_TEXTURE_2D, 0,
                GLint(GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(bitmap->GetPixelFormat())),
                GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()), 0,
                bitmap->GetSize(), pixelsResource.GetData()));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompressedTexImage2D failed.");
        }
    }

    UInt width  = bitmap->GetWidth();
    UInt height = bitmap->GetHeight();
    UInt levels = textureImage.IsMipMappingEnabled() ? static_cast<UInt>(GetMipmapCount(width, height)) : 1;

    if (textureImage.IsMipMappingEnabled()) {
        if (bitmap->HasNext() && IsMipMapChainComplete(bitmap)) {
            // Predefined Bitmap chain is used to generate MipMap chain.
            Int32 level = 0;
            while (bitmap->HasNext() && (static_cast<UInt>(level) < levels)) {
                bitmap = bitmap->GetNext().GetPointerToSharedInstance();
                ++level;

                Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
                // This is where the real upload happens.
                if (!bitmap->IsCompressed()) {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexImage2D, (GL_TEXTURE_2D, level,
                        GLint(GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(bitmap->GetPixelFormat())),
                        GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()), 0,
                        GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
                        GlTypeMapper::MapBitmapPixelFormatToGLType(bitmap->GetPixelFormat()),
                        pixelsResource.GetData()));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexImage2D failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompressedTexImage2D, (GL_TEXTURE_2D, level,
                        GLint(GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(bitmap->GetPixelFormat())),
                        GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()), 0,
                        bitmap->GetSize(), pixelsResource.GetData()));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompressedTexImage2D failed.");
                }
            }
        }
        else {
            // MipMap is requested without predefined bitmap chain. Thus, auto-generate MipMap chain by driver.
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenerateMipmap, (GL_TEXTURE_2D));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGenerateMipmap failed.");
        }
    }

    SetTextureStateCache(textureImage, CreateTextureStateCache());
    Diagnostics::VideoMemoryStatistic::OnTextureImageUploaded(textureImage);

    return true;
}

bool RenderDevice::ReuseTexture(Handle handle, BitmapTextureImage& textureImage, UInt unit)
{
    const Bitmap* bitmap = textureImage.GetBitmap().GetPointerToSharedInstance();
    if ((bitmap == 0) || (bitmap->IsCustomPixelFormat())) {
        FEATSTD_LOG_ERROR("Upload texture image failed, bitmap is not valid.\n");
        return false;
    }

    textureImage.SetVideoMemoryHandle(handle);

    // Activate Texture to ensure correct use
    bool rc = ActivateTextureImage(textureImage, unit);
    if (!rc) {
        return rc;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glPixelStorei, (GL_UNPACK_ALIGNMENT, static_cast<GLint>(bitmap->GetPackAlignment())));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glPixelStorei failed.");

    {
        Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
        // This is where the real upload happens.
        if (!bitmap->IsCompressed()) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexSubImage2D, (GL_TEXTURE_2D, 0,
                0, 0,
                GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()),
                GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
                GlTypeMapper::MapBitmapPixelFormatToGLType(bitmap->GetPixelFormat()),
                pixelsResource.GetData()));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexSubImage2D failed.");
        }
        else {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompressedTexSubImage2D, (GL_TEXTURE_2D, 0,
                0, 0,
                GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()),
                GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
                GetSize(bitmap), pixelsResource.GetData()));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompressedTexSubImage2D failed.");
        }
    }

    if (textureImage.IsMipMappingEnabled()) {
        if (bitmap->IsMipMapChainComplete()) {
            // Predefined Bitmap chain is used to generate MipMap chain.
            Int32 level = 0;
            while (bitmap->HasNext()) {
                bitmap = bitmap->GetNext().GetPointerToSharedInstance();
                ++level;

                Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
                if (!bitmap->IsCompressed()) {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexSubImage2D, (GL_TEXTURE_2D, level,
                        0, 0,
                        GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()),
                        GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
                        GlTypeMapper::MapBitmapPixelFormatToGLType(bitmap->GetPixelFormat()),
                        pixelsResource.GetData()));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexSubImage2D failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompressedTexSubImage2D, (GL_TEXTURE_2D, level,
                        0, 0,
                        GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()),
                        GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
                        GetSize(bitmap), pixelsResource.GetData()));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompressedTexSubImage2D failed.");
                }
            }
        }
        else {
            // MipMap is requested without predefined bitmap chain. Thus, auto-generate MipMap chain by driver.
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenerateMipmap, (GL_TEXTURE_2D));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGenerateMipmap failed.");
        }
    }

    SetTextureStateCache(textureImage, CreateTextureStateCache());

    return true;
}

bool RenderDevice::UploadCubeMapTextureImage(CubeMapTextureImage& textureImage, UInt unit)
{
    const Bitmap::SharedPointer& rightFace = textureImage.GetBitmapFace(CubeMapTextureImage::CubeMapPositiveXFace);
    const Bitmap::SharedPointer& leftFace  = textureImage.GetBitmapFace(CubeMapTextureImage::CubeMapNegativeXFace);
    const Bitmap::SharedPointer& upFace    = textureImage.GetBitmapFace(CubeMapTextureImage::CubeMapPositiveYFace);
    const Bitmap::SharedPointer& downFace  = textureImage.GetBitmapFace(CubeMapTextureImage::CubeMapNegativeYFace);
    const Bitmap::SharedPointer& frontFace = textureImage.GetBitmapFace(CubeMapTextureImage::CubeMapPositiveZFace);
    const Bitmap::SharedPointer& backFace  = textureImage.GetBitmapFace(CubeMapTextureImage::CubeMapNegativeZFace);

    bool success = (leftFace != 0) && (rightFace != 0) && (upFace != 0) && (downFace != 0) && (frontFace != 0) && (backFace != 0);
    if (!success) {
        FEATSTD_LOG_ERROR("Upload of CubeMap texture image failed, bitmaps are not valid.");
        return success;
    }

    if ((leftFace->IsCustomPixelFormat()) ||
        (rightFace->IsCustomPixelFormat()) ||
        (upFace->IsCustomPixelFormat()) ||
        (downFace->IsCustomPixelFormat()) ||
        (frontFace->IsCustomPixelFormat()) ||
        (backFace->IsCustomPixelFormat())) {
        FEATSTD_LOG_ERROR("Upload of CubeMap texture image failed, pixel format for bitmaps is not valid.");
        return false;
    }

    if ((leftFace->GetWidth() != rightFace->GetWidth()) ||
        (leftFace->GetWidth() != upFace->GetWidth()) ||
        (leftFace->GetWidth() != downFace->GetWidth()) ||
        (leftFace->GetWidth() != frontFace->GetWidth()) ||
        (leftFace->GetWidth() != backFace->GetWidth()) ||
        (leftFace->GetHeight() != rightFace->GetHeight()) ||
        (leftFace->GetHeight() != upFace->GetHeight()) ||
        (leftFace->GetHeight() != downFace->GetHeight()) ||
        (leftFace->GetHeight() != frontFace->GetHeight()) ||
        (leftFace->GetHeight() != backFace->GetHeight())) {
        FEATSTD_LOG_ERROR("Upload of CubeMap texture image failed, width/height for bitmaps is not valid.");
        return false;
    }

    Handle handle = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenTextures, (1, FeatStd::Internal::PointerToPointer<GLuint*>(&handle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("Texture generation for CubeMap failed.");

    textureImage.SetVideoMemoryHandle(handle);

    //On Upload always freshly activate texture unit and bind it to new handle.
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glActiveTexture, (GlTypeMapper::MapTextureUnit(unit)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glActiveTexture failed.");

    TextureImage::TextureTargetType target = textureImage.GetTextureTargetType();
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindTexture, (GlTypeMapper::MapTextureTargetType(target), static_cast<GLuint>(handle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindTexture failed.");

    if (GetCurrentRenderStateCache() != 0) {
        //Always update the current render state cache on upload.
        GetCurrentRenderStateCache()->activeTextureUnit = unit;
        GetCurrentRenderStateCache()->activeTextureHandle[unit][target] = handle;
    }

    const Bitmap::SharedPointer& bitmap = upFace;
    UInt32 width  = bitmap->GetWidth();
    UInt32 height = bitmap->GetHeight();
    UInt32 levels = 1;

    bool hasPredefinedMipMapChain = false;
    if (textureImage.IsMipMappingEnabled()) {
        // Check if each face has a texture image with a predefined mipmap chain.
        hasPredefinedMipMapChain =
            leftFace->IsMipMapChainComplete() && rightFace->IsMipMapChainComplete() &&
            upFace->IsMipMapChainComplete() && downFace->IsMipMapChainComplete() &&
            frontFace->IsMipMapChainComplete() && backFace->IsMipMapChainComplete();
        levels = GetMipmapCount(width, height);
    }

    success = UploadCubeMapFace(CubeMapTextureImage::CubeMapPositiveXFace, rightFace.GetPointerToSharedInstance(), hasPredefinedMipMapChain, levels)
           && UploadCubeMapFace(CubeMapTextureImage::CubeMapNegativeXFace, leftFace .GetPointerToSharedInstance(), hasPredefinedMipMapChain, levels)
           && UploadCubeMapFace(CubeMapTextureImage::CubeMapPositiveYFace, upFace   .GetPointerToSharedInstance(), hasPredefinedMipMapChain, levels)
           && UploadCubeMapFace(CubeMapTextureImage::CubeMapNegativeYFace, downFace .GetPointerToSharedInstance(), hasPredefinedMipMapChain, levels)
           && UploadCubeMapFace(CubeMapTextureImage::CubeMapPositiveZFace, frontFace.GetPointerToSharedInstance(), hasPredefinedMipMapChain, levels)
           && UploadCubeMapFace(CubeMapTextureImage::CubeMapNegativeZFace, backFace .GetPointerToSharedInstance(), hasPredefinedMipMapChain, levels);

    if (textureImage.IsMipMappingEnabled() && (!hasPredefinedMipMapChain)) {
        // MipMap is requested without predefined bitmap chain. Thus, auto-generate MipMap chain by driver.
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenerateMipmap, (GL_TEXTURE_CUBE_MAP));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("Generation of MipMap chain failed.");
    }

    SetTextureStateCache(textureImage, CreateTextureStateCache());
    Diagnostics::VideoMemoryStatistic::OnTextureImageUploaded(textureImage);

    return success;
}

bool RenderDevice::UploadCubeMapFace(CubeMapTextureImage::TargetFace face, const Bitmap* bitmap, bool mipmap, UInt levels) {

    if ((0 == bitmap) || (bitmap->IsCustomPixelFormat())) {
        FEATSTD_LOG_ERROR("Upload cube map texture image failed, bitmap is not valid.");
        return false;
    }

    // Here happens the real upload for every face.
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glPixelStorei, (GL_UNPACK_ALIGNMENT, static_cast<GLint>(bitmap->GetPackAlignment())));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glPixelStorei failed.");

    {
        Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
        // This is where the real upload happens.
        if (!bitmap->IsCompressed()) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexImage2D, (GlTypeMapper::MapTextureCubeMapTargetFace(face), 0,
                GLint(GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(bitmap->GetPixelFormat())),
                GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()), 0,
                GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
                GlTypeMapper::MapBitmapPixelFormatToGLType(bitmap->GetPixelFormat()),
                pixelsResource.GetData()));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexImage2D failed.");
        }
        else {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompressedTexImage2D, (GlTypeMapper::MapTextureCubeMapTargetFace(face), 0,
                GLint(GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(bitmap->GetPixelFormat())),
                GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()), 0,
                bitmap->GetSize(), pixelsResource.GetData()));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompressedTexImage2D failed.");
        }
    }

    if (mipmap) {
        // Predefined Bitmap chain is used to generate MipMap chain.
        Int32 level = 0;
        while (bitmap->HasNext() && IsMipMapChainComplete(bitmap) && (static_cast<UInt>(level) < levels)) {
            bitmap = bitmap->GetNext().GetPointerToSharedInstance();
            ++level;

            Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
            if (!bitmap->IsCompressed()) {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexImage2D, (GlTypeMapper::MapTextureCubeMapTargetFace(face), level,
                    GLint(GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(bitmap->GetPixelFormat())),
                    GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()), 0,
                    GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
                    GlTypeMapper::MapBitmapPixelFormatToGLType(bitmap->GetPixelFormat()),
                    pixelsResource.GetData()));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexImage2D failed.");
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompressedTexImage2D, (GlTypeMapper::MapTextureCubeMapTargetFace(face), level,
                    GLint(GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(bitmap->GetPixelFormat())),
                    GLsizei(bitmap->GetWidth()), GLsizei(bitmap->GetHeight()), 0,
                    bitmap->GetSize(), pixelsResource.GetData()));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompressedTexImage2D failed.");
            }
        }
    }

    return true;
}

bool RenderDevice::UnloadBitmapTextureImage(BitmapTextureImage& textureImage)
{
    return Renderer::UnloadBitmapTextureImage(textureImage, DeviceObject::NoHint);
}

bool RenderDevice::UnloadBitmapTextureImageInternal(BitmapTextureImage& textureImage)
{
    SetTextureStateCache(textureImage, 0);

    GLuint textureHandle = static_cast<GLuint>(textureImage.GetVideoMemoryHandle());

    if (GetCurrentRenderStateCache() != 0) {
        GetCurrentRenderStateCache()->ResetActiveTextureHandles(textureHandle);
    }
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteTextures, (1, &textureHandle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteTextures failed.");

    Diagnostics::VideoMemoryStatistic::OnTextureImageUnloaded(textureImage);
    // Reset video memory handle to 0.
    textureImage.SetVideoMemoryHandle(0);
    return true;
}

bool RenderDevice::DestroyTexture(Handle handle)
{
    const GLuint textureHandle = static_cast<GLuint>(handle);

    if (GetCurrentRenderStateCache() != 0) {
        GetCurrentRenderStateCache()->ResetActiveTextureHandles(textureHandle);
    }
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteTextures, (1, &textureHandle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteTextures failed.");

    return true;
}

bool RenderDevice::UnloadCubeMapTextureImage(CubeMapTextureImage& textureImage)
{
    SetTextureStateCache(textureImage, 0);

    GLuint textureHandle = static_cast<GLuint>(textureImage.GetVideoMemoryHandle());

    if (GetCurrentRenderStateCache() != 0) {
        GetCurrentRenderStateCache()->ResetActiveTextureHandles(textureHandle);
    }
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteTextures, (1, &textureHandle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteTextures failed.");

    Diagnostics::VideoMemoryStatistic::OnTextureImageUnloaded(textureImage);
    // Reset video memory handle to 0.
    textureImage.SetVideoMemoryHandle(0);
    return true;
}

bool RenderDevice::ActivateTextureImage(const TextureImage& textureImage, UInt unit)
{
    GLOBAL_EXPERIMENT_COND_ACTIVATE_DEFAULT_TEXTURE_IMAGE;

    return UpdateTextureBinding(textureImage.GetVideoMemoryHandle(), unit, textureImage.GetTextureTargetType());
}

bool RenderDevice::DeactivateTexture(UInt unit, TextureImage::TextureTargetType target)
{
    return UpdateTextureBinding(0, unit, target);
}

bool RenderDevice::UpdateTextureBinding(Handle textureHandle, UInt unit, TextureImage::TextureTargetType target)
{
    if (GetCurrentRenderStateCache() != 0) {
        if (GetCurrentRenderStateCache()->activeTextureUnit != unit) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glActiveTexture, (GlTypeMapper::MapTextureUnit(unit)));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glActiveTexture failed.");

            GetCurrentRenderStateCache()->activeTextureUnit = unit;
        }

        if (GetCurrentRenderStateCache()->activeTextureHandle[unit][target] != textureHandle) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindTexture, (GlTypeMapper::MapTextureTargetType(target), static_cast<GLuint>(textureHandle)));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindTexture failed.");

            GetCurrentRenderStateCache()->activeTextureHandle[unit][target] = textureHandle;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glActiveTexture, (GlTypeMapper::MapTextureUnit(unit)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glActiveTexture failed.");

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindTexture, (GlTypeMapper::MapTextureTargetType(target), static_cast<GLuint>(textureHandle)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindTexture failed.");
    }

    return true;
}

bool RenderDevice::SetTextureSubimage(const BitmapTextureImage& textureImage, UInt level, Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data, UInt unit, UInt compressedSize)
{
    const Bitmap::SharedPointer& bitmap = textureImage.GetBitmap();
    if ((bitmap == 0) || (bitmap->IsCustomPixelFormat())) {
        FEATSTD_LOG_ERROR("Set texture subimage failed, bitmap not valid.");
        return false;
    }

    // Activate Texture to ensure correct use
    bool rc = ActivateTextureImage(textureImage, unit);
    if (!rc) {
        return rc;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glPixelStorei, (GL_UNPACK_ALIGNMENT, static_cast<GLint>(bitmap->GetPackAlignment())));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glPixelStorei failed.");

    if (!bitmap->IsCompressed()) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexSubImage2D, (GL_TEXTURE_2D, GLint(level), xOffset, yOffset, static_cast<Int>(width), static_cast<Int>(height),
            GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
            GlTypeMapper::MapBitmapPixelFormatToGLType(bitmap->GetPixelFormat()),
            data));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexSubImage2D failed.");
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompressedTexSubImage2D, (GL_TEXTURE_2D, GLint(level), xOffset, yOffset, static_cast<Int>(width), static_cast<Int>(height),
            GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
            compressedSize, data));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompressedTexSubImage2D failed.");
    }

    return true;
}

bool RenderDevice::SetTextureSubimage(const CubeMapTextureImage& textureImage, CubeMapTextureImage::TargetFace face, UInt level,
    Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data, UInt unit, UInt compressedSize)
{
    const Bitmap::SharedPointer& bitmap = textureImage.GetBitmapFace(face);
    if ((bitmap == 0) || (bitmap->IsCustomPixelFormat())) {
        FEATSTD_LOG_ERROR("Set texture subimage failed, bitmap not valid.");
        return false;
    }

    // Activate Texture to ensure correct use
    bool rc = ActivateTextureImage(textureImage, unit);
    if (!rc) {
        return rc;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glPixelStorei, (GL_UNPACK_ALIGNMENT, static_cast<GLint>(bitmap->GetPackAlignment())));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glPixelStorei failed.");

    if (!bitmap->IsCompressed()) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexSubImage2D, (GlTypeMapper::MapTextureCubeMapTargetFace(face), GLint(level), xOffset, yOffset, static_cast<Int>(width), static_cast<Int>(height),
            GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
            GlTypeMapper::MapBitmapPixelFormatToGLType(bitmap->GetPixelFormat()),
            data));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexSubImage2D failed.");
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompressedTexSubImage2D, (GlTypeMapper::MapTextureCubeMapTargetFace(face), GLint(level), xOffset, yOffset, static_cast<Int>(width), static_cast<Int>(height),
            GlTypeMapper::MapBitmapPixelFormatToGLFormat(bitmap->GetPixelFormat()),
            compressedSize, data));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompressedTexSubImage2D failed.");
    }

    return true;
}

bool RenderDevice::CopyFromFramebufferToTexture(const BitmapTextureImage& textureImage, Int32 xTex, Int32 yTex, Int32 xFramebuffer, Int32 yFramebuffer, Int32 width, Int32 height, UInt unit)
{
    // Activate Texture to ensure correct use
    bool rc = ActivateTextureImage(textureImage, unit);
    if (!rc) {
        return rc;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCopyTexSubImage2D, (GL_TEXTURE_2D, 0, xTex, yTex, xFramebuffer, yFramebuffer, width, height));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCopyTexture failed.");
    return true;
}

bool RenderDevice::CopyFromFramebufferToCubeMapTextureFace(const CubeMapTextureImage& textureImage, CubeMapTextureImage::TargetFace face, Int32 xTex, Int32 yTex,
    Int32 xFramebuffer, Int32 yFramebuffer, Int32 width, Int32 height, UInt unit)
{
    // Activate Texture to ensure correct use
    bool rc = ActivateTextureImage(textureImage, unit);
    if (!rc) {
        return rc;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCopyTexSubImage2D, (GlTypeMapper::MapTextureCubeMapTargetFace(face), 0, xTex, yTex, xFramebuffer, yFramebuffer, width, height));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCopyTexture failed.");
        return true;
}

bool RenderDevice::ReadPixels(Int32 x, Int32 y, Int32 width, Int32 height, Bitmap::PixelFormat pixelFormat, Bitmap::PackAlignment packAlignment, UInt8* dstData)
{
    if (pixelFormat >= Bitmap::PixelFormatCount) {
        FEATSTD_LOG_ERROR("Pixel format not valid.");
        return false;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glPixelStorei, (GL_PACK_ALIGNMENT, static_cast<GLint>(packAlignment)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glPixelStorei failed.");


    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glReadPixels, (x, y, width, height, GlTypeMapper::MapBitmapPixelFormatToGLFormat(pixelFormat),
        GlTypeMapper::MapBitmapPixelFormatToGLType(pixelFormat), dstData));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glReadPixels failed.");
    return true;
}

/**
 *  Added as a helper function to allow for deprecation.
 */
bool SetTextureWrapModeHelper(const Texture& texture, TextureStateCache* textureStateCache)
{
    TextureImage::TextureTargetType target = TextureImage::Texture2D;
    if (texture.GetTextureImage() != 0) {
        target = texture.GetTextureImage()->GetTextureTargetType();
    }

    bool cacheInitialized = true;
    if (textureStateCache != 0) {
        cacheInitialized = textureStateCache->isWrapModeInitialized;
    }

    if ((textureStateCache == 0) || (textureStateCache->wrapModeU != texture.GetWrapModeU()) || (!cacheInitialized)) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameteri, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_WRAP_S, static_cast<Int>(GlTypeMapper::MapTextureWrapMode(texture.GetWrapModeU()))));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexParameteri for U failed.");

        if (textureStateCache != 0) {
            textureStateCache->wrapModeU = texture.GetWrapModeU();
        }
    }

    if ((textureStateCache == 0) || (textureStateCache->wrapModeV != texture.GetWrapModeV()) || (!cacheInitialized)) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameteri, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_WRAP_T, static_cast<Int>(GlTypeMapper::MapTextureWrapMode(texture.GetWrapModeV()))));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexParameteri for V failed.");

        if (textureStateCache != 0) {
            textureStateCache->wrapModeV = texture.GetWrapModeV();
            textureStateCache->isWrapModeInitialized = true;
        }
    }

    return true;
}

bool RenderDevice::SetTextureWrapMode(const Texture& texture)
{
    return SetTextureWrapModeHelper(texture, GetTextureStateCache(texture.GetTextureImage()));
}

/**
 *  Added as a helper function to allow for deprecation.
 */
bool SetTextureFilterHelper(const Texture& texture, TextureStateCache* textureStateCache)
{
    if (texture.GetTextureImage() == 0) {
        FEATSTD_LOG_ERROR("Set texture filter failed, texture image invalid.");
        return false;
    }

    bool cacheInitialized = true;
    if (textureStateCache != 0) {
        cacheInitialized = textureStateCache->isTextureFilterInitialized;
    }

    TextureImage::TextureTargetType target = texture.GetTextureImage()->GetTextureTargetType();

    const Texture::MinMagFilter magFilter = texture.GetMagnificationFilter();
    const Texture::MinMagFilter minFilter = texture.GetMinificationFilter();
    const Texture::MipMapFilter mipMapFilter = texture.GetMipMapFilter();

    // Set Magnification filter
    if ((textureStateCache == 0) || (textureStateCache->magFilter != magFilter) || (!cacheInitialized)) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameteri, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_MAG_FILTER, static_cast<Int>(GlTypeMapper::MapTextureMinMagFilter(magFilter))));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexParameteri failed.");

        if (textureStateCache != 0) {
            textureStateCache->magFilter = magFilter;
        }
    }

    if ((textureStateCache == 0) || (textureStateCache->isMipMappingEnabled != texture.GetTextureImage()->IsMipMappingEnabled()) ||
        (textureStateCache->mipMapFilter != mipMapFilter) || (textureStateCache->minFilter != minFilter) || (!cacheInitialized)) {
        // Set Minification filter without MipMap filter
        if ((mipMapFilter == Texture::MipMapNone) || (!texture.GetTextureImage()->IsMipMappingEnabled())) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameteri, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_MIN_FILTER, static_cast<Int>(GlTypeMapper::MapTextureMinMagFilter(minFilter))));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexParameteri in case of MipMapNone failed.");
        }
        // Set Minification filter and Mipmap filter if MipMapping is enabled
        else {
            if ((minFilter == Texture::MinMagNearest) && (mipMapFilter == Texture::MipMapNearest)) {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameteri, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexParameteri call for Minification filter failed.");
            }
            else if ((minFilter == Texture::MinMagNearest) && (mipMapFilter == Texture::MipMapLinear)) {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameteri, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexParameteri call for Minification filter failed.");
            }
            else if ((minFilter == Texture::MinMagLinear) && (mipMapFilter == Texture::MipMapNearest)) {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameteri, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexParameteri call for Minification filter failed.");
            }
            else if ((minFilter == Texture::MinMagLinear) && (mipMapFilter == Texture::MipMapLinear)) {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameteri, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexParameteri call for Minification filter failed.");
            }
            else {
                // do nothing
            }
        }

        if (textureStateCache != 0) {
            textureStateCache->isMipMappingEnabled = texture.GetTextureImage()->IsMipMappingEnabled();
            textureStateCache->mipMapFilter = mipMapFilter;
            textureStateCache->minFilter = minFilter;
            textureStateCache->isTextureFilterInitialized = true;
        }
    }

#if GL_EXT_texture_filter_anisotropic

    cacheInitialized = true;
    if (textureStateCache != 0) {
        cacheInitialized = textureStateCache->isMaxAnisotropyInitialized;
    }

    // Set maximum degree of anisotropic filter.
    if ((textureStateCache == 0) || (!(Math::FloatAlmostEqual(textureStateCache->maxAnisotropy, texture.GetMaxAnisotropy()))) || (!cacheInitialized)) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glTexParameterf, (GlTypeMapper::MapTextureTargetType(target), GL_TEXTURE_MAX_ANISOTROPY_EXT, static_cast<GLfloat>(texture.GetMaxAnisotropy())));
        CANDERA_GL_CHECK_ERROR_NO_RETURN(
            "glTexParameterf call for anisotropic filter failed. GL error code:\n"
            "Check if extension GL_EXT_texture_filter_anisotropic is supported.\n");

        if (textureStateCache != 0) {
            textureStateCache->maxAnisotropy = texture.GetMaxAnisotropy();
            textureStateCache->isMaxAnisotropyInitialized = true;
        }
    }
#endif

    return true;
}

bool RenderDevice::SetTextureFilter(const Texture& texture) {
    return SetTextureFilterHelper(texture, GetTextureStateCache(texture.GetTextureImage()));
}

bool RenderDevice::SetTextureLodMode(const Texture& /*texture*/)
{
    //This feature is not available in OpenGL ES 2.0.
    return true;
}

bool RenderDevice::ActivateTexture(const Texture& texture, UInt /*unit*/)
{
    TextureStateCache* textureStateCache = GetTextureStateCache(texture.GetTextureImage());
    bool result = SetTextureWrapModeHelper(texture, textureStateCache);
    result = result && SetTextureFilterHelper(texture, textureStateCache);

    return result;
}

bool RenderDevice::UploadVertexBuffer(VertexBuffer& vertexBuffer)
{
    const VertexGeometry* vertexGeometry = vertexBuffer.GetVertexGeometry();
    if (vertexGeometry == 0) {
        FEATSTD_LOG_ERROR("Upload vertex buffer failed, vertex geometry invalid.");
        return false; // Geometry is not set in vertex buffer.
    }
    Handle vHandle;
    Handle iHandle;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenBuffers, (1, FeatStd::Internal::PointerToPointer<GLuint*>(&vHandle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGenBuffers failed.");

    VertexGeometry::VertexArrayResource vertexArrayResource(vertexGeometry->GetVertexArrayResourceHandle());
    const ResourceDataHandle& indexArrayResourceHandle = vertexGeometry->GetIndexArrayResourceHandle();
    Internal::ResourceData indexArrayResource(indexArrayResourceHandle);
    const void* indexArray = indexArrayResource.GetData();
    if (indexArray != 0) {
        if (vertexGeometry->GetBufferType() == VertexGeometry::ArrayBuffer) {
            FEATSTD_LOG_ERROR("Upload vertex buffer failed, unexpected vertex format.");
            return false;
        }

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ARRAY_BUFFER, static_cast<GLuint>(vHandle)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer failed.");

        // This is where the real upload happens.
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBufferData, (GL_ARRAY_BUFFER,
                         static_cast<GLsizeiptr>(vertexGeometry->GetVertexCount() * vertexGeometry->GetVertexStride()),
                         vertexArrayResource.GetData(), GlTypeMapper::MapVertexGeometryUsage(vertexGeometry->GetBufferUsage())));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBufferData failed.");
        vertexBuffer.SetVertexArrayMemoryHandle(vHandle);

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenBuffers, (1, FeatStd::Internal::PointerToPointer<GLuint*>(&iHandle)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGenBuffers 2nd call failed.");

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ELEMENT_ARRAY_BUFFER, static_cast<GLuint>(iHandle)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer failed.");

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBufferData, (GL_ELEMENT_ARRAY_BUFFER,
                         static_cast<GLsizeiptr>(indexArrayResourceHandle.m_size), indexArray,
                         GlTypeMapper::MapVertexGeometryUsage(vertexGeometry->GetBufferUsage())));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBufferData failed.");

        vertexBuffer.SetIndexArrayMemoryHandle(iHandle);
    }
    else {
        if (vertexGeometry->GetBufferType() != VertexGeometry::ArrayBuffer) {
            FEATSTD_LOG_ERROR("Upload vertex buffer failed, unexpected vertex format. Expected array buffer.");
            return false;
        }

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ARRAY_BUFFER, static_cast<GLuint>(vHandle)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer failed.");

        // This is where the real upload happens
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBufferData, (GL_ARRAY_BUFFER,
                         static_cast<GLsizeiptr>(vertexGeometry->GetVertexCount() * vertexGeometry->GetVertexStride()),
                         vertexArrayResource.GetData(), GlTypeMapper::MapVertexGeometryUsage(vertexGeometry->GetBufferUsage())));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBufferData 2nd call failed.");

        vertexBuffer.SetVertexArrayMemoryHandle(vHandle);
    }

    if (GetCurrentRenderStateCache() != 0) {
        GetCurrentRenderStateCache()->activeVertexBufferHandle = vertexBuffer.GetVertexArrayMemoryHandle();
        GetCurrentRenderStateCache()->activeIndexBufferHandle = vertexBuffer.GetIndexBufferMemoryHandle();
    }

    Diagnostics::VideoMemoryStatistic::OnVertexBufferUploaded(vertexBuffer);
    return true;
}

namespace {

class DeactivateAttributesFromContextResourcePool : public ContextResourcePoolVisitor
{
public:
    DeactivateAttributesFromContextResourcePool(UInt vboHandle) : m_vboHandle(vboHandle) {}

protected:
    virtual void VisitContext() override
    {
        GLint maxAttribCount = 0;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_VERTEX_ATTRIBS, &maxAttribCount));
        CANDERA_EGL_CHECK_ERROR_NO_RETURN("glGetIntegerv of GL_MAX_VERTEX_ATTRIBS failed.");
        for (Int i = 0; i < maxAttribCount; i++) {
            GLint linkedVbo = 0;
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetVertexAttribiv, (static_cast<GLuint>(i), GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING, &linkedVbo));
            CANDERA_EGL_CHECK_ERROR_NO_RETURN("glGetVertexAttribiv of GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING failed.");
            if (static_cast<UInt>(linkedVbo) == m_vboHandle) {
                static_cast<void>(RenderDevice::DeactivateAttribute(static_cast<GLuint>(i)));
            }
        }
    }
private:
    UInt m_vboHandle;
};

}

bool RenderDevice::UnloadVertexBuffer(VertexBuffer& vertexBuffer)
{
    // First deactivate all attributes that are linked to vertex buffer given. This avoids crashes caused by dangling attributes.
    UInt vbo_handle = static_cast<UInt>(vertexBuffer.GetVertexArrayMemoryHandle());

    DeactivateAttributesFromContextResourcePool(vbo_handle).Visit();

    // Delete vertex array.
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteBuffers, (1, &vbo_handle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteBuffers with vertex array failed.");

    // Delete index array.
    vbo_handle = static_cast<GLuint>(vertexBuffer.GetIndexBufferMemoryHandle());
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteBuffers, (1, &vbo_handle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteBuffers with index buffer failed.");

    Diagnostics::VideoMemoryStatistic::OnVertexBufferUnloaded(vertexBuffer);

    vertexBuffer.SetVertexArrayMemoryHandle(0);
    vertexBuffer.SetIndexArrayMemoryHandle(0);

    if (GetCurrentRenderStateCache() != 0) {
        GetCurrentRenderStateCache()->activeVertexBufferHandle = 0;
        GetCurrentRenderStateCache()->activeIndexBufferHandle = 0;
    }

    return true;
}

bool RenderDevice::ActivateVertexBuffer(VertexBuffer& vertexBuffer)
{
    if (GetCurrentRenderStateCache() != 0) {
        if (GetCurrentRenderStateCache()->activeVertexBufferHandle != vertexBuffer.GetVertexArrayMemoryHandle()) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ARRAY_BUFFER, static_cast<GLuint>(vertexBuffer.GetVertexArrayMemoryHandle())));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer for GL_ARRAY_BUFFER failed.");

            GetCurrentRenderStateCache()->activeVertexBufferHandle = vertexBuffer.GetVertexArrayMemoryHandle();
        }

        if (GetCurrentRenderStateCache()->activeIndexBufferHandle != vertexBuffer.GetIndexBufferMemoryHandle()) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ELEMENT_ARRAY_BUFFER, static_cast<GLuint>(vertexBuffer.GetIndexBufferMemoryHandle())));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer for GL_ELEMENT_ARRAY_BUFFER failed.");

            GetCurrentRenderStateCache()->activeIndexBufferHandle = vertexBuffer.GetIndexBufferMemoryHandle();
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ARRAY_BUFFER, static_cast<GLuint>(vertexBuffer.GetVertexArrayMemoryHandle())));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer for GL_ARRAY_BUFFER failed.");

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ELEMENT_ARRAY_BUFFER, static_cast<GLuint>(vertexBuffer.GetIndexBufferMemoryHandle())));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer for GL_ELEMENT_ARRAY_BUFFER failed.");
    }

    return true;
}

bool RenderDevice::SetVertexBufferSubData(VertexBuffer& vertexBuffer, Int offset, Int size, const UInt8* data)
{
    //only allow update of array buffer so far
    if (GetCurrentRenderStateCache() != 0) {
        if (GetCurrentRenderStateCache()->activeVertexBufferHandle != vertexBuffer.GetVertexArrayMemoryHandle()) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ARRAY_BUFFER, static_cast<GLuint>(vertexBuffer.GetVertexArrayMemoryHandle())));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer for GL_ARRAY_BUFFER failed.");

            GetCurrentRenderStateCache()->activeVertexBufferHandle = vertexBuffer.GetVertexArrayMemoryHandle();
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ARRAY_BUFFER, static_cast<GLuint>(vertexBuffer.GetVertexArrayMemoryHandle())));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer for GL_ARRAY_BUFFER failed.");
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBufferSubData, (GL_ARRAY_BUFFER, offset, size, data));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBufferSubData failed.");
    return true;
}

bool RenderDevice::ActivateShader(Handle programHandle)
{
    GLOBAL_EXPERIMENT_COND_ACTIVATE_DEFAULT_SHADER;

    if (GetCurrentRenderStateCache() != 0) {
        if (GetCurrentRenderStateCache()->activeShaderHandle != programHandle) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUseProgram, (static_cast<GLuint>(programHandle)));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUseProgram failed.");

            GetCurrentRenderStateCache()->activeShaderHandle = programHandle;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUseProgram, (static_cast<GLuint>(programHandle)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUseProgram failed.");
    }

    return true;
}

bool RenderDevice::UploadShaderSource(Shader::ObjectType objectType, const void* shader, Handle& handle)
{

    GLint compiled = 0;

    //Create shader handle.
    GLint shaderHandle = 0;
    {
        FEATSTD_LINT_NEXT_EXPRESSION(838, "Violates MISRA C++ 2008 Required Rule 0-1-9: previously assigned value")
        CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(shaderHandle, glCreateShader, (GlTypeMapper::MapShaderObjectType(objectType)));
    }
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCreateShader failed.");

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glShaderSource, (shaderHandle, 1, FeatStd::Internal::PointerToPointer<const Char**>(&shader), 0));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glShaderSource failed.");
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCompileShader, (shaderHandle))
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCompileShader failed.");

    //Check shader compile state.
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetShaderiv, (shaderHandle, GL_COMPILE_STATUS, &compiled));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetShaderiv failed.");

    if (compiled == 0) {
        GLint infoLen = 0;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetShaderiv, (shaderHandle, GL_INFO_LOG_LENGTH, &infoLen));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetShaderiv failed.");

        if (infoLen > 1) {
            Char infoLog[1024];
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetShaderInfoLog, (shaderHandle, 1024, NULL, infoLog));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetShaderInfoLog failed.");
            CANDERA_DEVICE_LOG_ERROR("compile and load vertex shader infolog: %s", infoLog);
        }
        else {
            CANDERA_DEVICE_LOG_ERROR("compile and load vertex shader - no infolog");
        }

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteShader, (shaderHandle));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteShader failed.");
        return false;
    }

    handle = shaderHandle;

    return true;
}


bool RenderDevice::UploadShader(Shader& shader)
{
    Shader::ShaderResource vertexShaderResource(shader.GetVertexShaderResourceHandle());
    const void* vertexShader = vertexShaderResource.GetData();

    Shader::BuildType vertexShaderBuildType = Shader::ShaderSource;
    if ((vertexShader != 0) && (!GetShaderBuildType(vertexShader, vertexShaderBuildType))) {
        return false;
    }

    bool result = true;

    Handle programHandle = 0;

    unsigned int start = Candera::PerfCounterPlatform::Now();
    if (vertexShaderBuildType == Shader::ShaderProgramBinary) {
        result = UploadShaderBinaryProgram(vertexShader, programHandle);
        shader.SetShaderProgramMemoryHandle(programHandle);
    } else {
        Shader::ShaderResource fragmentShaderResource(shader.GetFragmentShaderResourceHandle());
        const void* fragmentShader = fragmentShaderResource.GetData();

        Shader::BuildType fragmentShaderBuildType = Shader::ShaderSource;
        if ((fragmentShader != 0) && (!GetShaderBuildType(fragmentShader, fragmentShaderBuildType))) {
            return false;
        }

        if (fragmentShaderBuildType == Shader::ShaderProgramBinary) {
            result = UploadShaderBinaryProgram(fragmentShader, programHandle);
            shader.SetShaderProgramMemoryHandle(programHandle);
        } else if ((vertexShader != 0) && (fragmentShader != 0)) {
            Handle vertexShaderHandle = 0;
            Handle fragmentShaderHandle = 0;

            if (vertexShaderBuildType == Shader::ShaderSource) {
                result = UploadShaderSource(Shader::VertexShader, vertexShader, vertexShaderHandle);
            }
            else if (vertexShaderBuildType == Shader::ShaderBinary) {
                result = UploadShaderBinary(Shader::VertexShader, vertexShader, vertexShaderHandle);
            }
            else {
                result = false;
            }
            shader.SetShaderObjectMemoryHandle(Shader::VertexShader, vertexShaderHandle);

            if (fragmentShaderBuildType == Shader::ShaderSource) {
                result = result && UploadShaderSource(Shader::FragmentShader, fragmentShader, fragmentShaderHandle);
            }
            else if (fragmentShaderBuildType == Shader::ShaderBinary) {
                result = result && UploadShaderBinary(Shader::FragmentShader, fragmentShader, fragmentShaderHandle);
            }
            else {
                result = false;
            }
            shader.SetShaderObjectMemoryHandle(Shader::FragmentShader, fragmentShaderHandle);


            result = result && LinkShader(vertexShaderHandle, fragmentShaderHandle, programHandle);
            shader.SetShaderProgramMemoryHandle(programHandle);
        } else {
            result = false;
        }
    }

    unsigned int stop = Candera::PerfCounterPlatform::Now();
    bool success = (vertexShaderBuildType == Shader::ShaderBinary) ? true : false;

#ifdef WIN32
    unsigned int duration = Candera::PerfCounterPlatform::ToMicroSec(Candera::PerfCounterPlatform::Duration(start, stop));
    CANDERA_DEVICE_LOG_FATAL("UploadShader %s took %d us", success ? "binary" : "source", duration);
#else
    unsigned int resolution = 1000 / CANDERA_PERFCOUNTER_RESOLUTION; // ms
    unsigned int duration = Candera::PerfCounterPlatform::Duration(start / resolution, stop / resolution);
    CANDERA_DEVICE_LOG_FATAL("[%d] UploadShader %s took %d ms", getpid(), success ? "binary" : "source", duration);
#endif

    return result;
}


#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED

//helper functions for dynamic acquisition of gl procs
static PFNGLGETPROGRAMBINARYOESPROC RetrieveGetProgramBinaryProc(){
#ifdef WIN32  //abstracted by OpenGLAdapter over GLEW, eglGetProcAddress not avaliable
    return glGetProgramBinaryOES;
#else
    static PFNGLGETPROGRAMBINARYOESPROC tmp = (PFNGLGETPROGRAMBINARYOESPROC)eglGetProcAddress("glGetProgramBinaryOES");
    // Check if OpenGLES2 Extension is avaliable
    if (0 == tmp){ // NO GLES2 Extension, check if OpenGLES3
        tmp = (PFNGLGETPROGRAMBINARYOESPROC)eglGetProcAddress("glGetProgramBinary");
        if (0==tmp){
            FEATSTD_LOG_WARN("Neither glGetProgramBinary nor glGetProgramBinaryOES avaliable.");
        }
    }
    return tmp;
#endif
}

static PFNGLPROGRAMBINARYOESPROC RetrieveProgramBinaryProc(){
#ifdef WIN32 //abstracted by OpenGLAdapter over GLEW, eglGetProcAddress not avaliable
    return glProgramBinaryOES;
#else
    static PFNGLPROGRAMBINARYOESPROC tmp = (PFNGLPROGRAMBINARYOESPROC)eglGetProcAddress("glProgramBinaryOES");
    // Check if OpenGLES2 Extension is avaliable
    if (0 == tmp){ // NO GLES2 Extension, check if OpenGLES3
        tmp = (PFNGLPROGRAMBINARYOESPROC)eglGetProcAddress("glProgramBinary");
        if (0==tmp){
            FEATSTD_LOG_WARN("Neither glProgramBinary nor glProgramBinaryOES avaliable.");
        }
    }
    return tmp;
#endif
}


bool RenderDevice::UploadStoredShaderBinaryProgram(Candera::Shader& shader, Candera::GlShaderStorageInterface::StorageObject & storageObject)
{
    static PFNGLPROGRAMBINARYOESPROC ProgramBinary = RetrieveProgramBinaryProc();
    EGLint success = EGL_FALSE;

    if (0 != ProgramBinary){
        if (0 != GlShaderStorageProvider::GetShaderCache()){
                Handle programHandle = glCreateProgram();
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(ProgramBinary, (programHandle, storageObject.m_binaryFormat, storageObject.m_data, storageObject.m_dataSize));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glProgramBinary(OES) failed.");
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetProgramiv, (programHandle, GL_LINK_STATUS, &success));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetProgramiv failed.");
                if (EGL_TRUE == success){
                    shader.SetShaderProgramMemoryHandle(programHandle);
                }
                else {
                    FEATSTD_LOG_WARN("Shader retrieved from storage failed to upload. Compiling again from text.");
                }
        }
        else {
            FEATSTD_LOG_WARN("GlShaderStorageProvider has no storge implementation set.");
        }
    }
    else {
        FEATSTD_LOG_WARN("Precompiled shader programs are not supported on this platform.");
    }

    return (success == EGL_TRUE);
}

bool RenderDevice::RetrieveShaderBinaryProgram(Candera::Shader& shader, GlShaderStorageInterface::StorageObject & storageObject)
{
    bool success = false;
    if (0 != GlShaderStorageProvider::GetShaderCache()){
        GLsizei   binaryLength = 0;
        GLenum binaryFormat = 0xF0F0F0F0; //set to make debugging easier. for all tested platforms it should be 1 after glGetProgramBinary.
        static PFNGLGETPROGRAMBINARYOESPROC GetProgramBinary = RetrieveGetProgramBinaryProc();
        if (0 != GetProgramBinary){
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetProgramiv, (shader.GetProgramHandle(), GL_PROGRAM_BINARY_LENGTH_OES, &binaryLength));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetProgramiv failed.");
            if (binaryLength > 0){
                storageObject.m_data = FEATSTD_NEW_ARRAY(Char, binaryLength);
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(GetProgramBinary, (shader.GetProgramHandle(), binaryLength, NULL, &binaryFormat, storageObject.m_data));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetProgramBinary(OES) failed.");
                storageObject.m_binaryFormat = binaryFormat;
                storageObject.m_dataSize = binaryLength;
                success = binaryFormat != 0xF0F0F0F0;  //binary format needs to change if call was successful.
            }
        }
    }
    return success;
}
#endif

bool RenderDevice::LinkShader(Handle& vertexShaderHandle, Handle& fragmentShaderHandle, Handle& programHandle,
    const Char* const* transformVaryings, SizeType transformVaryingsCount)
{
    FEATSTD_UNUSED2(transformVaryings, transformVaryingsCount);
    GLint linked = 0;
    // Create the program object
    {
        CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(programHandle, glCreateProgram, ());
    }
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCreateProgram failed.");

    if (programHandle == 0) {
        FEATSTD_LOG_ERROR("LinkShader failed, program handle invalid.");
        return false;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glAttachShader, (static_cast<GLuint>(programHandle), static_cast<GLuint>(vertexShaderHandle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glAttachShader for vertex shader failed.");

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glAttachShader, (static_cast<GLuint>(programHandle), static_cast<GLuint>(fragmentShaderHandle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glAttachShader for fragment shader failed.");

    // Link the program
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glLinkProgram, (static_cast<GLuint>(programHandle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glLinkProgram failed.");

    // Check the link status
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetProgramiv, (static_cast<GLuint>(programHandle), GL_LINK_STATUS, &linked));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetProgramiv failed.");

    if (linked == 0) {
        GLint infoLen = 0;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetShaderiv, (static_cast<GLuint>(programHandle), GL_INFO_LOG_LENGTH, &infoLen));

        if (infoLen > 1) {
            Char infoLog[256];
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetShaderInfoLog, (static_cast<GLuint>(programHandle), Math::Minimum(infoLen, 256), NULL, infoLog));
            FEATSTD_LOG_ERROR("Link shader failed:\n %s", infoLog);
        }
        else {
            FEATSTD_LOG_ERROR("LinkShader failed, (info log empty).");
        }

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteProgram, (static_cast<GLuint>(programHandle)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteProgram failed.");
        return false;
    }
    return true;
}

Int RenderDevice::GetAttributeLocation(Handle programHandle, ShaderParamNames::AttributeSemantic attributeSemantic)
{
    GLOBAL_EXPERIMENT_COND_USE_DEFAULT_SHADER_INSTEAD_OF(programHandle);

    const Char* attributeName = ShaderParamNames::GetAttributeName(attributeSemantic);
    return RenderDevice::GetAttributeLocation(programHandle, attributeName);
}

Int RenderDevice::GetAttributeLocation(Handle programHandle, const Char* attributeName)
{
    GLOBAL_EXPERIMENT_COND_USE_DEFAULT_SHADER_INSTEAD_OF(programHandle);

    Int attributeLocation = -1;

    if (attributeName != 0) {
        CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(attributeLocation, glGetAttribLocation, (static_cast<GLuint>(programHandle), attributeName));
    }

    return attributeLocation;
}

static bool GetProgramParameter(Handle programHandle, GLenum parameterName, Int* parameter)
{
    GLOBAL_EXPERIMENT_COND_USE_DEFAULT_SHADER_INSTEAD_OF(programHandle);

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetProgramiv, (static_cast<GLuint>(programHandle), parameterName, parameter));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetProgramiv failed.");
    return true;
}

bool RenderDevice::GetActiveAttributeCount(Handle programHandle, Int& count)
{
    return GetProgramParameter(programHandle, GL_ACTIVE_ATTRIBUTES, &count);
}

bool RenderDevice::GetActiveAttributeMaxLength(Handle programHandle, Int& maxLength)
{
    return GetProgramParameter(programHandle, GL_ACTIVE_ATTRIBUTE_MAX_LENGTH, &maxLength);
}

bool RenderDevice::GetActiveAttribute(Handle programHandle, Int index, Int nameBufferSize, Char* name, Int& nameLength, Int& size, Shader::UniformType& type)
{
    GLOBAL_EXPERIMENT_COND_USE_DEFAULT_SHADER_INSTEAD_OF(programHandle);

    GLenum glType = GL_FLOAT;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetActiveAttrib, (static_cast<GLuint>(programHandle),
        static_cast<UInt32>(index),
        static_cast<GLsizei>(nameBufferSize),
        &nameLength,
        &size,
        &glType,
        name));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetActiveAttrib failed.");
    type = GlTypeMapper::MapGlUniformType(glType);
    return true;
}

namespace {

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
class DeactivateShaderFromContextResourcePool : public ContextResourcePoolVisitor
{
public:
    DeactivateShaderFromContextResourcePool(Handle programHandle) : m_programHandle(programHandle) {}
protected :
    virtual void VisitContext() override
    {
        GLint oldProgramHandle = 0;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_CURRENT_PROGRAM, &oldProgramHandle));
        CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv of GL_CURRENT_PROGRAM failed.");
        if (static_cast<Handle>(oldProgramHandle) == m_programHandle) {
            FEATSTD_UNUSED(RenderDevice::ActivateShader(0));
        }
    }

private:
    Handle m_programHandle;
};
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

}

bool RenderDevice::DeleteShader(Handle vertexShaderHandle, Handle fragmentShaderHandle, Handle programHandle)
{
    bool result = true;

    DeactivateShaderFromContextResourcePool(programHandle).Visit();

    if (GetCurrentRenderStateCache() != 0) {
        RenderStateCache* cache = GetCurrentRenderStateCache();
        if (cache->activeShaderHandle == programHandle) {
            cache->activeShaderHandle = 0;
        }
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteShader, (static_cast<GLuint>(fragmentShaderHandle)));
    CANDERA_GL_CHECK_ERROR_LOG_INFO("Fragment Shader deletion failed. Missing Unload() calls before destruction of GL context?")

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteShader, (static_cast<GLuint>(vertexShaderHandle)));
    CANDERA_GL_CHECK_ERROR_LOG_INFO("Vertex Shader deletion failed. Missing Unload() calls before destruction of GL context?")

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteProgram, (static_cast<GLuint>(programHandle)));
    CANDERA_GL_CHECK_ERROR_LOG_INFO("Shader Program deletion failed. Missing Unload() calls before destruction of GL context?")
    return result;
}

bool RenderDevice::SetUniform(Handle programHandle, const Char* name, const void* data, Shader::UniformType type, UInt count)
{
    GLOBAL_EXPERIMENT_COND_USE_DEFAULT_SHADER_INSTEAD_OF(programHandle);

    Int location = -1;
    if (!GetUniformLocation(programHandle, name, location)) {
        FEATSTD_LOG_WARN("Uniform \"%s\" does not exist for current shader.", (name == 0)? "": name);
        return false;
    }
    return SetUniform(location, data, type, count);
}

template <>
bool RenderDevice::SetUniform<Shader::Float>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform1fv, (location, static_cast<Int>(count), PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform1fv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::FloatVec2>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform2fv, (location, static_cast<Int>(count), PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform2fv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::FloatVec3>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform3fv, (location, static_cast<Int>(count), PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform3fv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::FloatVec4>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform4fv, (location, static_cast<Int>(count), PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform4fv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::Integer>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform1iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform1iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::IntegerVec2>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform2iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform2iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::IntegerVec3>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform3iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform3iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::IntegerVec4>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform4iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform4iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::Bool>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform1iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform1iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::BoolVec2>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform2iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform2iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::BoolVec3>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform3iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform3iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::BoolVec4>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform4iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform4iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::FloatMat2>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniformMatrix2fv, (location, static_cast<Int>(count), GL_FALSE, PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniformMatrix2fv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::FloatMat3>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniformMatrix3fv, (location, static_cast<Int>(count), GL_FALSE, PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniformMatrix3fv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::FloatMat4>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniformMatrix4fv, (location, static_cast<Int>(count), GL_FALSE, PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniformMatrix4fv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::Sampler2D>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform1iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform1iv failed.");
    return true;
}

template <>
bool RenderDevice::SetUniform<Shader::SamplerCube>(Int location, const void* data, UInt count)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glUniform1iv, (location, static_cast<Int>(count), PointerToPointer<const GLint*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glUniform1iv failed.");
    return true;
}

bool RenderDevice::SetUniform(Int location, const void* data, Shader::UniformType type, UInt count)
{
    bool ret = false;
    switch(type) {
        case Shader::Float: ret = SetUniform<Shader::Float>(location, data, count); break;
        case Shader::FloatVec2: ret = SetUniform<Shader::FloatVec2>(location, data, count); break;
        case Shader::FloatVec3: ret = SetUniform<Shader::FloatVec3>(location, data, count); break;
        case Shader::FloatVec4: ret = SetUniform<Shader::FloatVec4>(location, data, count); break;

        case Shader::Integer: ret = SetUniform<Shader::Integer>(location, data, count); break;
        case Shader::IntegerVec2: ret = SetUniform<Shader::IntegerVec2>(location, data, count); break;
        case Shader::IntegerVec3: ret = SetUniform<Shader::IntegerVec3>(location, data, count); break;
        case Shader::IntegerVec4: ret = SetUniform<Shader::IntegerVec4>(location, data, count); break;

        case Shader::Bool: ret = SetUniform<Shader::Bool>(location, data, count); break;
        case Shader::BoolVec2: ret = SetUniform<Shader::BoolVec2>(location, data, count); break;
        case Shader::BoolVec3: ret = SetUniform<Shader::BoolVec3>(location, data, count); break;
        case Shader::BoolVec4: ret = SetUniform<Shader::BoolVec4>(location, data, count); break;

        case Shader::FloatMat2: ret = SetUniform<Shader::FloatMat2>(location, data, count); break;
        case Shader::FloatMat3: ret = SetUniform<Shader::FloatMat3>(location, data, count); break;
        case Shader::FloatMat4: ret = SetUniform<Shader::FloatMat4>(location, data, count); break;

        case Shader::Sampler2D: ret = SetUniform<Shader::Sampler2D>(location, data, count); break;
        case Shader::SamplerCube: ret = SetUniform<Shader::SamplerCube>(location, data, count); break;

        default: break;
    }

    return ret;
}

bool RenderDevice::GetUniformLocation(Handle programHandle, const Char* name, Int& location)
{
    GLOBAL_EXPERIMENT_COND_USE_DEFAULT_SHADER_INSTEAD_OF(programHandle);

    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(location, glGetUniformLocation, (static_cast<GLuint>(programHandle), name));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetUniformLocation failed.");
    return (location != -1);
}

bool RenderDevice::GetUniformLocation(Handle programHandle, ShaderParamNames::UniformSemantic name, Int& location)
{
    GLOBAL_EXPERIMENT_COND_USE_DEFAULT_SHADER_INSTEAD_OF(programHandle);

    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(location, glGetUniformLocation, (static_cast<GLuint>(programHandle), ShaderParamNames::GetUniformName(name)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetUniformLocation failed.");
    return (location != -1);
}

bool RenderDevice::BindAttribute(Int attribute,
                                 const VertexGeometry::VertexElementFormat& vertexElement,
                                 UInt stride)
{

#if (GL_OES_vertex_half_float == 0)
    if((vertexElement.Type >= VertexGeometry::Float16_1) && (vertexElement.Type <= VertexGeometry::Float16_4)) {
        FEATSTD_LOG_ERROR("Bind attribute failed, Half Float not supported.");
        return false;
    }
#endif

    if ((attribute == -1) || (!ActivateAttribute(static_cast<UInt>(attribute)))) {
        FEATSTD_LOG_ERROR("Bind attribute failed, ID not found.");
        return false;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glVertexAttribPointer, (static_cast<UInt>(attribute), GlTypeMapper::MapVertexDataTypeSize(vertexElement.Type),
                          GlTypeMapper::MapVertexDataType(vertexElement.Type),
                          static_cast<GLboolean>(VertexGeometry::IsVertexElementNormalized(vertexElement.Type)),
                          static_cast<Int>(stride), ScalarToPointer<const void*, SizeType>(static_cast<SizeType>(vertexElement.Offset))));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glVertexAttribPointer failed.");
    return true;
}

bool RenderDevice::UnbindArrayBuffers()
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ARRAY_BUFFER, 0));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer in case of GL_ARRAY_BUFFER failed.");

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBindBuffer, (GL_ELEMENT_ARRAY_BUFFER, 0));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBindBuffer in case of GL_ELEMENT_ARRAY_BUFFER failed.");
    return true;
}

bool RenderDevice::BindClientAttribute(Int attribute,
                                       const VertexGeometry::VertexElementFormat& vertexElement,
                                       UInt stride,
                                       const void* base)
{
#if (GL_OES_vertex_half_float == 0)
    if((vertexElement.Type >= VertexGeometry::Float16_1) && (vertexElement.Type <= VertexGeometry::Float16_4)) {
        FEATSTD_LOG_ERROR("Bind attribute failed, Half Float not supported.");
        return false;
    }
#endif

    if ((attribute == -1) || (!ActivateAttribute(static_cast<UInt>(attribute)))) {
        FEATSTD_LOG_ERROR("Bind client attribute failed, ID not found.");
        return false;
    }

    const Char* ptr = PointerToPointer<const Char*>(base) +vertexElement.Offset;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glVertexAttribPointer, (static_cast<UInt>(attribute), GlTypeMapper::MapVertexDataTypeSize(vertexElement.Type),
    GlTypeMapper::MapVertexDataType(vertexElement.Type), GL_FALSE,
    static_cast<Int>(stride), ptr));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glVertexAttribPointer failed.");
    return true;
}

bool RenderDevice::BindInstanceAttribute(Int attribute,
                                         const VertexGeometry::VertexElementFormat& vertexElement,
                                         UInt stride,
                                         UInt divisor)
{
    FEATSTD_UNUSED(attribute);
    FEATSTD_UNUSED(vertexElement);
    FEATSTD_UNUSED(stride);
    FEATSTD_UNUSED(divisor);
    return false;
}

bool RenderDevice::BindInstanceClientAttribute(Int attribute,
                                               const VertexGeometry::VertexElementFormat& vertexElement,
                                               UInt stride,
                                               const void* base,
                                               UInt divisor)
{
    FEATSTD_UNUSED(attribute);
    FEATSTD_UNUSED(vertexElement);
    FEATSTD_UNUSED(stride);
    FEATSTD_UNUSED(base);
    FEATSTD_UNUSED(divisor);
    return false;
}

bool RenderDevice::SetConstantVertexAttribute(Handle programHandle, const Char* name, const void* data, Shader::ConstantAttributeType type)
{
    GLint index;
    {
        CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(index, glGetAttribLocation, (static_cast<GLuint>(programHandle), name));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetAttribLocation failed.");
    }

    if (index == -1) {
        FEATSTD_LOG_ERROR("Set constant vertex attribute failed, ID not found.");
        return false;
    }

    return SetConstantVertexAttribute(index, data, type);
}

bool RenderDevice::SetConstantVertexAttribute(Int index, const void* data, Shader::ConstantAttributeType type)
{
    static_cast<void>(DeactivateAttribute(index));

    bool ret = false;
    switch (type) {
        case Shader::FloatAttribute: ret = SetConstantVertexAttribute<Shader::FloatAttribute>(index, data); break;
        case Shader::FloatVec2Attribute: ret = SetConstantVertexAttribute<Shader::FloatVec2Attribute>(index, data); break;
        case Shader::FloatVec3Attribute: ret = SetConstantVertexAttribute<Shader::FloatVec3Attribute>(index, data); break;
        case Shader::FloatVec4Attribute: ret = SetConstantVertexAttribute<Shader::FloatVec4Attribute>(index, data); break;

        default: break;
    }

    return ret;
}

template<>
bool RenderDevice::SetConstantVertexAttribute<Shader::FloatAttribute>(Int index, const void* data)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glVertexAttrib1fv, (static_cast<UInt>(index), PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glVertexAttrib1fv failed.");
    return true;
}

template<>
bool RenderDevice::SetConstantVertexAttribute<Shader::FloatVec2Attribute>(Int index, const void* data)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glVertexAttrib2fv, (static_cast<UInt>(index), PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glVertexAttrib2fv failed.");
    return true;
}

template<>
bool RenderDevice::SetConstantVertexAttribute<Shader::FloatVec3Attribute>(Int index, const void* data)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glVertexAttrib3fv, (static_cast<UInt>(index), PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glVertexAttrib3fv failed.");
    return true;
}

template<>
bool RenderDevice::SetConstantVertexAttribute<Shader::FloatVec4Attribute>(Int index, const void* data)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glVertexAttrib4fv, (static_cast<UInt>(index), PointerToPointer<const GLfloat*>(data)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glVertexAttrib4fv failed.");
    return true;
}

static bool EnableVertexAttribArray(GLuint index)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnableVertexAttribArray, (index));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnableVertexAttribArray failed.");
    return true;
}

static bool DisableVertexAttribArray(GLuint index)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisableVertexAttribArray, (index));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisableVertexAttribArray failed.");
    return true;
}

static bool SetVertexAttribArray(RenderStateCache* cache, UInt index, bool enable)
{
    if (0 != cache) {
        const UInt maxCachedVertexAttribsCount = sizeof(cache->enabledVertexAttribs) * 8;
        if (maxCachedVertexAttribsCount > index) {
            UInt32 shiftedIndexBit = static_cast<UInt32>(1U) << index;
            if (enable) {
                if ((cache->enabledVertexAttribs & shiftedIndexBit) == shiftedIndexBit) {
                    return true;
                }

                if (EnableVertexAttribArray(static_cast<GLuint>(index))) {
                    cache->enabledVertexAttribs |= shiftedIndexBit;
                    return true;
                }
            }
            else {
                if ((cache->enabledVertexAttribs | shiftedIndexBit) == cache->enabledVertexAttribs) {
                    return true;
                }

                if (DisableVertexAttribArray(static_cast<GLuint>(index))) {
                    cache->enabledVertexAttribs &= ~shiftedIndexBit;
                    return true;
                }
            }

            return false;
        }
        else {
            FEATSTD_DEBUG_ASSERT(index < maxCachedVertexAttribsCount);
            FEATSTD_LOG_ERROR("RenderStateCache can hold %d vertex attribs, but %d are used. Cache needs to be extended.",
                maxCachedVertexAttribsCount, index);
            return false;
        }
    }

    return enable ? EnableVertexAttribArray(static_cast<GLuint>(index)) : DisableVertexAttribArray(static_cast<GLuint>(index));
}

bool RenderDevice::ActivateAttribute(UInt index)
{
    return SetVertexAttribArray(GetCurrentRenderStateCache(), index, /* enable = */true);
}

bool RenderDevice::DeactivateAttribute(UInt index)
{
    return SetVertexAttribArray(GetCurrentRenderStateCache(), index, /* enable = */false);
}

bool RenderDevice::GetActiveUniformCount(Handle programHandle, Int& count)
{
    return GetProgramParameter(programHandle, GL_ACTIVE_UNIFORMS, &count);
}

bool RenderDevice::GetActiveUniformMaxLength(Handle programHandle, Int& maxLength)
{
    return GetProgramParameter(programHandle, GL_ACTIVE_UNIFORM_MAX_LENGTH, &maxLength);
}

bool RenderDevice::GetActiveUniform(Handle programHandle, Int index, Int nameBufferSize, Char* name,  Int& nameLength, Int& size, Shader::UniformType& type)
{
    GLOBAL_EXPERIMENT_COND_USE_DEFAULT_SHADER_INSTEAD_OF(programHandle);

    GLenum glType = GL_FLOAT;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetActiveUniform, (static_cast<GLuint>(programHandle),
                       static_cast<UInt32>(index),
                       static_cast<GLsizei>(nameBufferSize),
                       &nameLength,
                       &size,
                       &glType,
                       name));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetActiveUniform failed.");
    type = GlTypeMapper::MapGlUniformType(glType);
    return true;
}

const Char* RenderDevice::GetDeviceInfo(DeviceInfo deviceInfo)
{
    const GLubyte* deviceInfoString = 0;
    FEATSTD_LINT_NEXT_EXPRESSION(838, "Violates MISRA C++ 2008 Required Rule 0-1-9: previously assigned value")
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(deviceInfoString, glGetString, (GlTypeMapper::MapDeviceInfo(deviceInfo)));
    // In case glGetString fails, a null string is returned.
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetString failed.")
    return FeatStd::Internal::PointerToPointer<const Char*>(deviceInfoString);
}

UInt RenderDevice::GetMaxTextureUnitsSupportedByDevice()
{
    GLint count = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_TEXTURE_IMAGE_UNITS, &count));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
    return static_cast<UInt>(count);
}

UInt RenderDevice::GetMaxTextureUnitsSupportedByCandera()
{
    static UInt maxTextureUnitsSupported = (CANDERA_MAX_TEXTURE_UNIT_COUNT <= GetMaxTextureUnitsSupportedByDevice()) ? CANDERA_MAX_TEXTURE_UNIT_COUNT : GetMaxTextureUnitsSupportedByDevice();
    return maxTextureUnitsSupported;
}

Float RenderDevice::GetMaxAnisotropySupportedByDevice()
{
    GLfloat degree = 1.0F;
#if GL_EXT_texture_filter_anisotropic
    // Retrieve max degree of anisotropy.
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetFloatv, (GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &degree));
    CANDERA_GL_CHECK_ERROR_NO_RETURN(
                                     "glGetFloatv for anisotropic Filter failed. GL error:\n"
                                     "Check if extension GL_EXT_texture_filter_anisotropic is supported.");
#endif
    return static_cast<Float>(degree);
}

UInt RenderDevice::GetMaxTextureSizeSupportedByDevice()
{
    GLint size = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_TEXTURE_SIZE, &size));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
    return static_cast<UInt>(size);
}

UInt RenderDevice::GetMaxCombinedTextureUnitsSupportedByDevice()
{
    GLint units = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS, &units));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
        return static_cast<UInt>(units);
}

UInt RenderDevice::GetMaxCubeMapTextureSizeSupportedByDevice()
{
    GLint size = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_CUBE_MAP_TEXTURE_SIZE, &size));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
        return static_cast<UInt>(size);
}

UInt RenderDevice::GetMaxFragmentUniformVectorsSupportedByDevice()
{
    GLint vectors = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_FRAGMENT_UNIFORM_VECTORS, &vectors));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
        return static_cast<UInt>(vectors);
}

UInt RenderDevice::GetMaxRenderBufferSizeSupportedByDevice()
{
    GLint size = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_RENDERBUFFER_SIZE, &size));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
        return static_cast<UInt>(size);
}

UInt RenderDevice::GetMaxVaryingVectorsSupportedByDevice()
{
    GLint vectors = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_VARYING_VECTORS, &vectors));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
        return static_cast<UInt>(vectors);
}

UInt RenderDevice::GetMaxVertexAttribsSupportedByDevice()
{
    GLint attribs = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_VERTEX_ATTRIBS, &attribs));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
        return static_cast<UInt>(attribs);
}

UInt RenderDevice::GetMaxVertexTextureUnitsSupportedByDevice()
{
    GLint units = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS, &units));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
        return static_cast<UInt>(units);
}

UInt RenderDevice::GetMaxVertexUniformVectorsSupportedByDevice()
{
    GLint vectors = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_VERTEX_UNIFORM_VECTORS, &vectors));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")
    return static_cast<UInt>(vectors);
}

void RenderDevice::GetMaxViewportSizeSupportedByDevice(UInt32& width, UInt32& height)
{
    GLint viewPort[2] = {0, 0};
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_VIEWPORT_DIMS, viewPort));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glGetIntegerv failed.")

    width = static_cast<UInt32>(viewPort[0]);
    height = static_cast<UInt32>(viewPort[1]);
}

UInt32 RenderDevice::GetMaxIndexElementSupportedByDevice()
{
#ifdef GL_OES_element_index_uint
    return 0xFFFFFF;
#else
    return 0xFFFF;
#endif
}

bool RenderDevice::IsDxtCompressionSupportedByDevice()
{
#ifdef GL_EXT_texture_compression_s3tc
    return true;
#else
    return false;
#endif
}

bool RenderDevice::ActivateColorWriteEnabled(bool enableRed, bool enableGreen, bool enableBlue, bool enableAlpha)
{
    if (GetCurrentRenderStateCache() != 0) {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(731, CANDERA_LINT_REASON_BOOLEANEQUALITY)
        if ((enableRed != GetCurrentRenderStateCache()->isRedWriteEnabled)   || (enableGreen != GetCurrentRenderStateCache()->isGreenWriteEnabled) ||
            (enableBlue != GetCurrentRenderStateCache()->isBlueWriteEnabled) || (enableAlpha != GetCurrentRenderStateCache()->isAlphaWriteEnabled)) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glColorMask, (static_cast<GLboolean>(enableRed), static_cast<GLboolean>(enableGreen),
                            static_cast<GLboolean>(enableBlue), static_cast<GLboolean>(enableAlpha)));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glColorMask failed.");

            GetCurrentRenderStateCache()->isRedWriteEnabled = enableRed;
            GetCurrentRenderStateCache()->isGreenWriteEnabled = enableGreen;
            GetCurrentRenderStateCache()->isBlueWriteEnabled = enableBlue;
            GetCurrentRenderStateCache()->isAlphaWriteEnabled = enableAlpha;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glColorMask, (static_cast<GLboolean>(enableRed), static_cast<GLboolean>(enableGreen),
                    static_cast<GLboolean>(enableBlue), static_cast<GLboolean>(enableAlpha)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glColorMask failed.");
    }

    return true;
}

bool RenderDevice::ActivateBlendColor(const Color& color)
{
    //Set constant blending color if different to color in cache
    if (GetCurrentRenderStateCache() != 0) {
        if (color != GetCurrentRenderStateCache()->blendColor) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBlendColor, (color.GetRed(), color.GetGreen(), color.GetBlue(), color.GetAlpha()));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBlendColor failed.");
            GetCurrentRenderStateCache()->blendColor = color;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBlendColor, (color.GetRed(), color.GetGreen(), color.GetBlue(), color.GetAlpha()));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBlendColor failed.");
    }
    return true;
}

bool RenderDevice::ActivateBlendMode(RenderMode::BlendFactor sourceFactor, RenderMode::BlendFactor destFactor, RenderMode::BlendOperation operation)
{
    return ActivateBlendModeSeparate(sourceFactor, destFactor, operation, sourceFactor, destFactor, operation);
}

bool RenderDevice::ActivateBlendModeSeparate(RenderMode::BlendFactor sourceRGBFactor, RenderMode::BlendFactor destRGBFactor, RenderMode::BlendOperation operationRGB,
                                             RenderMode::BlendFactor sourceAlphaFactor, RenderMode::BlendFactor destAlphaFactor, RenderMode::BlendOperation operationAlpha)
{
    if (GetCurrentRenderStateCache() != 0) {
        if ((sourceRGBFactor != GetCurrentRenderStateCache()->blendSrcRGB) || (destRGBFactor != GetCurrentRenderStateCache()->blendDstRGB) ||
           (sourceAlphaFactor != GetCurrentRenderStateCache()->blendSrcAlpha) || (destAlphaFactor != GetCurrentRenderStateCache()->blendDstAlpha)) {
            {
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBlendFuncSeparate, (GlTypeMapper::MapBlendFactor(sourceRGBFactor), GlTypeMapper::MapBlendFactor(destRGBFactor),
                                    GlTypeMapper::MapBlendFactor(sourceAlphaFactor), GlTypeMapper::MapBlendFactor(destAlphaFactor)));
            }
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBlendFuncSeparate failed.");
            GetCurrentRenderStateCache()->blendSrcRGB = sourceRGBFactor;
            GetCurrentRenderStateCache()->blendDstRGB = destRGBFactor;
            GetCurrentRenderStateCache()->blendSrcAlpha = sourceAlphaFactor;
            GetCurrentRenderStateCache()->blendDstAlpha = destAlphaFactor;
        }
        if ((operationRGB != GetCurrentRenderStateCache()->blendOpRGB) || (operationAlpha != GetCurrentRenderStateCache()->blendOpAlpha)) {
            {
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBlendEquationSeparate, (GlTypeMapper::MapBlendOperation(operationRGB), GlTypeMapper::MapBlendOperation(operationAlpha)));
            }
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBlendEquationSeparate failed.");
            GetCurrentRenderStateCache()->blendOpRGB = operationRGB;
            GetCurrentRenderStateCache()->blendOpAlpha = operationAlpha;
        }
    }
    else {
        {
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBlendFuncSeparate, (GlTypeMapper::MapBlendFactor(sourceRGBFactor), GlTypeMapper::MapBlendFactor(destRGBFactor),
                                GlTypeMapper::MapBlendFactor(sourceAlphaFactor), GlTypeMapper::MapBlendFactor(destAlphaFactor)));
        }
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBlendFuncSeparate failed.");

        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glBlendEquationSeparate, (GlTypeMapper::MapBlendOperation(operationRGB), GlTypeMapper::MapBlendOperation(operationAlpha)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glBlendEquationSeparate failed.");
    }

    return true;
}

bool RenderDevice::ActivateSampleCoverage(Float coverage, bool invert)
{
    // Set coverage factor and invert value if different to cached values.
    if (GetCurrentRenderStateCache() != 0) {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(731, CANDERA_LINT_REASON_BOOLEANEQUALITY)
        if ((!(Math::FloatAlmostEqual(coverage, GetCurrentRenderStateCache()->sampleCoverage))) || (invert != GetCurrentRenderStateCache()->isSampleCoverageInverted)) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glSampleCoverage, (static_cast<GLclampf>(coverage), static_cast<GLboolean>(invert)));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glSampleCoverage failed.");
            GetCurrentRenderStateCache()->sampleCoverage = coverage;
            GetCurrentRenderStateCache()->isSampleCoverageInverted = invert;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glSampleCoverage, (static_cast<GLclampf>(coverage), static_cast<GLboolean>(invert)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glSampleCoverage failed.");
    }

    return true;
}

bool RenderDevice::ActivateStencilFunction(RenderMode::ComparisonFunction compFunc, Int32 refValue, UInt32 mask, RenderMode::StencilPlane face)
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
    GLenum glCompFunc = GlTypeMapper::MapComparisonFunction(compFunc);

    if (GetCurrentRenderStateCache() != 0) {
        bool hasStateChanged = false;

        if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::FrontFace)) {
            if ((GetCurrentRenderStateCache()->stencilFunctionFrontFace != glCompFunc) ||
                (GetCurrentRenderStateCache()->stencilFunctionRefValueFrontFace != refValue) ||
                (GetCurrentRenderStateCache()->stencilFunctionMaskFrontFace != mask)) {
                hasStateChanged = true;
            }
        }
        if (((face == RenderMode::FrontAndBackFace) || (face == RenderMode::BackFace)) && (!hasStateChanged)) {
            if ((GetCurrentRenderStateCache()->stencilFunctionBackFace != glCompFunc) ||
                (GetCurrentRenderStateCache()->stencilFunctionRefValueBackFace != refValue) ||
                (GetCurrentRenderStateCache()->stencilFunctionMaskBackFace != mask)) {
                hasStateChanged = true;
             }
        }
        if (hasStateChanged) {
            {
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glStencilFuncSeparate, (GlTypeMapper::MapStencilPlane(face), glCompFunc, refValue, mask));
            }
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glStencilFuncSeparate failed.");

            if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::FrontFace)) {
                    GetCurrentRenderStateCache()->stencilFunctionFrontFace = glCompFunc;
                    GetCurrentRenderStateCache()->stencilFunctionRefValueFrontFace = refValue;
                    GetCurrentRenderStateCache()->stencilFunctionMaskFrontFace = mask;
            }
            if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::BackFace)) {
                    GetCurrentRenderStateCache()->stencilFunctionBackFace = glCompFunc;
                    GetCurrentRenderStateCache()->stencilFunctionRefValueBackFace = refValue;
                    GetCurrentRenderStateCache()->stencilFunctionMaskBackFace = mask;
            }
        }
    }
    else {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glStencilFuncSeparate, (GlTypeMapper::MapStencilPlane(face), glCompFunc, refValue, mask));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glStencilFuncSeparate failed.");
    }
    return true;
}

bool RenderDevice::ActivateStencilOperation(RenderMode::StencilOperation stencilFail, RenderMode::StencilOperation depthFail,
        RenderMode::StencilOperation depthPass, RenderMode::StencilPlane face)
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
    GLenum glStencilFail = GlTypeMapper::MapStencilOperation(stencilFail);
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
    GLenum glDepthFail = GlTypeMapper::MapStencilOperation(depthFail);
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
    GLenum glDepthPass = GlTypeMapper::MapStencilOperation(depthPass);

    if (GetCurrentRenderStateCache() != 0) {
        bool hasStateChanged = false;

        if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::FrontFace)) {
            if ((GetCurrentRenderStateCache()->stencilOperationStencilFailFrontFace != glStencilFail) ||
                (GetCurrentRenderStateCache()->stencilOperationDepthFailFrontFace != glDepthFail) ||
                (GetCurrentRenderStateCache()->stencilOperationDepthPassFrontFace != glDepthPass)) {
                    hasStateChanged = true;
            }
        }
        if (((face == RenderMode::FrontAndBackFace) || (face == RenderMode::BackFace)) && (!hasStateChanged)) {
            if ((GetCurrentRenderStateCache()->stencilOperationStencilFailBackFace != glStencilFail) ||
                (GetCurrentRenderStateCache()->stencilOperationDepthFailBackFace != glDepthFail) ||
                (GetCurrentRenderStateCache()->stencilOperationDepthPassBackFace != glDepthPass)) {
                    hasStateChanged = true;
            }
        }
        if (hasStateChanged) {
            {
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glStencilOpSeparate, (GlTypeMapper::MapStencilPlane(face), glStencilFail, glDepthFail, glDepthPass));
            }
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glStencilOpSeparate failed.");
            if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::FrontFace)) {
                GetCurrentRenderStateCache()->stencilOperationStencilFailFrontFace = glStencilFail;
                GetCurrentRenderStateCache()->stencilOperationDepthFailFrontFace = glDepthFail;
                GetCurrentRenderStateCache()->stencilOperationDepthPassFrontFace = glDepthPass;
                }
            if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::BackFace)) {
                GetCurrentRenderStateCache()->stencilOperationStencilFailBackFace = glStencilFail;
                GetCurrentRenderStateCache()->stencilOperationDepthFailBackFace = glDepthFail;
                GetCurrentRenderStateCache()->stencilOperationDepthPassBackFace = glDepthPass;
            }
        }
    }
    else {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
         CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glStencilOpSeparate, (GlTypeMapper::MapStencilPlane(face), glStencilFail, glDepthFail, glDepthPass));
         CANDERA_GL_CHECK_ERROR_LOG_ERROR("glStencilOpSeparate failed.");
    }

    return true;
}

bool RenderDevice::ActivateStencilWriteMask(UInt32 mask, RenderMode::StencilPlane face)
{
    if (GetCurrentRenderStateCache() != 0) {
        bool hasStateChanged = false;

        if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::FrontFace)) {
            if (GetCurrentRenderStateCache()->stencilWriteMaskFrontFace != mask) {
                hasStateChanged = true;
            }
        }
        if (((face == RenderMode::FrontAndBackFace) || (face == RenderMode::BackFace)) && (!hasStateChanged)) {
            if (GetCurrentRenderStateCache()->stencilWriteMaskBackFace != mask) {
                hasStateChanged = true;
            }
        }
        if (hasStateChanged) {
            {
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glStencilMaskSeparate, (GlTypeMapper::MapStencilPlane(face), mask));
            }
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glStencilMaskSeparate failed.");

            if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::FrontFace)) {
                GetCurrentRenderStateCache()->stencilWriteMaskFrontFace = mask;
            }
            if ((face == RenderMode::FrontAndBackFace) || (face == RenderMode::BackFace)) {
                GetCurrentRenderStateCache()->stencilWriteMaskBackFace = mask;
            }
        }
    }
    else {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, CANDERA_LINT_REASON_ENUMCONVERSION)
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glStencilMaskSeparate, (GlTypeMapper::MapStencilPlane(face), mask));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glStencilMaskSeparate failed.");
    }

    return true;
}

bool RenderDevice::ActivateCoverageOperation(RenderMode::CoverageOperation operation)
{
#if GL_NV_coverage_sample
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCoverageOperationNV, (GlTypeMapper::MapCoverageOperation(operation)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCoverageOperationNV failed for coverage buffer. GL error: Is extension GL_NV_coverage_sample supported?");
    return true;
#else
    FEATSTD_UNUSED(operation);
    return false;
#endif
}

bool RenderDevice::ActivateRenderState(RenderState state, UInt32 value)
{
    switch (state) {
        // Culling.
        case CullFaceMode: {
            RenderMode::Culling culling = static_cast<RenderMode::Culling>(value);
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (culling != GetCurrentRenderStateCache()->culling))) {
                if (culling != RenderMode::NoCulling) {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCullFace, (GlTypeMapper::MapCullFaceMode(value)));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCullFace failed.");

                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_CULL_FACE));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_CULL_FACE) failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_CULL_FACE));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_CULL_FACE) failed.");
                }

                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->culling = culling;
                }
            }
            break;
        }

        // Dithering.
        case DitheringEnable: {
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (GetCurrentRenderStateCache()->isDitheringEnabled != value))) {
                if (value != 0){
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_DITHER));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_DITHER) failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_DITHER));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_DITHER) failed.");
                }

                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->isDitheringEnabled = value;
                }
            }

            break;
        }

        // Winding.
        case WindingOrder: {
            //Set winding order if different from cache
            if (GetCurrentRenderStateCache() != 0) {
                if (GetCurrentRenderStateCache()->frontFace !=  static_cast<RenderMode::Winding>(value)) {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glFrontFace, (GlTypeMapper::MapWindingOrder(value)));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glFrontFace failed.");
                    GetCurrentRenderStateCache()->frontFace = static_cast<RenderMode::Winding>(value);
                }
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glFrontFace, (GlTypeMapper::MapWindingOrder(value)));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glFrontFace failed.");
            }
            break;
        }

        // DepthWrite.
        case DepthWriteEnable: {
            //Set depth buffer writes if different from cache
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (GetCurrentRenderStateCache()->isDepthWriteEnabled != value))) {
                if (value != 0){
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDepthMask, (GL_TRUE));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDepthMask(GL_TRUE) failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDepthMask, (GL_FALSE));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDepthMask(GL_FALSE) failed.");
                }

                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->isDepthWriteEnabled = value;
                }
            }
            break;
        }

        // Depth test.
        case DepthTestEnable: {
            //Set depth buffer tests if different from cache
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (GetCurrentRenderStateCache()->isDepthTestEnabled != value))) {
                if (value != 0){
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_DEPTH_TEST));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_DEPTH_TEST) failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_DEPTH_TEST));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_DEPTH_TEST)) failed.");
                }

                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->isDepthTestEnabled = value;
                }
            }
            break;
        }

        // DepthCompareFunc.
        case DepthComparisonFunction: {
            //Set depth test compare function if different from cache
            if (GetCurrentRenderStateCache() != 0) {
                if (GetCurrentRenderStateCache()->depthComparisonFunction != static_cast<RenderMode::ComparisonFunction>(value)) {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDepthFunc, (GlTypeMapper::MapComparisonFunction(value)));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDepthFunc failed.");
                    GetCurrentRenderStateCache()->depthComparisonFunction = static_cast<RenderMode::ComparisonFunction>(value);
                }
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDepthFunc, (GlTypeMapper::MapComparisonFunction(value)));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDepthFunc failed.");
            }
            break;
        }

        // Stencil test.
        case StencilTestEnable: {
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (GetCurrentRenderStateCache()->isStencilTestEnabled != value))) {
                if (value != 0) {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_STENCIL_TEST));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_STENCIL_TEST failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_STENCIL_TEST));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_STENCIL_TEST) failed.");
                }
                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->isStencilTestEnabled = value;
                }
            }
            break;
        }

        // Scissor test.
        case ScissorTestEnable: {
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (GetCurrentRenderStateCache()->isScissorTestEnabled != value))) {
                if (value != 0){
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_SCISSOR_TEST));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_SCISSOR_TEST) failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_SCISSOR_TEST));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_SCISSOR_TEST) failed.");
                }
                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->isScissorTestEnabled = value;
                }
            }
            break;
        }

        // Blending.
        case BlendingEnable: {
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (GetCurrentRenderStateCache()->isBlendingEnabled != value))) {
                if (value != 0){
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_BLEND));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_BLEND) failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_BLEND));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_BLEND) failed.");
                }
                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->isBlendingEnabled = value;
                }
            }
            break;
        }

        // Multisampling coverage mask.
        case SampleCoverageMaskEnable: {
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (GetCurrentRenderStateCache()->isSampleCoverageMaskEnabled != value))) {
                if (value != 0){
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_SAMPLE_COVERAGE));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_SAMPLE_COVERAGE) failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_SAMPLE_COVERAGE));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_SAMPLE_COVERAGE) failed.");
                }

                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->isSampleCoverageMaskEnabled = value;
                }
            }
            break;
        }

        // Alpha value to multisampling coverage mask.
        case SampleAlphaToCoverageMaskEnable: {
            if ((GetCurrentRenderStateCache() == 0 ) || ((GetCurrentRenderStateCache() != 0) && (GetCurrentRenderStateCache()->isSampleAlphaToCoverageMaskEnabled != value))) {
                if (value != 0){
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_SAMPLE_ALPHA_TO_COVERAGE));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_SAMPLE_ALPHA_TO_COVERAGE) failed.");
                }
                else {
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_SAMPLE_ALPHA_TO_COVERAGE));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_SAMPLE_ALPHA_TO_COVERAGE) failed.");
                }
                if (GetCurrentRenderStateCache() != 0) {
                    GetCurrentRenderStateCache()->isSampleAlphaToCoverageMaskEnabled = value;
                }
            }
            break;
        }

        // Enable coverage-buffer writes.
        case CoverageWriteEnable:
            // No caching of this extension yet, as caching on platforms that support this extension is normally done by driver anyway.
#if GL_NV_coverage_sample
            if (value != 0) {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCoverageMaskNV, (GL_TRUE));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCoverageMaskNV(GL_TRUE) failed. GL error: Is extension GL_NV_coverage_sample supported?");
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glCoverageMaskNV, (GL_FALSE));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glCoverageMaskNV(GL_FALSE) failed. GL error: Is extension GL_NV_coverage_sample supported?");
            }
#endif
            break;

        case RasterizerDiscardEnable:
            if (value != 0) {
                // Cannot enable because the feature is not available in ES2.0
                return false;
            }
            break;

        default:
            return false;
    }

    return true;
}

UInt32 RenderDevice::GetRenderState(RenderState state)
{
    UInt32 renderStateValue;

    switch (state) {
        // Blending enabled.
        case BlendingEnable: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue =  GetCurrentRenderStateCache()->isBlendingEnabled;
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(renderStateValue, glIsEnabled, (GL_BLEND));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_BLEND) failed.");
            }
            break;
        }

        // Cull face mode.
        case CullFaceMode: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->culling;
            }
            else {
                GLboolean isCullingEnabled;
                CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(isCullingEnabled, glIsEnabled, (GL_CULL_FACE));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_CULL_FACE) failed.");
                if (isCullingEnabled == GL_TRUE) {
                    GLint cullMode = 0;
                    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_CULL_FACE_MODE, &cullMode));
                    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_CULL_FACE_MODE, &cullMode) failed.");
                    if (cullMode == GL_FRONT) {
                        renderStateValue = RenderMode::FrontFaceCulling;
                    }
                    else if (cullMode == GL_BACK) {
                        renderStateValue = RenderMode::BackFaceCulling;
                    }
                    else {
                        //Note: GL_FRONT_AND_BACK is deliberately no RenderMode enum in Candera.
                        renderStateValue = c_glInvalidParameter;
                }
                }
                else {
                    renderStateValue = RenderMode::NoCulling;
                }
            }
            break;
        }

        // Depth write enabled.
        case DepthWriteEnable: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->isDepthWriteEnabled;
            }
            else {
                GLboolean value = GL_FALSE;
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetBooleanv, (GL_DEPTH_WRITEMASK, &value));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetBooleanv(GL_DEPTH_WRITEMASK, &value) failed.");
                renderStateValue = value;
            }
            break;
        }

        // Depth test enabled.
        case DepthTestEnable: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->isDepthTestEnabled;
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(renderStateValue, glIsEnabled, (GL_DEPTH_TEST));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_DEPTH_TEST) failed.");
            }
            break;
        }

        // Depth comparison function.
        case DepthComparisonFunction: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->depthComparisonFunction;
            }
            else {
                GLint value = 0;
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_DEPTH_FUNC, &value));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_DEPTH_FUNC, &value) failed.");
                //Map to RenderMode::ComparisonFunction enumerator values.
                renderStateValue = static_cast<UInt32>(static_cast<GLenum>(value) -
                    GlTypeMapper::MapComparisonFunction(RenderMode::CompareNever));
            }
            break;
        }

        // Dithering enabled.
        case DitheringEnable: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->isDitheringEnabled;
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(renderStateValue, glIsEnabled, (GL_DITHER));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_DITHER) failed.");
            }
            break;
        }

        // Scissor test enabled.
        case ScissorTestEnable: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->isScissorTestEnabled;
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(renderStateValue, glIsEnabled, (GL_SCISSOR_TEST));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_SCISSOR_TEST) failed.");
            }
            break;
        }

        // Sample coverage mask enabled.
        case SampleCoverageMaskEnable: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->isSampleCoverageMaskEnabled;
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(renderStateValue, glIsEnabled, (GL_SAMPLE_COVERAGE));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_SAMPLE_COVERAGE) failed.");
            }
            break;
        }

        // Sample alpha to coverage mask enabled.
        case SampleAlphaToCoverageMaskEnable: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->isSampleAlphaToCoverageMaskEnabled;
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(renderStateValue, glIsEnabled, (GL_SAMPLE_ALPHA_TO_COVERAGE));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_SAMPLE_ALPHA_TO_COVERAGE) failed.");
            }
            break;
        }

        // Stencil test enabled.
        case StencilTestEnable: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->isStencilTestEnabled;
            }
            else {
                CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(renderStateValue, glIsEnabled, (GL_STENCIL_TEST));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_STENCIL_TEST) failed.");
            }
            break;
        }

        // Winding order of front face.
        case WindingOrder: {
            if (GetCurrentRenderStateCache() != 0) {
                renderStateValue = GetCurrentRenderStateCache()->frontFace;
            }
            else {
                GLint value = 0;
                CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_FRONT_FACE, &value));
                CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_FRONT_FACE, &value) failed.");
                //Map to RenderMode::Winding enumerator values.
                renderStateValue = static_cast<UInt32>(static_cast<GLenum>(value) -
                    GlTypeMapper::MapComparisonFunction(RenderMode::ClockWise));
            }
            break;
        }

        default:
            // valid path, as getter for CoverageWriteEnable is not supported by extension GL_NV_coverage_sample.
            renderStateValue = c_glInvalidParameter;
            FEATSTD_LOG_ERROR("GetRenderState requested for invalid state.");
            break;
    }

    return renderStateValue;
}

bool RenderDevice::ActivateDepthBias(Float scaleFactor, Float units)
{
    RenderStateCache* currentRenderStateCache = GetCurrentRenderStateCache();
    if ((currentRenderStateCache == 0) ||
        ((!Math::FloatAlmostEqual(currentRenderStateCache->depthBiasScaleFactor, scaleFactor)) ||
         (!(Math::FloatAlmostEqual(currentRenderStateCache->depthBiasUnits, units))))) {
        if ((scaleFactor == 0.0F) && (units == 0.0F)) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDisable, (GL_POLYGON_OFFSET_FILL));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDisable(GL_POLYGON_OFFSET_FILL) failed.");
        }
        else {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glEnable, (GL_POLYGON_OFFSET_FILL));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glEnable(GL_POLYGON_OFFSET_FILL) failed.");

            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glPolygonOffset, (static_cast<GLfloat>(scaleFactor), static_cast<GLfloat>(units)));
            CANDERA_GL_CHECK_ERROR_LOG_ERROR("glPolygonOffset failed.");
        }
        if (currentRenderStateCache != 0) {
            currentRenderStateCache->depthBiasScaleFactor = scaleFactor;
            currentRenderStateCache->depthBiasUnits = units;
        }
    }

    return true;
}

void RenderDevice::ActivateViewport(const Rectangle &viewport, Float renderTargetWidth, Float renderTargetHeight)
{
    // Create absolute OpenGL pixel coordinates out of normalized viewport.
    GLint x = static_cast<GLint>(viewport.GetLeft() * renderTargetWidth + 0.5F);
    GLint y = static_cast<GLint>((renderTargetHeight - ((viewport.GetTop() + viewport.GetHeight()) * renderTargetHeight)) + 0.5F);
    GLsizei width = static_cast<GLsizei>(viewport.GetWidth() * renderTargetWidth + 0.5F);
    GLsizei height = static_cast<GLsizei>(viewport.GetHeight() * renderTargetHeight + 0.5F);

    if (GetCurrentRenderStateCache() != 0) {
        if ((GetCurrentRenderStateCache()->viewportX != x) || (GetCurrentRenderStateCache()->viewportY != y) ||
            (GetCurrentRenderStateCache()->viewportWidth != width) || (GetCurrentRenderStateCache()->viewportHeight != height)) {
            CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glViewport, (x, y, width, height));
            CANDERA_GL_CHECK_ERROR_NO_RETURN("glViewport failed.")

            GetCurrentRenderStateCache()->viewportX = x;
            GetCurrentRenderStateCache()->viewportY = y;
            GetCurrentRenderStateCache()->viewportWidth = width;
            GetCurrentRenderStateCache()->viewportHeight = height;
        }
    }
    else {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glViewport, (x, y, width, height));
        CANDERA_GL_CHECK_ERROR_NO_RETURN("glViewport failed.")
    }
}

void RenderDevice::ActivateViewport(const Candera::Camera& camera)
{
    RenderTarget3D* renderTarget = camera.GetRenderTarget();

    if (renderTarget == 0) {
        FEATSTD_LOG_ERROR("Activate viewport failed, render target invalid.");
        return;
    }
    Rectangle viewport = camera.GetViewport();

    ActivateViewport(viewport,
        static_cast<Float>(renderTarget->GetActualWidth()),
        static_cast<Float>(renderTarget->GetActualHeight()));
}

static const Camera*& CurrentCamera()
{
    static const Camera* camera = 0;
    return camera;
}

void RenderDevice::SetActiveCamera(const Camera* camera)
{
    CurrentCamera() = camera;
}

const Camera* RenderDevice::GetActiveCamera()
{
    return CurrentCamera();
}

SizeType RenderDevice::GetNumberOfActiveLights()
{
    SizeType result = 0;
    if (m_activeLights != 0) {
        result = m_activeLights->GetSize();
    }
    return result;
}

const Light* RenderDevice::GetFirstActiveLight()
{
    if (m_activeLights != 0) {
        m_activeLightIterator = &*m_activeLights->Begin();
    }
    return (m_activeLightIterator != 0) ? m_activeLightIterator : 0;
}

const Light* RenderDevice::GetNextActiveLight()
{
    if (m_activeLightIterator != 0) {
        m_activeLightIterator = m_activeLightIterator->LightListNode.GetNext();
    }
    return m_activeLightIterator;
}

void RenderDevice::SetActiveLights(const RenderDevice::LightList *lightList)
{
    m_activeLights = lightList;
}

bool RenderDevice::DrawVertexBuffer(const VertexBuffer& vertexBuffer, SizeType elementStart, SizeType elementCount)
{
    const VertexGeometry* geometry = vertexBuffer.GetVertexGeometry();
    if (geometry == 0) {
        return false;
    }

    if (geometry->GetBufferType() == VertexGeometry::ArrayBuffer) {
        if ((elementStart + elementCount) > geometry->GetVertexCount()) {
            FEATSTD_DEBUG_FAIL();
            return false;
        }

        GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawArrays, (GlTypeMapper::MapVertexPrimitiveType(vertexBuffer.GetPrimitiveType()),
            static_cast<GLint>(elementStart), static_cast<GLsizei>(elementCount)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawArrays(...) failed.");
    }
    else {
        Internal::ResourceData indexArrayResource(ResourceDataHandle::InvalidHandle());
        if (geometry->GetMemoryPool() == VertexGeometry::SystemMemory) {
            indexArrayResource = Internal::ResourceData(geometry->GetIndexArrayResourceHandle());
        }

        GLenum type = GlTypeMapper::MapVertexIndexFormat(geometry->GetBufferType());
        const void* indices = indexArrayResource.GetData();
        if (GL_UNSIGNED_BYTE == type) {
            indices = FeatStd::Internal::PointerToPointer<const UInt8*>(indices) + elementStart;
        }
        else if (GL_UNSIGNED_SHORT == type) {
            indices = FeatStd::Internal::PointerToPointer<const UInt16*>(indices) + elementStart;
        }
        else {
            FEATSTD_DEBUG_FAIL();
            return false;
        }

        if ((elementStart + elementCount) > geometry->GetIndexCount()) {
            FEATSTD_DEBUG_FAIL();
            return false;
        }

        //Note: As this assumption already occured in some bug reports:
        //If the index buffer is in VideoMemory, glDrawElements parameter indices is only treated as an offset by OpenGL ES.
        //Therefore indices == 0 is not a problem in this case.
        GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawElements, (GlTypeMapper::MapVertexPrimitiveType(vertexBuffer.GetPrimitiveType()),
            static_cast<GLsizei>(elementCount), type, indices));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawElements(...) failed.");
    }

    return true;
}

bool RenderDevice::DrawInstancedVertexBuffer(const VertexBuffer& vertexBuffer, UInt32 instanceCount,
    SizeType elementStart, SizeType elementCount)
{
    FEATSTD_UNUSED4(vertexBuffer, instanceCount, elementStart, elementCount);
    return false;
}

bool RenderDevice::DrawTriangles(UInt32 vertexCount)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawArrays, (GL_TRIANGLES, 0, static_cast<Int>(vertexCount)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawArrays(GL_TRIANGLES, ...) failed.");
        return true;
}

bool RenderDevice::DrawTrianglesIndexed(UInt32 indexCount, const void* indices)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawElements, (GL_TRIANGLES, static_cast<Int>(indexCount), GL_UNSIGNED_SHORT, indices));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawElements(GL_TRIANGLES, ...) failed.");
    return true;
}

bool RenderDevice::DrawTriangleStrip(UInt32 vertexCount)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawArrays, (GL_TRIANGLE_STRIP, 0, static_cast<Int>(vertexCount)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawArrays(GL_TRIANGLE_STRIP, ...) failed.");
        return true;
}

bool RenderDevice::DrawTriangleStripIndexed(UInt32 indexCount, const void* indices)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawElements, (GL_TRIANGLE_STRIP, static_cast<Int>(indexCount), GL_UNSIGNED_SHORT, indices));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawElements(GL_TRIANGLE_STRIP, ...) failed.");
    return true;
}

bool RenderDevice::DrawTriangleFan(UInt32 vertexCount)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawArrays, (GL_TRIANGLE_FAN, 0, static_cast<Int>(vertexCount)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawArrays(GL_TRIANGLE_FAN, ...) failed.");
        return true;
}

bool RenderDevice::DrawTriangleFanIndexed(UInt32 indexCount, const void* indices)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawElements, (GL_TRIANGLE_FAN, static_cast<Int>(indexCount), GL_UNSIGNED_SHORT, indices));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawElements(GL_TRIANGLE_FAN, ...) failed.");
        return true;
}

bool RenderDevice::SetLineWidth(Float lineWidth)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glLineWidth, (lineWidth));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glLineWidth failed.");
    return true;
}

bool RenderDevice::DrawLines(UInt32 vertexCount)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawArrays, (GL_LINES, 0, static_cast<Int>(vertexCount)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawArrays(GL_LINES, ...) failed.");
    return true;
}

bool RenderDevice::DrawLinesIndexed(UInt32 indexCount, const void* indices)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawElements, (GL_LINES, static_cast<Int>(indexCount), GL_UNSIGNED_SHORT, indices));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawElements(GL_LINES, ...) failed.");
    return true;
}

bool RenderDevice::DrawLineStrip(UInt32 vertexCount)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawArrays, (GL_LINE_STRIP, 0, static_cast<Int>(vertexCount)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawArrays(GL_LINE_STRIP, ...) failed.");
    return true;
}

bool RenderDevice::DrawLineStripIndexed(UInt32 indexCount, const void* indices)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawElements, (GL_LINE_STRIP, static_cast<Int>(indexCount), GL_UNSIGNED_SHORT, indices));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawElements(GL_LINE_STRIP, ...) failed.");
    return true;
}

bool RenderDevice::DrawLineLoop(UInt32 vertexCount)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawArrays, (GL_LINE_LOOP, 0, static_cast<Int>(vertexCount)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawArrays(GL_LINE_LOOP, ...) failed.");
    return true;
}

bool RenderDevice::DrawLineLoopIndexed(UInt32 indexCount, const void* indices)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawElements, (GL_LINE_LOOP, static_cast<Int>(indexCount), GL_UNSIGNED_SHORT, indices));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawElements(GL_LINE_LOOP, ...) failed.");
    return true;
}

bool RenderDevice::DrawPoints(UInt32 vertexCount)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawArrays, (GL_POINTS, 0, static_cast<Int>(vertexCount)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawArrays(GL_POINTS, ...) failed.");
        return true;
}

bool RenderDevice::DrawPointsIndexed(UInt32 indexCount, const void* indices)
{
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDrawElements, (GL_POINTS, static_cast<Int>(indexCount), GL_UNSIGNED_SHORT, indices));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDrawElements(GL_POINTS, ...) failed.");
    return true;
}

void RenderDevice::Hint(HintTarget target, HintMode mode)
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glHint, (GlTypeMapper::MapHintTarget(target), GlTypeMapper::MapHintMode(mode)));
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glHint failed.")
}

bool RenderDevice::UploadQuery(Query&)
{
    FEATSTD_LOG_ERROR("Queries are not supported in OpenGL ES 2.0.");
    return false;
}


bool RenderDevice::UnloadQuery(Query&)
{
    FEATSTD_LOG_ERROR("Queries are not supported in OpenGL ES 2.0.");
    return false;
}


bool RenderDevice::BeginQuery(const Query&)
{
    FEATSTD_LOG_ERROR("Queries are not supported in OpenGL ES 2.0.");
    return false;
}


bool RenderDevice::EndQuery(const Query&)
{
    FEATSTD_LOG_ERROR("Queries are not supported in OpenGL ES 2.0.");
    return false;
}


bool RenderDevice::IsQueryResultAvailable(const Query&)
{
    FEATSTD_LOG_ERROR("Queries are not supported in OpenGL ES 2.0.");
    return false;
}

UInt32 RenderDevice::GetQueryResult(const Query&)
{
    FEATSTD_LOG_ERROR("Queries are not supported in OpenGL ES 2.0.");
    return 0xFFffFFff;
}


bool RenderDevice::IsQueryActive(const Query&)
{
    FEATSTD_LOG_ERROR("Queries are not supported in OpenGL ES 2.0.");
    return false;
}


void RenderDevice::Finish()
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glFinish, ());
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glFinish failed.")
}

void RenderDevice::Flush()
{
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glFlush, ());
    CANDERA_GL_CHECK_ERROR_NO_RETURN("glFlush failed.")
}

static bool ResyncClearColor(RenderStateCache* cache)
{
    GLfloat fValues[4] = {0.0F, 0.0F, 0.0F, 0.0F};
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetFloatv, (GL_COLOR_CLEAR_VALUE, fValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetFloatv(GL_COLOR_CLEAR_VALUE, ...) failed.");
    cache->clearColor = Color(fValues[0], fValues[1], fValues[2], fValues[3]);
    return true;
}

static bool ResyncClearDepth(RenderStateCache* cache)
{
    GLfloat fValues = 0.0F;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetFloatv, (GL_DEPTH_CLEAR_VALUE, &fValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetFloatv(GL_DEPTH_CLEAR_VALUE, ...) failed.");
    cache->clearDepth = fValues;
    return true;
}

static bool ResyncClearStencil(RenderStateCache* cache)
{
    GLint iValues = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_CLEAR_VALUE, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_CLEAR_VALUE, ...) failed.");
    cache->clearStencil = iValues;
    return true;
}

static bool ResyncColorWrite(RenderStateCache* cache)
{
    GLboolean bValues[4] = {GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE};
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetBooleanv, (GL_COLOR_WRITEMASK, bValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetBooleanv(GL_COLOR_WRITEMASK, ...) failed.");
    cache->isRedWriteEnabled = (bValues[0] != GL_FALSE);
    cache->isGreenWriteEnabled = (bValues[1] != GL_FALSE);
    cache->isBlueWriteEnabled = (bValues[2] != GL_FALSE);
    cache->isAlphaWriteEnabled = (bValues[3] != GL_FALSE);
    return true;
}

static bool ResyncBlendColor(RenderStateCache* cache)
{
    GLfloat fValues[4] = {0.0F, 0.0F, 0.0F, 0.0F};
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetFloatv, (GL_BLEND_COLOR, fValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetFloatv(GL_BLEND_COLOR, ...) failed.");
    cache->blendColor = Color(fValues[0], fValues[1], fValues[2], fValues[3]);
    return true;
}

static bool ResyncBlendMode(RenderStateCache* cache)
{
    GLint iValues = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_BLEND_SRC_RGB, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_BLEND_SRC_RGB, ...) failed.");
    cache->blendSrcRGB = GlTypeMapper::MapGlBlendFactor(iValues);

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_BLEND_DST_RGB, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_BLEND_DST_RGB, ...) failed.");
    cache->blendDstRGB = GlTypeMapper::MapGlBlendFactor(iValues);

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_BLEND_SRC_ALPHA, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_BLEND_SRC_ALPHA, ...) failed.");
    cache->blendSrcAlpha = GlTypeMapper::MapGlBlendFactor(iValues);

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_BLEND_DST_ALPHA, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_BLEND_DST_ALPHA, ...) failed.");
    cache->blendDstAlpha = GlTypeMapper::MapGlBlendFactor(iValues);

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_BLEND_EQUATION_RGB, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_BLEND_EQUATION_RGB, ...) failed.");
    cache->blendOpRGB = GlTypeMapper::MapGlBlendOperation(iValues);

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_BLEND_EQUATION_ALPHA, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_BLEND_EQUATION_ALPHA, ...) failed.");
    cache->blendOpAlpha = GlTypeMapper::MapGlBlendOperation(iValues);
    return true;
}

static bool ResyncSampleCoverage(RenderStateCache* cache)
{
    GLfloat fValues = 0.0F;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetFloatv, (GL_SAMPLE_COVERAGE_VALUE, &fValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetFloatv(GL_SAMPLE_COVERAGE_VALUE, ...) failed.");
    cache->sampleCoverage = fValues;

    GLboolean bValues = GL_FALSE;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetBooleanv, (GL_SAMPLE_COVERAGE_INVERT, &bValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetBooleanv(GL_SAMPLE_COVERAGE_INVERT, ...) failed.");
    cache->isSampleCoverageInverted = (bValues != GL_FALSE);
    return true;
}

static bool ResyncStencilFunction(RenderStateCache* cache)
{
    GLint iValues = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_FUNC, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_FUNC, ...) failed.");
    cache->stencilFunctionFrontFace = iValues;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_REF, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_REF, ...) failed.");
    cache->stencilFunctionRefValueFrontFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_VALUE_MASK, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_VALUE_MASK, ...) failed.");
    cache->stencilFunctionMaskFrontFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_BACK_FUNC, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_BACK_FUNC, ...) failed.");
    cache->stencilFunctionBackFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_BACK_REF, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_BACK_REF, ...) failed.");
    cache->stencilFunctionRefValueBackFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_BACK_VALUE_MASK, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_BACK_VALUE_MASK, ...) failed.");
    cache->stencilFunctionMaskBackFace = iValues;
    return true;
}

static bool ResyncStencilOperation(RenderStateCache* cache)
{
    GLint iValues = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_FAIL, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_FAIL, ...) failed.");
    cache->stencilOperationStencilFailFrontFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_PASS_DEPTH_FAIL, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_PASS_DEPTH_FAIL, ...) failed.");
    cache->stencilOperationDepthFailFrontFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_PASS_DEPTH_PASS, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_PASS_DEPTH_PASS, ...) failed.");
    cache->stencilOperationDepthPassFrontFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_BACK_FAIL, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_BACK_FAIL, ...) failed.");
    cache->stencilOperationStencilFailBackFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_BACK_PASS_DEPTH_FAIL, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_BACK_PASS_DEPTH_FAIL, ...) failed.");
    cache->stencilOperationDepthFailBackFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_BACK_PASS_DEPTH_PASS, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_BACK_PASS_DEPTH_PASS, ...) failed.");
    cache->stencilOperationDepthPassBackFace = iValues;
    return true;
}

static bool ResyncStencilWriteMask(RenderStateCache* cache)
{
    GLint iValues = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_WRITEMASK, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_WRITEMASK, ...) failed.");
    cache->stencilWriteMaskFrontFace = iValues;

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_STENCIL_BACK_WRITEMASK, &iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_STENCIL_BACK_WRITEMASK, ...) failed.");
    cache->stencilWriteMaskBackFace = iValues;
    return true;
}

static bool ResyncCullFace(RenderStateCache* cache)
{
    GLboolean isCullingEnabled;
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(isCullingEnabled, glIsEnabled, (GL_CULL_FACE));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_CULL_FACE) failed.");
    if (isCullingEnabled != 0) {
        GLint cullMode = 0;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_CULL_FACE_MODE, &cullMode));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_CULL_FACE_MODE, &cullMode) failed.");
        if (cullMode == GL_FRONT) {
            cache->culling = RenderMode::FrontFaceCulling;
        }
        else if (cullMode == GL_BACK) {
            cache->culling = RenderMode::BackFaceCulling;
        }
        else {
            //Note: GL_FRONT_AND_BACK is deliberately no RenderMode enum in Candera.
            cache->culling = RenderMode::NoCulling;
        }
    }
    else {
        cache->culling = RenderMode::NoCulling;
    }
    return true;
}

static bool ResyncDitheringEnabled(RenderStateCache* cache)
{
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(cache->isDitheringEnabled, glIsEnabled, (GL_DITHER));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_DITHER) failed.");
    return true;
}

static bool ResyncWindingOrder(RenderStateCache* cache)
{
    GLint value = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_FRONT_FACE, &value));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_FRONT_FACE, &value) failed.");
    //Map to RenderMode::Winding enumerator values.
    cache->frontFace = static_cast<RenderMode::Winding>(value - GlTypeMapper::MapWindingOrder(RenderMode::ClockWise));
    return true;
}

static bool ResyncDepthWriteEnabled(RenderStateCache* cache)
{
    GLboolean value = GL_FALSE;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetBooleanv, (GL_DEPTH_WRITEMASK, &value));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetBooleanv(GL_DEPTH_WRITEMASK, &value) failed.");
    cache->isDepthWriteEnabled = value;
    return true;
}

static bool ResyncDepthTestEnabled(RenderStateCache* cache)
{
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(cache->isDepthTestEnabled, glIsEnabled, (GL_DEPTH_TEST));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_DEPTH_TEST) failed.");
    return true;
}

static bool ResyncDepthComparisonFunction(RenderStateCache* cache)
{
    GLint value = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_DEPTH_FUNC, &value));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_DEPTH_FUNC, &value) failed.");
    //Map to RenderMode::ComparisonFunction enumerator values.
    cache->depthComparisonFunction = static_cast<RenderMode::ComparisonFunction>(value - GlTypeMapper::MapComparisonFunction(RenderMode::CompareNever));
    return true;
}

static bool ResyncStencilTestEnabled(RenderStateCache* cache)
{
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(cache->isStencilTestEnabled, glIsEnabled, (GL_STENCIL_TEST));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_STENCIL_TEST) failed.");
    return true;
}

static bool ResyncScissorTestEnabled(RenderStateCache* cache)
{
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(cache->isScissorTestEnabled, glIsEnabled, (GL_SCISSOR_TEST));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_SCISSOR_TEST) failed.");
    return true;
}

static bool ResyncBlendingEnabled(RenderStateCache* cache)
{
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(cache->isBlendingEnabled, glIsEnabled, (GL_BLEND));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_BLEND) failed.");
    return true;
}

static bool ResyncSampleCoverageMask(RenderStateCache* cache)
{
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(cache->isSampleCoverageMaskEnabled, glIsEnabled, (GL_SAMPLE_COVERAGE));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_SAMPLE_COVERAGE) failed.");
    return true;
}

static bool ResyncSampleAlphaToCoverageMaskEnabled(RenderStateCache* cache)
{
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(cache->isSampleAlphaToCoverageMaskEnabled, glIsEnabled, (GL_SAMPLE_ALPHA_TO_COVERAGE));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled(GL_SAMPLE_ALPHA_TO_COVERAGE) failed.");
    return true;
}

static bool ResyncRasterizerDiscardEnabled(RenderStateCache* cache)
{
    FEATSTD_UNUSED(cache);
    return true;
}

static bool ResyncDepthBias(RenderStateCache* cache)
{
    GLboolean isEnabled;
    GLfloat floatValue = 0.0F;
    {
        CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(isEnabled, glIsEnabled, (GL_POLYGON_OFFSET_FILL));
    }
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glIsEnabled failed.");
    if (isEnabled != 0) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetFloatv, (GL_POLYGON_OFFSET_FACTOR, &floatValue));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetFloatv(GL_POLYGON_OFFSET_FACTOR, ...) failed.");

        cache->depthBiasScaleFactor = floatValue;

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetFloatv, (GL_POLYGON_OFFSET_UNITS, &floatValue));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetFloatv(GL_POLYGON_OFFSET_UNITS, ...) failed.");

        cache->depthBiasUnits = floatValue;
    }
    else {
        cache->depthBiasScaleFactor = 0.0F;
        cache->depthBiasUnits = 0.0F;
    }
    return true;
}

static bool ResyncViewport(RenderStateCache* cache)
{
    GLint iValues[4] = {0, 0, 0, 0};
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_VIEWPORT, iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_VIEWPORT, ...) failed.");
    cache->viewportX = iValues[0];
    cache->viewportY = iValues[1];
    cache->viewportWidth = static_cast<GLsizei>(iValues[2]);
    cache->viewportHeight = static_cast<GLsizei>(iValues[3]);
    return true;
}

static bool ResyncScissor(RenderStateCache* cache)
{
    GLint iValues[4] = { 0, 0, 0, 0 };
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_SCISSOR_BOX, iValues));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_SCISSOR_BOX, ...) failed.");
    cache->scissorX = iValues[0];
    cache->scissorY = iValues[1];
    cache->scissorWidth = static_cast<GLsizei>(iValues[2]);
    cache->scissorHeight = static_cast<GLsizei>(iValues[3]);
    return true;
}

static bool ResyncShaderHandle(RenderStateCache* cache)
{
    GLint shaderHandle = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_CURRENT_PROGRAM, &shaderHandle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_CURRENT_PROGRAM, ...) failed.");
    cache->activeShaderHandle = static_cast<Handle>(shaderHandle);
    return true;
}

static bool ResyncTextureImageHandles(RenderStateCache* cache)
{
    GLint activeTextureUnit = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_ACTIVE_TEXTURE, &activeTextureUnit));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_ACTIVE_TEXTURE, ...) failed.");
    cache->activeTextureUnit = static_cast<UInt32>(activeTextureUnit - GL_TEXTURE0);

    for (UInt32 unit = 0; unit < CANDERA_MAX_TEXTURE_UNIT_COUNT; ++unit) {
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glActiveTexture, (GlTypeMapper::MapTextureUnit(unit)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glActiveTexture failed.");

        GLint textureHandle = 0;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_TEXTURE_BINDING_2D, &textureHandle));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_TEXTURE_BINDING_2D,...) failed.");
        cache->activeTextureHandle[unit][TextureImage::Texture2D] = textureHandle;

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_TEXTURE_BINDING_CUBE_MAP, &textureHandle));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_TEXTURE_BINDING_CUBE_MAP,...) failed.");
        cache->activeTextureHandle[unit][TextureImage::TextureCubeMap] = textureHandle;
    }

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glActiveTexture, (activeTextureUnit));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glActiveTexture failed.");

    return true;
}

static bool ResyncVertexBufferHandles(RenderStateCache* cache)
{
    GLint handle = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_ARRAY_BUFFER_BINDING, &handle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_ARRAY_BUFFER_BINDING, ...) failed.");
    cache->activeVertexBufferHandle = static_cast<Handle>(handle);

    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_ELEMENT_ARRAY_BUFFER_BINDING, &handle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_ELEMENT_ARRAY_BUFFER_BINDING, ...) failed.");
    cache->activeIndexBufferHandle = static_cast<Handle>(handle);

    return true;
}

static bool ResyncVertexAttribArray(RenderStateCache* cache)
{
    GLint maxVertexAttribs = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetIntegerv, (GL_MAX_VERTEX_ATTRIBS, &maxVertexAttribs));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_MAX_VERTEX_ATTRIBS, ...) failed.");

    GLint cachedVertexAttribsCount = sizeof(cache->enabledVertexAttribs) * 8;
    cachedVertexAttribsCount = maxVertexAttribs > cachedVertexAttribsCount ? cachedVertexAttribsCount : maxVertexAttribs;

    UInt32 enabledVertexAttribs = 0;

    for (GLint i = cachedVertexAttribsCount - 1; i >= 0; --i) {
        GLint enabledVertexAttrib = 0;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGetVertexAttribiv, (i, GL_VERTEX_ATTRIB_ARRAY_ENABLED, &enabledVertexAttrib));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGetIntegerv(GL_MAX_VERTEX_ATTRIBS, ...) failed.");
        enabledVertexAttribs <<= 1;
        enabledVertexAttribs |= ((enabledVertexAttrib == GL_FALSE) ? 0U : 1U);
    }

    FEATSTD_COMPILETIME_ASSERT(sizeof(enabledVertexAttribs) == sizeof(cache->enabledVertexAttribs));
    cache->enabledVertexAttribs = enabledVertexAttribs;

    return true;
}

void RenderDevice::ResyncCurrentRenderStateCache(RenderStateCacheResyncOption resetOption)
{
    typedef bool(*ResyncFunctionType)(RenderStateCache* object);
    static ResyncFunctionType resyncFunction[] = {
        &ResyncClearColor,
        &ResyncClearDepth,
        &ResyncClearStencil,
        &ResyncColorWrite,
        &ResyncBlendColor,
        &ResyncBlendMode,
        &ResyncSampleCoverage,
        &ResyncStencilFunction,
        &ResyncStencilOperation,
        &ResyncStencilWriteMask,
        &ResyncCullFace,
        &ResyncDitheringEnabled,
        &ResyncWindingOrder,
        &ResyncDepthWriteEnabled,
        &ResyncDepthTestEnabled,
        &ResyncDepthComparisonFunction,
        &ResyncStencilTestEnabled,
        &ResyncScissorTestEnabled,
        &ResyncBlendingEnabled,
        &ResyncSampleCoverageMask,
        &ResyncSampleAlphaToCoverageMaskEnabled,
        &ResyncRasterizerDiscardEnabled,
        &ResyncDepthBias,
        &ResyncViewport,
        &ResyncScissor,
        &ResyncShaderHandle,
        &ResyncTextureImageHandles,
        &ResyncVertexBufferHandles,
        &ResyncVertexAttribArray
    };

    FEATSTD_COMPILETIME_ASSERT((sizeof(resyncFunction) / sizeof(ResyncFunctionType)) == ResyncAll);

    RenderStateCache* cache = GetCurrentRenderStateCache();
    if (cache == 0) {
        return;
    }
    bool resyncResult = true;
    if (resetOption == ResyncAll) {
        for (Int resyncOptionIndex = 0; resyncOptionIndex < static_cast<Int>(ResyncAll); ++resyncOptionIndex) {
            resyncResult = resyncResult && resyncFunction[resyncOptionIndex](cache);
        }
    }
    else {
        resyncResult = resyncFunction[resetOption](cache);
    }
    if (!resyncResult) {
        FEATSTD_LOG_WARN("resync failed!");
    }
}
CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(553, "Check if extension is enabled.")
#if EGL_KHR_fence_sync
FEATSTD_LINT_NEXT_EXPRESSION(1924, "C-style cast in macro EGL_NO_SYNC_KHR")
static const SynchronizationFence::Handle c_InvalidFenceHandle = PointerToPointer<SynchronizationFence::Handle>(EGL_NO_SYNC_KHR);
FEATSTD_LINT_NEXT_EXPRESSION(1924, "Lower case literal suffix l in macro EGL_FOREVER_KHR")
FEATSTD_LINT_NEXT_EXPRESSION(1960, "Lower case literal suffix l in macro EGL_FOREVER_KHR")
const UInt64 RenderDevice::c_IgnoreWaitTimeOut = static_cast<UInt64>(EGL_FOREVER_KHR);
#else
const UInt64 RenderDevice::c_IgnoreWaitTimeOut = ~0ULL;
#endif

bool RenderDevice::CreateSynchronizationFence(SynchronizationFence::Handle& fenceHandle)
{
#if EGL_KHR_fence_sync
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(fenceHandle, eglCreateSyncKHR, (EGLWrapper::GetInstance().GetCurrentDisplay(), EGL_SYNC_FENCE_KHR, 0));
    if (fenceHandle == c_InvalidFenceHandle) {
        CANDERA_EGL_CHECK_ERROR_NO_RETURN("eglCreateSyncKHR failed!");
    }
    return (fenceHandle != c_InvalidFenceHandle);
#else
    FEATSTD_UNUSED(fenceHandle);
    FEATSTD_LOG_WARN("EGL_KHR_fence_sync extension is disabled. Check if it is supported and enable it for fence synchronization support!");
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(715, "fenceHandle referenced when EGL_KHR_fence_sync is defined.")
    return false;
#endif
}

SynchronizationFence::WaitResult RenderDevice::ClientWaitSynchronizationFence(SynchronizationFence::Handle fenceHandle, UInt64 timeout, bool flush)
{
    SynchronizationFence::WaitResult result = SynchronizationFence::Fail;
#if EGL_KHR_fence_sync
    if (fenceHandle != c_InvalidFenceHandle) {
        EGLint eglResult;
        CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(eglResult, eglClientWaitSyncKHR, (EGLWrapper::GetInstance().GetCurrentDisplay(), fenceHandle, flush ? (EGL_SYNC_FLUSH_COMMANDS_BIT_KHR) : 0, timeout));
        CANDERA_EGL_CHECK_ERROR_NO_RETURN("eglClientWaitSyncKHR failed!");
        switch (eglResult) {
            case EGL_TIMEOUT_EXPIRED_KHR: result = SynchronizationFence::TimeoutExpired; break;
            case EGL_CONDITION_SATISFIED_KHR: result = SynchronizationFence::Processed; break;
            default: result = SynchronizationFence::Fail;
        }
    }
#else
    FEATSTD_UNUSED(fenceHandle);
    FEATSTD_UNUSED(timeout);
    FEATSTD_UNUSED(flush);
#endif
    return result;
}

bool RenderDevice::IsSynchronizationFenceSignaled(SynchronizationFence::Handle fenceHandle)
{
    bool result = false;
#if EGL_KHR_fence_sync
    EGLint value = EGL_UNSIGNALED_KHR;
    EGLBoolean eglResult;
    CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(eglResult, eglGetSyncAttribKHR, (EGLWrapper::GetInstance().GetCurrentDisplay(), fenceHandle, EGL_SYNC_STATUS_KHR, &value));
    CANDERA_EGL_CHECK_ERROR_NO_RETURN("eglGetSyncAttribKHR failed!");
    FEATSTD_LINT_NEXT_EXPRESSION(838, "Violates MISRA C++ 2008 Required Rule 0-1-9: previously assigned value")
    result = (eglResult == EGL_TRUE) && (value == EGL_SIGNALED_KHR);
#else
    FEATSTD_UNUSED(fenceHandle);
#endif
    return result;
}

void RenderDevice::DestroySynchronizationFence(SynchronizationFence::Handle fenceHandle)
{
#if EGL_KHR_fence_sync
    if (fenceHandle != c_InvalidFenceHandle) {
        EGLBoolean eglResult;
        CANDERA_RENDERDEVICE_OPENGL_RETURN_FUNCTION_CALL(eglResult, eglDestroySyncKHR, (EGLWrapper::GetInstance().GetCurrentDisplay(), fenceHandle));
        if (eglResult == EGL_TRUE) {
            CANDERA_EGL_CHECK_ERROR_NO_RETURN("eglDestroySyncKHR failed!");
        }
    }
#else
    FEATSTD_UNUSED(fenceHandle);
#endif
}

// Direct Texture.

bool RenderDevice::UploadDirectTextureImage(DirectTextureImage& textureImage, UInt unit)
{
    Handle handle = 0;
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenTextures, (1, FeatStd::Internal::PointerToPointer<GLuint*>(&handle)));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGenTextures failed.");

    textureImage.SetVideoMemoryHandle(handle);
    bool rc = ActivateTextureImage(textureImage, unit);
    if (!rc) {
        return rc;
    }

#ifdef GL_VIV_direct_texture

    void* logicalAddress[4];
    UInt physicalAddress[4];
    textureImage.GetLogicalAddress(logicalAddress);
    textureImage.GetPhysicalAddress(physicalAddress);

    if ((logicalAddress[0] == 0) && (physicalAddress[0] == ~static_cast<UInt>(0U))) {
        // No buffer specified, should be allocated by the driver

        GLvoid* pixels[4] = { 0 };
        // This is where the real upload happens.
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(
            glTexDirectVIV, (GL_TEXTURE_2D,
            static_cast<GLsizei>(textureImage.GetWidth()),
            static_cast<GLsizei>(textureImage.GetHeight()),
            GlTypeMapper::MapDirectTextureFormat(textureImage.GetFormat()),
            pixels));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexDirectVIV failed.");

        // Since memory is already mapped to CPU virtual space, return the
        // address of the underlying buffers.
        void* address[4];

        address[0] = /*MapMemory*/(pixels[0]);
        address[1] = /*MapMemory*/(pixels[1]);
        address[2] = /*MapMemory*/(pixels[2]);
        address[3] = /*MapMemory*/(pixels[3]);
        textureImage.SetVideoMemoryAddress(address);
    }
    else {

        // Buffer has already been allocated, user has provided either a logical or a physical address.
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(
            glTexDirectVIVMap, (GL_TEXTURE_2D,
            static_cast<GLsizei>(textureImage.GetWidth()),
            static_cast<GLsizei>(textureImage.GetHeight()),
            GlTypeMapper::MapDirectTextureFormat(textureImage.GetFormat()),
            static_cast<GLvoid**>(logicalAddress),
            static_cast<GLuint*>(physicalAddress)) );
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexDirectVIVMap failed.");

        // Video memory address is the same as logical address
        textureImage.SetVideoMemoryAddress(logicalAddress);

        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(
            glTexDirectInvalidateVIV, (GL_TEXTURE_2D));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexDirectInvalidateVIV failed.");
    }
#endif

    SetTextureStateCache(textureImage, CreateTextureStateCache());
    Diagnostics::VideoMemoryStatistic::OnTextureImageUploaded(textureImage);

    return true;
}

bool RenderDevice::UnloadDirectTextureImage(DirectTextureImage& textureImage)
{
    SetTextureStateCache(textureImage, 0);

#ifdef GL_VIV_direct_texture
    // Reset the video memory addresses to 0.
    // Issue CGI1-20319: Do not reset logical and physical address, they can remain valid after unload since are managed by client
    void* address[4] = { 0, 0, 0, 0 };
    textureImage.SetVideoMemoryAddress(address);
#endif

    GLuint textureHandle = static_cast<GLuint>(textureImage.GetVideoMemoryHandle());

    if (GetCurrentRenderStateCache() != 0) {
        GetCurrentRenderStateCache()->ResetActiveTextureHandles(textureHandle);
    }
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteTextures, (1, &textureHandle));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteTextures failed.");

    Diagnostics::VideoMemoryStatistic::OnTextureImageUnloaded(textureImage);
    // Reset video memory handle to 0.
    textureImage.SetVideoMemoryHandle(0);
    return true;
}

bool RenderDevice::LockDirectTextureImage(DirectTextureImage& textureImage)
{
    FEATSTD_UNUSED(textureImage);
#ifdef GL_VIV_direct_texture
    // With the VIV_direct_texture extension the texture memory is mapped on
    // upload. Lock doesn't need to change anything.
#endif
    return true;
}

bool RenderDevice::UnlockDirectTextureImage(DirectTextureImage& textureImage, UInt unit)
{
    bool rc = ActivateTextureImage(textureImage, unit);
    if (!rc) {
        return rc;
    }

#ifdef GL_VIV_direct_texture
    CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(
        glTexDirectInvalidateVIV, (GL_TEXTURE_2D));
    CANDERA_GL_CHECK_ERROR_LOG_ERROR("glTexDirectInvalidateVIV failed.");
#endif

    if (textureImage.IsMipMappingEnabled()) {
        // MipMap is requested without predefined bitmap chain. Thus, auto-generate MipMap chain by driver.
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenerateMipmap, (GL_TEXTURE_2D));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGenerateMipmap failed.");
    }

    return true;
}

// External Texture.

bool RenderDevice::UploadExternalTextureImage(ExternalTextureImage & textureImage, UInt unit)
{
    EglKhrExternalTextureImage * image = Dynamic_Cast<EglKhrExternalTextureImage*>(&textureImage);
    if (0 != image){
        if (0 == image->GetWidth() && 0 == image->GetHeight()){
            FEATSTD_LOG_ERROR("No width or height set for EglKhrExternalTextureImage.");
            return false;
        }
        Handle handle = 0;
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glGenTextures, (1, FeatStd::Internal::PointerToPointer<GLuint*>(&handle)));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glGenTextures failed.");
        image->SetVideoMemoryHandle(handle);

        bool rc = ActivateTextureImage(textureImage, unit);
        if (!rc) {
            return rc;
        }

        CANDERA_RENDERDEVICE_CLEAR_EGL_ERRORS();
        EGLImageKHR img = Candera::EglExtensions::GetInstance().EglCreateImageKhr(
            Candera::Internal::EGLWrapper::GetInstance().GetCurrentDisplay(), //current display. on Genivi this doesn't matter as we have only one connection. Has to be considered for other platforms.
            EGL_NO_CONTEXT,
            image->GetExternalBufferType(), // This can be set from application code if needed.
            static_cast<EGLClientBuffer>(image->GetExternalBuffer()), // actual buffer.
            image->GetAttributeList()
            );
        if (img != 0)
        {
            //map eglImage to active texture.
            //clear error
            CANDERA_RENDERDEVICE_CLEAR_GL_ERRORS();

            Candera::EglExtensions::GetInstance().GlEglImageTargetTexture2Does(GlTypeMapper::MapTextureTargetType(image->GetTextureTargetType()), img);
            err = glGetError();
            if (err != GL_NO_ERROR){
                Candera::EglExtensions::GetInstance().EglDestroyImageKhr(Candera::Internal::EGLWrapper::GetInstance().GetCurrentDisplay(), image->GetEglImage());
                glDeleteTextures(1, FeatStd::Internal::PointerToPointer<GLuint*>(&handle));
                image->SetVideoMemoryHandle(0);
                image->SetEglImage(EGL_NO_IMAGE_KHR);
                FEATSTD_LOG_ERROR("GlEglImageTargetTexture2Does failed with error %d.", err);
                return false;
            }

            image->SetEglImage(img);
        }
        else { //cleanup
            eglerr = eglGetError();
            FEATSTD_LOG_ERROR("EglCreateImageKhr failed with error %d.", eglerr);
            glDeleteTextures(1, FeatStd::Internal::PointerToPointer<GLuint*>(&handle));
            return false;
        }

        SetTextureStateCache(textureImage, CreateTextureStateCache());
        Diagnostics::VideoMemoryStatistic::OnTextureImageUploaded(textureImage);

        return true;
    }

    return false;
}

bool RenderDevice::UnloadExternalTextureImage(ExternalTextureImage& textureImage)
{
    EglKhrExternalTextureImage * image = Dynamic_Cast<EglKhrExternalTextureImage*>(&textureImage);
    if (0 != image){
        SetTextureStateCache(textureImage, 0);

        Candera::EglExtensions::GetInstance().EglDestroyImageKhr(Candera::Internal::EGLWrapper::GetInstance().GetCurrentDisplay(), image->GetEglImage());
        image->SetEglImage(EGL_NO_IMAGE_KHR);

        GLuint textureHandle = static_cast<GLuint>(textureImage.GetVideoMemoryHandle());

        if (GetCurrentRenderStateCache() != 0) {
            GetCurrentRenderStateCache()->ResetActiveTextureHandles(textureHandle);
        }
        CANDERA_RENDERDEVICE_OPENGL_FUNCTION_CALL(glDeleteTextures, (1, &textureHandle));
        CANDERA_GL_CHECK_ERROR_LOG_ERROR("glDeleteTextures failed.");

        Diagnostics::VideoMemoryStatistic::OnTextureImageUnloaded(textureImage);
        // Reset video memory handle to 0.
        image->SetVideoMemoryHandle(0);
        return true;
    }
    return false;
}

} // namespace Candera
