//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlTypeMapper.h"
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    using namespace Diagnostics;
    FEATSTD_LOG_SET_REALM(LogRealm::CanderaPlatformDevice);

struct UniformTypeMappingPair {
    GLenum glType;
    Shader::UniformType canderaType;
};

static const UniformTypeMappingPair uniformTypeMappingTable[] =
{
    { GL_FLOAT, Shader::Float },
    { GL_FLOAT_VEC2, Shader::FloatVec2 },
    { GL_FLOAT_VEC3, Shader::FloatVec3 },
    { GL_FLOAT_VEC4, Shader::FloatVec4 },

    { GL_INT, Shader::Integer },
    { GL_INT_VEC2, Shader::IntegerVec2 },
    { GL_INT_VEC3, Shader::IntegerVec3 },
    { GL_INT_VEC4, Shader::IntegerVec4 },

    { GL_BOOL, Shader::Bool },
    { GL_BOOL_VEC2, Shader::BoolVec2 },
    { GL_BOOL_VEC3, Shader::BoolVec3 },
    { GL_BOOL_VEC4, Shader::BoolVec4 },

    { GL_FLOAT_MAT2, Shader::FloatMat2 },
    { GL_FLOAT_MAT3, Shader::FloatMat3 },
    { GL_FLOAT_MAT4, Shader::FloatMat4 },

    { GL_SAMPLER_2D, Shader::Sampler2D },
    { GL_SAMPLER_CUBE, Shader::SamplerCube }
};

Shader::UniformType GlTypeMapper::MapGlUniformType(GLenum glType)
{
    static const Int32 tableEntryCount = sizeof(uniformTypeMappingTable)/sizeof(UniformTypeMappingPair);
    for (Int32 i = 0; i < tableEntryCount; ++i) {
        if (uniformTypeMappingTable[i].glType == glType) {
            return uniformTypeMappingTable[i].canderaType;
        }
    }

    FEATSTD_LOG_ERROR("RenderDevice - GlTypeMapper: unexpected uniform type");

    return Shader::Float;
}

#ifndef GL_EXT_texture_compression_dxt1
#ifndef GL_COMPRESSED_RGB_S3TC_DXT1_EXT
static const GLenum GL_COMPRESSED_RGB_S3TC_DXT1_EXT = GL_INVALID_ENUM;
#endif
#ifndef GL_COMPRESSED_RGBA_S3TC_DXT1_EXT
static const GLenum GL_COMPRESSED_RGBA_S3TC_DXT1_EXT = GL_INVALID_ENUM;
#endif
#endif

#ifndef GL_EXT_texture_compression_s3tc
#ifndef GL_COMPRESSED_RGBA_S3TC_DXT3_EXT
static const GLenum GL_COMPRESSED_RGBA_S3TC_DXT3_EXT = GL_INVALID_ENUM;
#endif
#ifndef GL_COMPRESSED_RGBA_S3TC_DXT5_EXT
static const GLenum GL_COMPRESSED_RGBA_S3TC_DXT5_EXT = GL_INVALID_ENUM;
#endif
#endif

const GlTypeMapper::PixelFormatProperties GlTypeMapper::c_PixelFormatProperties[] =
{
    // format                               type                                    bits per pixel
    { GL_RGB,                               GL_UNSIGNED_SHORT_5_6_5,                16 }, // RgbUnsignedShort565PixelFormat
    { GL_RGBA,                              GL_UNSIGNED_SHORT_4_4_4_4,              16 }, // RgbaUnsignedShort4444PixelFormat
    { GL_RGBA,                              GL_UNSIGNED_SHORT_5_5_5_1,              16 }, // RgbaUnsignedShort5551PixelFormat
    { GL_DEPTH_COMPONENT,                   GL_UNSIGNED_SHORT,                      16 }, // DepthUnsignedShortPixelFormat
    { GL_RGB,                               GL_UNSIGNED_BYTE,                       32 }, // RgbUnsignedBytePixelFormat
    { GL_RGBA,                              GL_UNSIGNED_BYTE,                       32 }, // RgbaUnsignedBytePixelFormat
    { GL_ALPHA,                             GL_UNSIGNED_BYTE,                       32 }, // AlphaUnsignedBytePixelFormat
    { GL_LUMINANCE,                         GL_UNSIGNED_BYTE,                       32 }, // LuminanceUnsignedBytePixelFormat
    { GL_LUMINANCE_ALPHA,                   GL_UNSIGNED_BYTE,                       32 }, // LuminanceAlphaUnsignedBytePixelFormat
    { GL_COVERAGE_COMPONENT_NV,             GL_UNSIGNED_BYTE,                       32 }, // CoverageUnsignedBytePixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // Etc2CompressedRgbPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // Etc2CompressedSrgbPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // Etc2Alpha1CompressedRgbaPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // Etc2Alpha1CompressedSrgbaPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // Etc2EacCompressedRgbaPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // Etc2EacCompressedSrgbaPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // EacCompressedUnsignedRPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // EacCompressedSignedRPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // EacCompressedUnsignedRGPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // EacCompressedSignedRGPixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // SrgbUnsignedBytePixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // SrgbaUnsignedBytePixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // RgbFloatingPointR11G11B10PixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // RgbFloatingPoint9E5PixelFormat
    { GL_INVALID_ENUM,                      GL_INVALID_ENUM,                         0 }, // RgbaUnsignedIntRev10A2PixelFormat
    { GL_COMPRESSED_RGB_S3TC_DXT1_EXT,      GL_COMPRESSED_RGB_S3TC_DXT1_EXT,         4 }, // Dxt1CompressedRgbFormat 
    { GL_COMPRESSED_RGBA_S3TC_DXT1_EXT,     GL_COMPRESSED_RGBA_S3TC_DXT1_EXT,        4 }, // Dxt1CompressedRgbaFormat
    { GL_COMPRESSED_RGBA_S3TC_DXT3_EXT,     GL_COMPRESSED_RGBA_S3TC_DXT3_EXT,        8 }, // Dxt3CompressedRgbaFormat
    { GL_COMPRESSED_RGBA_S3TC_DXT5_EXT,     GL_COMPRESSED_RGBA_S3TC_DXT5_EXT,        8 }  // Dxt5CompressedRgbaFormat
};

RenderMode::BlendFactor GlTypeMapper::MapGlBlendFactor(GLenum glFactor)
{
    static const UInt8 factorCount = sizeof(BlendFactorMapping)/sizeof(GLenum);
    for (UInt8 i = 0; i < factorCount; ++i)
    {
        if (BlendFactorMapping[i] == glFactor) {
            return static_cast<RenderMode::BlendFactor>(i);
        }
    }

    FEATSTD_LOG_ERROR("RenderDevice - GlTypeMapper: unexpected blend factor");

    return RenderMode::Zero;
}

RenderMode::BlendOperation GlTypeMapper::MapGlBlendOperation(GLenum glOperation)
{
    static const UInt8 opCount = sizeof(BlendOperationMapping)/sizeof(GLenum);
    for (UInt8 i = 0; i < opCount; ++i)
    {
        if (BlendOperationMapping[i] == glOperation) {
            return static_cast<RenderMode::BlendOperation>(i);
        }
    }

    FEATSTD_LOG_ERROR("RenderDevice - GlTypeMapper: unexpected blend operation");

    return RenderMode::Add;
}

} // namespace Candera

