//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#include "WarpingProperties.h"
#include <Candera/System/Diagnostics/Log.h>

namespace Candera
{

    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaPlatformDevice);

void WarpingProperties::SetWarpMatrix(WarpMatrix& warpMatrix, UInt index)
{
    if (index >= c_maxWarpMatrixCount) {
        FEATSTD_LOG_ERROR("Failed to set warp matrix, index out of range.");
        return;
    }

    m_warpMatrix[index].AcquireData(warpMatrix);
    SetWarpMatrixDirtyFlag(true);
}

void WarpingProperties::ResetWarpMatrix(UInt index)
{
    if (index >= c_maxWarpMatrixCount) {
        FEATSTD_LOG_ERROR("Failed to reset warp matrix, index out of range.");
        return;
    }
    m_warpMatrix[index].ClearData();
    m_warpMatrixWeight[index] = 0.0F;

    SetWarpMatrixDirtyFlag(true);
}


void WarpingProperties::SetWarpingEnabled(bool isEnabled) 
{ 
    m_isWarpingEnabled = isEnabled;
}

WarpingProperties::WarpingProperties() :
    m_isWarpingEnabled(false),
    m_isWarpMatrixDirty(true),
    m_defaultMatrix(),
    m_warpMatrix(),
    m_warpImageBounds(0.0F, 0.0F, 1.0F, 1.0F),
    m_displayOrientation(DisplayOrientation::Unchanged)
{
    m_warpMatrixWeight[0] = 1.0F;
    m_actualWarpMatrixWeight[0] = 0.0F;
    for (Int i = 0; i < c_maxWarpMatrixCount; ++i) {
        m_warpMatrixWeight[i] = 0.0F;
        m_actualWarpMatrixWeight[i] = 0.0F;
    }
}


void WarpingProperties::SetWarpMatrixWeight(UInt index, Float weight)
{
    if (index >= c_maxWarpMatrixCount) {
        FEATSTD_LOG_ERROR("Failed to set warp matrix weight, index out of range.");
        return;
    }
    m_warpMatrixWeight[index] = weight;
}

Float WarpingProperties::GetWarpMatrixWeight(UInt index) const
{
    if (index >= c_maxWarpMatrixCount) {
        FEATSTD_LOG_ERROR("Failed to retrieve warp matrix weight, index out of range.");
        return 0.0F;
    }

    return m_warpMatrixWeight[index];
}

const WarpMatrix& WarpingProperties::GetWarpMatrix(UInt index) const
{
    if (index >= c_maxWarpMatrixCount) {
        FEATSTD_LOG_ERROR("Failed to retrieve warp matrix, index out of range.");
        return m_defaultMatrix;
    }

    return m_warpMatrix[index];
}

WarpMatrix& WarpingProperties::GetWarpMatrix(UInt index)
{

    if (index >= c_maxWarpMatrixCount) {
        FEATSTD_LOG_ERROR("Failed to retrieve warp matrix, index out of range.");
        return m_defaultMatrix;
    }

    return m_warpMatrix[index];
}

bool WarpingProperties::IsWarpingMatrixValid() const
{
    
    bool isValid = false;

    // at least one warping matrix should be valid 
    for (Int i = 0; i < c_maxWarpMatrixCount; ++i) {
        if (0 != m_warpMatrix[i].GetGenericData()) {
            isValid = true;
        }
    }
    return isValid;
}

Int WarpingProperties::GetActualWarpMatrixCount() const
{
    Int count = 0;
    for (Int i = 0; i < c_maxWarpMatrixCount; ++i) {
        if (0 != m_warpMatrix[i].GetGenericData()) {
            ++count;
        }
    }
    return count;
}

const Float* WarpingProperties::GetActualWarpMatrixWeight()
{

    Int warpMatrixCount = GetActualWarpMatrixCount();
    Int count = 0;
    
    for (Int i = 0; i < c_maxWarpMatrixCount; i++) {
        m_actualWarpMatrixWeight[i] = 0.0F;
        if ((0 != m_warpMatrix[i].GetGenericData()) && (count < warpMatrixCount)) {
            m_actualWarpMatrixWeight[count] = m_warpMatrixWeight[i];
            count++;
        }
    }

    return &m_actualWarpMatrixWeight[0];
}
}   // namespace Candera
