//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderDevice2DOver3D.h"
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Base/ImageSource3D.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapConvertor.h>

#include <Candera/Engine3D/Core/BitmapTextureImage.h>
#include <Candera/Engine3D/Core/ProxyTextureImage.h>
#include <Candera/Engine2D/Core/BitmapImage2D.h>
#include <Candera/Engine2D/Core/VertexBuffer2D.h>
#include <Candera/Engine2D/Core/VertexGeometry2D.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/System/Monitor/GlobalExperimentPublicIF.h>

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3D.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/SurfaceAllocator2DOver3D.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/GeometryAllocator2DOver3D.h>

#include <CanderaPlatform/Device/Common/BitmapConverter/GenericLinear2DDirectAccessBitmap.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapPixelConvertor.h>

namespace Candera
{
    using namespace Internal;

static Texture::MinMagFilter CastFilter(RenderDevice2D::Filter filter) {
    static Texture::MinMagFilter  table[] = {
        Texture::MinMagNearest,     //Nearest = 0,
        Texture::MinMagLinear       //Bilinear  = 1,
    };

    if ((filter >= 0 )&& (static_cast<UInt>(filter) < (sizeof(table) / sizeof(table[0])))) {
        return table[filter];
    }

    return Texture::MinMagNearest;
}

static Texture::MipMapFilter CastMipMapFilter(RenderDevice2D::MipMapFilter filter) {
    static Texture::MipMapFilter  table[] = {
        Texture::MipMapNone,        //DisabledMipMapFilter,
        Texture::MipMapNearest,     //NearestMipMapFilter,
        Texture::MipMapLinear       //LinearMipMapFilter,
    };

    if ((filter >= 0 )&& (static_cast<UInt>(filter) < (sizeof(table) / sizeof(table[0])))) {
        return table[filter];
    }

    return Texture::MipMapNone;
}

static RenderMode::BlendFactor CastFactor(RenderDevice2D::BlendFactor factor) {
    static RenderMode::BlendFactor table[] = {
        RenderMode::Zero,                        //Zero = 0,
        RenderMode::One,                         //One  = 1,
        RenderMode::SourceColor,                 //SourceColor = 2,
        RenderMode::InverseSourceColor,          //InverseSourceColor = 3,
        RenderMode::SourceAlpha,                 //SourceAlpha = 4,
        RenderMode::InverseSourceAlpha,          //InverseSourceAlpha = 5,
        RenderMode::DestColor,                   //DestColor = 6,ation;
        RenderMode::InverseDestColor,            //InverseDestColor = 7,
        RenderMode::DestAlpha,                   //DestAlpha = 8,
        RenderMode::InverseDestAlpha,            //InverseDestAlpha = 9,
        RenderMode::ConstantColor,               //ConstantColor = 10,
        RenderMode::InverseConstantColor,        //InverseConstantColor = 11,
        RenderMode::ConstantAlpha,               //ConstantAlpha = 12,
        RenderMode::InverseConstantAlpha         //InverseConstantAlpha = 13
    };

    if ((factor >= 0) && (static_cast<UInt>(factor) < (sizeof(table) / sizeof(table[0])))) {
        return table[factor];
    }

    return RenderMode::Zero;
}

static RenderMode::BlendOperation CastOperation(RenderDevice2D::BlendOperation operation) {
    static RenderMode::BlendOperation table[] = {
        RenderMode::Add,                //Add = 0,
        RenderMode::Subtract,           //Subtract = 1,
        RenderMode::ReverseSubtract     //ReverseSubtract = 2
    };

    if ((operation >= 0) && (static_cast<UInt>(operation) < (sizeof(table) / sizeof(table[0])))) {
        return table[operation];
    }

    return RenderMode::Add;
}


bool RenderDevice2D::SetSurface(Handle context, RenderDevice2D::SurfaceType sidx, Handle sname) {
    if (context == 0) {
        return false;
    }

    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        Texture* texture = context2d.GetState().srcTexture.GetPointerToSharedInstance();
        context2d.GetState().srcPackOrder = RenderDevice2D::UnknownPackOrder;
        if (texture != 0) {
            texture->SetTextureImage(SurfaceAllocator2DOver3D::GetTextureImage(sname));
            context2d.GetState().srcPackOrder = GetPackOrder(sname);
        }
        context2d.ChooseProgram(sname);
        return true;
    }
    else {
        if (sidx == RenderDevice2D::MaskSurface) {
            Texture* texture = context2d.GetState().mskTexture.GetPointerToSharedInstance();
            context2d.GetState().mskPackOrder = RenderDevice2D::UnknownPackOrder;
            if (texture != 0) {
                texture->SetTextureImage(SurfaceAllocator2DOver3D::GetTextureImage(sname));
                context2d.GetState().mskPackOrder = GetPackOrder(sname);
            }
            context2d.ChooseProgram(sname);
            return true;
        }
    }
    return false;
}

bool RenderDevice2D::SetTransformationMatrix(Handle context, RenderDevice2D::SurfaceType sidx, const Matrix3x2& mat) {
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        context2d.GetState().srcMatrix = mat;
        return true;
    }
    else {
        if (sidx == RenderDevice2D::MaskSurface) {
            context2d.GetState().mskTrans = mat;
            return true;
        }
    }
    return false;
}

bool RenderDevice2D::SetBlendOperation(Handle context, RenderDevice2D::SurfaceType sidx, RenderDevice2D::BlendOperation op, RenderDevice2D::BlendFactor srcFactor, RenderDevice2D::BlendFactor dstFactor) {
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        RenderMode* renderMode = context2d.GetState().renderMode.GetPointerToSharedInstance();
        if (renderMode != 0) {
            bool blendEffectivelyDisabled =
                (srcFactor == One) &&
                (dstFactor == Zero) &&
                ((op == Add) || (op == Subtract));
            if (blendEffectivelyDisabled) {
                renderMode->SetBlendingEnabled(false);
            }
            else {
                renderMode->SetBlendingEnabled(true);
                renderMode->SetBlendMode(
                    CastFactor(srcFactor), CastFactor(dstFactor), CastOperation(op));
            }
        }

        return true;
    }
    else {
        if (sidx == RenderDevice2D::MaskSurface) {
            // currently not used
            return true;
        }
    }
    return false;
}

bool RenderDevice2D::SetColorBlendOperation(ContextHandle2D context, RenderDevice2D::SurfaceType sidx, RenderDevice2D::BlendOperation op, RenderDevice2D::BlendFactor srcFactor, RenderDevice2D::BlendFactor dstFactor) {
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        RenderMode* renderMode = context2d.GetState().renderMode.GetPointerToSharedInstance();
        if (renderMode != 0) {
            const RenderMode::BlendMode& blendMode = renderMode->GetBlendMode();
            bool blendEffectivelyDisabled =
                (srcFactor == One) &&
                (dstFactor == Zero) &&
                ((op == Add) || (op == Subtract)) &&
                (blendMode.sourceAlphaFactor == RenderMode::One) &&
                (blendMode.destAlphaFactor == RenderMode::Zero) &&
                ((blendMode.operationAlpha == RenderMode::Add) ||
                    (blendMode.operationAlpha == RenderMode::Subtract));
            if (blendEffectivelyDisabled) {
                renderMode->SetBlendingEnabled(false);
            }
            else {
                renderMode->SetBlendingEnabled(true);
            }

            //Blend mode has to be set always.
            //It can happen that the blending enabled flag gets overwritten by SetAlphaBlendOperation.
            //Then color blend settings would be wrong if not set.
            renderMode->SetBlendModeSeparate(
                CastFactor(srcFactor), CastFactor(dstFactor), CastOperation(op),
                blendMode.sourceAlphaFactor, blendMode.destAlphaFactor, blendMode.operationAlpha);


        }

        return true;
    }
    else {
        if (sidx == RenderDevice2D::MaskSurface) {
            // currently not used
            return true;
        }
    }
    return false;
}

bool RenderDevice2D::SetAlphaBlendOperation(ContextHandle2D context, RenderDevice2D::SurfaceType sidx, RenderDevice2D::BlendOperation op, RenderDevice2D::BlendFactor srcFactor, RenderDevice2D::BlendFactor dstFactor) {
    if (context == 0) {
        return false;
    }

    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        RenderMode* renderMode = context2d.GetState().renderMode.GetPointerToSharedInstance();
        if (renderMode != 0) {
            const RenderMode::BlendMode& blendMode = renderMode->GetBlendMode();
            bool blendEffectivelyDisabled =
                (srcFactor == One) &&
                (dstFactor == Zero) &&
                ((op == Add) || (op == Subtract)) &&
                (blendMode.sourceRGBFactor == RenderMode::One) &&
                (blendMode.destRGBFactor == RenderMode::Zero) &&
                ((blendMode.operationRGB == RenderMode::Add) ||
                    (blendMode.operationRGB == RenderMode::Subtract));
            if (blendEffectivelyDisabled) {
                renderMode->SetBlendingEnabled(false);
            }
            else {
                renderMode->SetBlendingEnabled(true);
            }

            //Blend mode has to be set always.
            //It can happen that the blending enabled flag gets overwritten by SetColorBlendOperation.
            //Then alpha blend settings would be wrong if not set.
            renderMode->SetBlendModeSeparate(
                blendMode.sourceRGBFactor, blendMode.destRGBFactor, blendMode.operationRGB,
                CastFactor(srcFactor), CastFactor(dstFactor), CastOperation(op));
        }

        return true;
    }
    else {
        if (sidx == RenderDevice2D::MaskSurface) {
            // currently not used
            return true;
        }
    }
    return false;
}


bool RenderDevice2D::SetSurfaceConstColor(ContextHandle2D context, RenderDevice2D::SurfaceType sidx, Float r, Float g, Float b, Float a) {
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        Color& srcColor = context2d.GetState().srcConstColor;
        srcColor.SetRed(r);
        srcColor.SetGreen(g);
        srcColor.SetBlue(b);
        srcColor.SetAlpha(a);

        return true;
    }
    else {
        if (sidx == RenderDevice2D::MaskSurface) {
            // currently not used
            return true;
        }
    }
    return false;
}

bool RenderDevice2DOver3D::SetGradientColor(ContextHandle2D context, const Color& color)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::UnconditionalScalarToPointer<Context2DOver3D *>(context);
    context2d.GetState().gradientColor = color;
    return true;
}

bool RenderDevice2DOver3D::SetGradientDirection(ContextHandle2D context, const Vector2& direction)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::UnconditionalScalarToPointer<Context2DOver3D *>(context);
    context2d.GetState().gradientDirection = direction;
    static_cast<void>(context2d.GetState().gradientDirection.Normalize());
    return true;
}

bool RenderDevice2DOver3D::SetGradientCenter(ContextHandle2D context, const Vector2& center)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::UnconditionalScalarToPointer<Context2DOver3D *>(context);
    context2d.GetState().gradientCenter = center;
    return true;
}

bool RenderDevice2DOver3D::SetGradientMagnitude(ContextHandle2D context, Float magnitude)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::UnconditionalScalarToPointer<Context2DOver3D *>(context);
    context2d.GetState().gradientMagnitude = magnitude;
    return true;
}

bool RenderDevice2DOver3D::SetFlipMode(ContextHandle2D context, bool flipH, bool flipV)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::UnconditionalScalarToPointer<Context2DOver3D *>(context);
    context2d.GetState().flipH = flipH;
    context2d.GetState().flipV = flipV;
    return true;
}

bool Internal::RenderDevice2DOver3D::SetOutlineParameters(ContextHandle2D context, UInt8 outlineWidth, Color outlineColor, Int colorFactor)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::UnconditionalScalarToPointer<Context2DOver3D *>(context);
    context2d.GetState().outlineColor = outlineColor;
    context2d.GetState().outlineWidth = outlineWidth;
    context2d.GetState().colorFactor = colorFactor;

    return true;
}


bool Internal::RenderDevice2DOver3D::SetDropShadowParameters(ContextHandle2D context, bool enabled, Color shadowColor, Int angle, Int distance, Int scale, Int colorFactor, bool blurFilter)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::UnconditionalScalarToPointer<Context2DOver3D *>(context);
    context2d.GetState().enableShadow = enabled;
    context2d.GetState().shadowColor = shadowColor;
    context2d.GetState().angle = angle;
    context2d.GetState().distance = distance;
    context2d.GetState().scale = scale;
    context2d.GetState().colorFactor = colorFactor;
    context2d.GetState().blurFilter = blurFilter;

    return true;
}


bool RenderDevice2D::SetColorTransformation(ContextHandle2D context, RenderDevice2D::SurfaceType sidx, const Matrix4& matrix) {
    if (context == 0) {
        return false;
    }

    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        static const Matrix4 identity;
        bool enabled = MemoryPlatform::Compare(
            matrix.GetData(),
            identity.GetData(),
            4 * 4 * sizeof(Float)) != 0;
        if (enabled) {
            context2d.GetState().srcColorMatrix = matrix;
            context2d.GetState().srcColorMatrixEnabled = true;
        }
        else {
            context2d.GetState().srcColorMatrixEnabled = false;
        }

        return true;
    }
    return false;
}

bool RenderDevice2D::SetActiveArea(Handle context, RenderDevice2D::SurfaceType sidx, Float left, Float top, Float width, Float height) {
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        context2d.GetState().srcArea.Set(left, top, width, height);
        return true;
    }
    else {
        if (sidx == RenderDevice2D::DestinationSurface) {
            context2d.GetState().dstArea.Set(left, top, width, height);
            return true;
        }
    }
    return false;
}

bool RenderDevice2D::GetActiveArea(ContextHandle2D context, RenderDevice2D::SurfaceType sidx, Float *left, Float *top, Float *width, Float *height){
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (sidx == RenderDevice2D::SourceSurface) {
        Rectangle& srcArea = context2d.GetState().srcArea;
        *left = srcArea.GetLeft();
        *top = srcArea.GetTop();
        *width = srcArea.GetWidth();
        *height = srcArea.GetHeight();

        Float replacementWidth;
        Float replacementHeight;
        TextureImage * srcTextureImage =
            (context2d.GetState().srcTexture != 0) ? context2d.GetState().srcTexture->GetTextureImage().GetPointerToSharedInstance() : 0;

        bool imageAvailable = (srcTextureImage != 0);
        ImageSource3D* imageSource = 0;
        if (imageAvailable)
        {
            imageSource = srcTextureImage->ToImageSource3D();
            imageAvailable = (imageSource != 0);
        }

        if (imageAvailable) {
            replacementWidth = static_cast<Float>(
                imageSource->GetWidth());
            replacementHeight = static_cast<Float>(
                imageSource->GetHeight());
        }
        else{
            replacementWidth = -1.F;
            replacementHeight = -1.F;
        }

        if ((*width < 0.0F) && (replacementWidth > 0.0F)){
            *width = replacementWidth - *left;
        }
        if ((*height < 0.0F) && (replacementHeight > 0.0F)){
            *height = replacementHeight - *top;
        }

        return true;
    }
    else {
        if (sidx == RenderDevice2D::DestinationSurface) {
            Rectangle& dstArea = context2d.GetState().dstArea;
            *left = dstArea.GetLeft();
            *top = dstArea.GetTop();
            *width = dstArea.GetWidth();
            *height = dstArea.GetHeight();

            if (*width < 0.0F){
                *width = context2d.GetTarget().width - *left;
            }
            if (*height < 0.0F){
                *height = context2d.GetTarget().height - *top;
            }

            return true;
        }
    }
    return false;
}

bool RenderDevice2D::SetFilter(ContextHandle2D context, Filter filter)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);

    Texture* texture = context2d.GetState().srcTexture.GetPointerToSharedInstance();
    if (texture != 0) {
        Texture::MinMagFilter texFilter = CastFilter(filter);
        texture->SetMagnificationFilter(texFilter);
        texture->SetMinificationFilter(texFilter);
    }
    return true;
}

bool RenderDevice2D::SetMipMapFilter(ContextHandle2D context, MipMapFilter filter)
{
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);

    Texture* texture = context2d.GetState().srcTexture.GetPointerToSharedInstance();
    if (texture != 0) {
        Texture::MipMapFilter texFilter = CastMipMapFilter(filter);
        texture->SetMipMapFilter(texFilter);
    }
    return true;
}
bool RenderDevice2D::Blit(Handle context) {
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    GLOBAL_EXPERIMENT_COND_IGNORE_DRAW_CALL_2D
    return context2d.Draw();
}

bool RenderDevice2D::Clear(Handle context, Float r, Float g, Float b, Float a) {
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    return context2d.Clear(Color(r, g, b, a));
}

bool RenderDevice2D::GetUpdatedArea(ContextHandle2D context, Float *left, Float *top, Float *width, Float *height) {
    if (context == 0) {
        return false;
    }
    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    Rectangle& lastUpdatedArea = context2d.GetState().lastUpdatedArea;
    *left = lastUpdatedArea.GetLeft();
    *top = lastUpdatedArea.GetTop();
    *width = lastUpdatedArea.GetWidth();
    *height = lastUpdatedArea.GetHeight();

    return true;
}

static MemoryManagement::SharedPointer<BitmapTextureImage> CreateTextureImage(
    const MemoryManagement::SharedPointer<Bitmap>& bitmap,
    bool isMipMappingEnabled)
{
    MemoryManagement::SharedPointer<BitmapTextureImage> texImage = BitmapTextureImage::Create();
    if(texImage == 0) {
        CANDERA_DEVICE_LOG_WARN("Failed to create texture image");
        return MemoryManagement::SharedPointer<BitmapTextureImage>();
    }
    if (!texImage->SetMipMappingEnabled(isMipMappingEnabled)) {
        CANDERA_DEVICE_LOG_WARN("Failed to set mip mapping enabled.");
    }
    if (!texImage->SetBitmap(bitmap)) {
        CANDERA_DEVICE_LOG_WARN("Failed to set bitmap.");
    }
    if (!texImage->Upload()) {
        CANDERA_DEVICE_LOG_WARN("Failed to upload texture image");
        return MemoryManagement::SharedPointer<BitmapTextureImage>();
    }

    return texImage;
}

bool RenderDevice2D::UploadBitmapImage2D(BitmapImage2D& image)
{
    if (image.Get2DSurfaceHandle() != 0) {
        return true;
    }

    if (image.GetBitmap() == 0) {
        CANDERA_DEVICE_LOG_WARN("No bitmap attached to bitmap image 2D.");
        return false;
    }

    const Bitmap* bitmap = image.GetBitmap().GetPointerToSharedInstance();

    SurfaceHandle surfaceHandle = 0;
    if (!RenderDevice2D::CreateSurfaces(1, &surfaceHandle)) {
        CANDERA_DEVICE_LOG_WARN("Faild to create surface handle.");
        return false;
    }

    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo =
        SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(surfaceHandle);
    if (surfaceInfo == 0){
        CANDERA_DEVICE_LOG_WARN("Wrong surface handle.");
        return false;
    }

    MemoryManagement::SharedPointer<BitmapTextureImage> texImage =
        CreateTextureImage(image.GetBitmap(), image.IsMipMappingEnabled());
    if (texImage.PointsToNull()) {
        CANDERA_DEVICE_LOG_WARN("Failed to create texture image.");
        return false;
    }

    surfaceInfo->pixelFormat = bitmap->GetPixelFormat();
    surfaceInfo->packOrder =
        bitmap->IsVerticallyFlipped() ?
            RenderDevice2D::VerticallyFlippedDisplayPackOrder :
            RenderDevice2D::DisplayPackOrder;

    surfaceInfo->textureImage = texImage;

    image.SetSurfaceHandle(surfaceHandle);
    return true;
}

bool RenderDevice2D::UnloadBitmapImage2D(BitmapImage2D& image)
{
    SurfaceHandle surfaceHandle = image.Get2DSurfaceHandle();
    if (surfaceHandle == 0) {
        return true;
    }
    bool success = RenderDevice2D::DestroySurfaces(1, &surfaceHandle);

    if (success) {
        image.SetSurfaceHandle(0);
        return true;
    }
    else {
        return false;
    }
}

bool RenderDevice2D::CreateSurfaces(Int count, SurfaceHandle *pSurface)
{
    return SurfaceAllocator2DOver3D::GetInstance().GetBase().CreateSurfaces(count, pSurface);
}

bool RenderDevice2D::AssignBuffer(SurfaceHandle surface, UInt32 width, UInt32 height,
                                  Int iPixelFormat,
                                  Bitmap::PackAlignment eColorPackAlignment, RenderDevice2D::PackOrder eColorPackOrder,
                                  const void* addressBuffer, DisposerFn disposerFn)
{
    if (!RenderDevice2D::CreateBuffer(surface, width, height, iPixelFormat, eColorPackAlignment, eColorPackOrder, addressBuffer)) {
        return false;
    }

    if (disposerFn != 0) {
        disposerFn(addressBuffer);
    }

    return true;
}


bool RenderDevice2D::DetachBuffer(SurfaceHandle surface)
{
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(surface);
    if (surfaceInfo == 0){
        return false;
    }

    return surfaceInfo->Clear();
}

bool RenderDevice2D::DestroySurfaces(Int count, SurfaceHandle *pSurface)
{
    return SurfaceAllocator2DOver3D::GetInstance().GetBase().DestroySurfaces(count, pSurface);
}


bool RenderDevice2D::CreateBuffer(
    SurfaceHandle surface, UInt32 uiWidth, UInt32 uiHeight,
    Int iPixelFormat,
    Bitmap::PackAlignment eColorPackAlignment, RenderDevice2D::PackOrder eColorPackOrder,
    const void* addressBuffer)
{
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(surface);
    if (surfaceInfo == 0){
        CANDERA_DEVICE_LOG_WARN("Wrong surface handle");
        return false;
    }

    if ((uiWidth > FeatStd::Internal::Limits<UInt16>::cMaxValue) || (uiHeight > FeatStd::Internal::Limits<UInt16>::cMaxValue)) {
        CANDERA_DEVICE_LOG_WARN("Failed to create bitmap. Width/height exceeds limit.");
        return false;
    }

    MemoryManagement::SharedPointer<Bitmap> bitmap = Bitmap::Create(static_cast<UInt16>(uiWidth), static_cast<UInt16>(uiHeight), iPixelFormat,
        eColorPackAlignment, FeatStd::Internal::PointerToPointer<const UInt8*>(addressBuffer), 0, (eColorPackOrder == VerticallyFlippedDisplayPackOrder));
    if(bitmap == 0) {
        CANDERA_DEVICE_LOG_WARN("Failed to recreate bitmap");
        return false;
    }
    MemoryManagement::SharedPointer<BitmapTextureImage> texImage =
        CreateTextureImage(bitmap, false);
    if (texImage.PointsToNull()) {
        CANDERA_DEVICE_LOG_WARN("Failed to create texture image.");
        return false;
    }
    texImage->SetDisposedAfterUpload(true);

    surfaceInfo->pixelFormat = iPixelFormat;
    surfaceInfo->packOrder = eColorPackOrder;

    surfaceInfo->textureImage = texImage;

    return true;
}

bool RenderDevice2D::CreateBuffer(SurfaceHandle surface, UInt32 width, UInt32 height, UInt32 stride,
    TextRendering::GlyphBitmap::Format format, const void* addressBuffer)
{
    Int alignment = 1;
    if ((format != TextRendering::GlyphBitmap::Unknown) && (format != TextRendering::GlyphBitmap::Monochrome)) {
        if (width != stride) {
            for (UInt i = 3; i > 0; i --) {
                UInt32 currentAlignment = static_cast<UInt32>(static_cast<UInt>(1U) << i);
                UInt32 currentAlignmentMask = currentAlignment - 1U;
                if ((stride & currentAlignmentMask) == 0) {
                    alignment = currentAlignment;
                    break;
                }
            }
            UInt32 alignedStride = ((width + alignment - 1) / alignment) * alignment;
            if (alignedStride != stride) {
                CANDERA_DEVICE_LOG_WARN("Unsupported glyph bitmap stride %d for width %d.", stride, width);
                return false;
            }
        }
    }
    else {
        CANDERA_DEVICE_LOG_WARN("Unsupported glyph bitmap format %d.", format);
        return false;
    }

    bool success = RenderDevice2D::CreateBuffer(surface, width, height,
        static_cast<Int>(Bitmap::AlphaUnsignedBytePixelFormat),
        Bitmap::PackAlignment(alignment), RenderDevice2D::DisplayPackOrder, addressBuffer);
    if (!success) {
        CANDERA_DEVICE_LOG_WARN("Failed to create glyph bitmap buffer.");
        return false;
    }

    return true;
}

bool RenderDevice2D::UpdateSubImage(SurfaceHandle surface,
                                    Int xOffset, Int yOffset,
                                    UInt width, UInt height,
                                    const UInt8* data)
{
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(surface);
    if (surfaceInfo == 0) {
        CANDERA_DEVICE_LOG_WARN("Wrong surface handle");
        return false;
    }
    MemoryManagement::SharedPointer<BitmapTextureImage> bmpTexImage =
        Dynamic_Cast<MemoryManagement::SharedPointer<BitmapTextureImage>, MemoryManagement::SharedPointer<TextureImage> >(surfaceInfo->textureImage);
    if (bmpTexImage == 0) {
        CANDERA_DEVICE_LOG_WARN("No bitmap attached to surface handle");
        return false;
    }
    return bmpTexImage->Update(xOffset, yOffset, width, height, data);
}

UInt32 RenderDevice2D::GetWidth(SurfaceHandle pSurface)
{
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(pSurface);
    if (surfaceInfo != 0){
        if (surfaceInfo->textureImage != 0){
            ImageSource3D* imageSource = surfaceInfo->textureImage->ToImageSource3D();
            if (imageSource != 0) {
                if (imageSource->GetWidth() > 0) {
                    return imageSource->GetWidth();
                }
            }
        }
    }
    return 0;
}

UInt32 RenderDevice2D::GetHeight(SurfaceHandle pSurface)
{
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(pSurface);
    if (surfaceInfo != 0){
        if (surfaceInfo->textureImage != 0){
            ImageSource3D* imageSource = surfaceInfo->textureImage->ToImageSource3D();
            if (imageSource != 0) {
                if (imageSource->GetHeight() > 0) {
                    return imageSource->GetHeight();
                }
            }
        }
    }
    return 0;
}

RenderDevice2D::PackOrder RenderDevice2D::GetPackOrder(SurfaceHandle pSurface)
{
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(pSurface);
    if (surfaceInfo != 0){
        if (surfaceInfo->textureImage != 0){
            return surfaceInfo->packOrder;
        }
    }
    return RenderDevice2D::UnknownPackOrder;
}

Int RenderDevice2D::GetPixelFormat(SurfaceHandle pSurface) {
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(pSurface);
    if (surfaceInfo != 0){
        if (surfaceInfo->textureImage != 0){
            return surfaceInfo->pixelFormat;
        }
    }
    return 0;
}



/**
 * @brief Class used as image source for attached native handles.
 */
class AttachedImageSource3D : public ImageSource3D
{
public:
    AttachedImageSource3D() :
        m_isMipMap(false),
        m_width(0),
        m_height(0),
        m_handle(0)
    {
    }

    ~AttachedImageSource3D() {}

    virtual Handle GetVideoMemoryHandle() const { return m_handle; }
    virtual bool IsMipMappingEnabled() const { return m_isMipMap; }
    virtual Int GetHeight() const { return m_height; }
    virtual Int GetWidth() const { return m_width; }

    virtual void Sync() {}
    virtual void WaitSync() {}

    bool m_isMipMap;
    Int m_width;
    Int m_height;
    Handle m_handle;
};


bool RenderDevice2D::AttachNativeHandle(
    SurfaceHandle surfaceHandle, UInt32 uiWidth, UInt32 uiHeight,
    Int iPixelFormat,
    Handle nativeHandle)
{
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(surfaceHandle);
    if (surfaceInfo == 0){
        CANDERA_DEVICE_LOG_WARN("Wrong surface handle");
        return false;
    }

    AttachedImageSource3D* imageSource = FEATSTD_NEW(AttachedImageSource3D);
    if(imageSource == 0) {
        CANDERA_DEVICE_LOG_WARN("Failed to create texture image");
        return false;
    }
    imageSource->m_handle = nativeHandle;
    imageSource->m_width = uiWidth;
    imageSource->m_height = uiHeight;

    MemoryManagement::SharedPointer<ProxyTextureImage> texImage =
        ProxyTextureImage::Create(
        imageSource,
        &ProxyTextureImage::ImageSourceDisposer::Dispose);
    if(texImage == 0) {
        CANDERA_DEVICE_LOG_WARN("Failed to create texture image");
        return false;
    }
    if (!texImage->Upload()) {
        CANDERA_DEVICE_LOG_WARN("Failed to upload texture image");
        return false;
    }

    surfaceInfo->pixelFormat = iPixelFormat;
    surfaceInfo->packOrder = VerticallyFlippedDisplayPackOrder;

    surfaceInfo->textureImage = texImage;

    return true;
}

bool RenderDevice2D::ConvertBitmap(
    UInt32 dstWidth, UInt32 dstHeight,
    Int dstPixelFormat,
    Bitmap::PackAlignment dstPackAlignment,
    UInt8* dstBuffer,
    UInt32 srcWidth, UInt32 srcHeight,
    Int srcPixelFormat,
    Bitmap::PackAlignment srcPackAlignment,
    const UInt8* srcBuffer)
{
#ifdef FEATSTD_32BIT_PLATFORM
    UInt32 dstSize = ~static_cast<UInt32>(0U);
    UInt32 srcSize = ~static_cast<UInt32>(0U);
    return GenericBitmapConvertor::Convert(
        dstWidth, dstHeight, dstPixelFormat, dstPackAlignment, dstBuffer, &dstSize, 0,
        srcWidth, srcHeight, srcPixelFormat, srcPackAlignment, srcBuffer,  srcSize, 0);
#else
    FEATSTD_UNUSED(dstWidth);
    FEATSTD_UNUSED(dstHeight);
    FEATSTD_UNUSED(dstPixelFormat);
    FEATSTD_UNUSED(dstPackAlignment);
    FEATSTD_UNUSED(dstBuffer);
    FEATSTD_UNUSED(srcWidth);
    FEATSTD_UNUSED(srcHeight);
    FEATSTD_UNUSED(srcPixelFormat);
    FEATSTD_UNUSED(srcPackAlignment);
    FEATSTD_UNUSED(srcBuffer);
    return false;
#endif
}

bool RenderDevice2D::SetPerspectiveWarpingEnabled(ContextHandle2D /*context*/, bool /*enable*/)
{
    return false;
}

bool RenderDevice2D::IsPerspectiveWarpingEnabled(ContextHandle2D /*context*/)
{
    return false;
}

bool RenderDevice2D::SetPerspectiveTransformationMatrix(ContextHandle2D /*context*/, const Matrix4& /*mat*/)
{
    return false;
}

bool RenderDevice2D::SetPerspectiveProjectionMatrix(ContextHandle2D /*context*/, const Matrix4& /*mat*/)
{
    return false;
}

bool RenderDevice2D::SetPerspectiveViewport(ContextHandle2D /* context */, const Rectangle& /* viewport */)
{
    return false;
}


bool RenderDevice2D::CalculatePerspectiveProjectionMatrix(Matrix4& /* matrix */,
 const Camera2D* /* camera */, const Rectangle& /* viewport */,
 Float /* nearZ */, Float /* farZ */, Float /* fovY */, Float /* aspectRatio */)
{
    return false;
}

bool RenderDevice2D::CalculatePerspectiveTransformationMatrix(Matrix4& /* matrix */, const Vector3& /* position */,
    const Vector3& /* rotation */, const Vector3& /* scale */, const Vector3& /* pivot */)
{
    return false;
}

UInt32 RenderDevice2D::GetSize(const Bitmap* bitmap, UInt memoryPool)
{
    return RenderDevice::GetSize(bitmap, static_cast<TextureImage::TextureMemoryPool>(memoryPool));
}

bool RenderDevice2D::SetScissorRectangle(ContextHandle2D context, const Rectangle& scissorRectangle)
{
    if (0 == context) {
        return false;
    }

    Context2DOver3D& context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D*>(context);
    const Context2DOver3DTarget& target = context2d.GetTarget();

    Float renderTargetWidth = target.actualWidth;
    Float renderTargetHeight = target.actualHeight;

    context2d.GetState().userScissorRectangle.Set(scissorRectangle.GetLeft(), scissorRectangle.GetTop(),
        scissorRectangle.GetWidth() < 0.0F ? renderTargetWidth : scissorRectangle.GetWidth(),
        scissorRectangle.GetHeight() < 0.0F ? renderTargetHeight : scissorRectangle.GetHeight());

    return true;
}

bool RenderDevice2D::GetScissorRectangle(ContextHandle2D context, Rectangle& scissorRectangle)
{
    if (0 == context) {
        return false;
    }

    Context2DOver3D& context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D*>(context);
    scissorRectangle = context2d.GetState().userScissorRectangle;
    return true;
}

bool RenderDevice2D::SetScissorEnabled(ContextHandle2D context, bool isEnabled)
{
    if (0 == context) {
        return false;
    }

    Context2DOver3D& context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D*>(context);
    context2d.GetState().isScissorEnabled = isEnabled;
    return true;
}

bool RenderDevice2D::IsScissorEnabled(ContextHandle2D context)
{
    if (0 == context) {
        return false;
    }

    Context2DOver3D& context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D*>(context);
    return context2d.GetState().isScissorEnabled;
}

bool RenderDevice2D::IsMipMappingSupported() {
    return true;
}

bool RenderDevice2D::HasBuffer(SurfaceHandle surface)
{
    SurfaceAllocator2DOver3D::SurfaceInfo* surfaceInfo = SurfaceAllocator2DOver3D::GetInstance().GetBase().GetDataFromSurfaceHandle(surface);
    if (surfaceInfo == 0) {
        CANDERA_DEVICE_LOG_WARN("Wrong surface handle");
        return false;
    }

    return (!surfaceInfo->textureImage.PointsToNull()) && (surfaceInfo->textureImage->IsUploaded());
}

//Functions for handling VertexBuffer2D

bool RenderDevice2D::SetGeometry(ContextHandle2D context, GeometryHandle geometry, const Rectangle& boundingRectangle)
{
    if (context == 0) {
        return false;
    }

    Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
    if (geometry != 0) {

        VertexBuffer::SharedPointer vertexBuffer = GeometryAllocator2DOver3D::GetVertexBuffer(geometry);
        if (!vertexBuffer.PointsToNull()) {
            Context2DOver3DDevicePool& pool = Context2DOver3DDevicePool::GetInstance();
            pool.SetCustomVertexBuffer(vertexBuffer, boundingRectangle, false);
            context2d.SetCustomGeometry(geometry);
            return true;
        }
    }
    else {
        Context2DOver3DDevicePool& pool = Context2DOver3DDevicePool::GetInstance();
        pool.SetCustomVertexBuffer(VertexBuffer::SharedPointer(0));
        context2d.SetCustomGeometry(0);
        return true;
    }


    return false;
}

GeometryHandle RenderDevice2D::GetGeometry(ContextHandle2D context)
{
    GeometryHandle returnValue = 0;
    if (0 != context) {
        Context2DOver3D &context2d = *FeatStd::Internal::ScalarToPointer<Context2DOver3D *>(context);
        return context2d.GetCustomGeometry();
    }

    return returnValue;
}


bool RenderDevice2D::UpdateVertexBuffer2D(VertexBuffer2D& vb, Int offset, Int size, const UInt8* data)
{

    GeometryHandle geometryHandle = vb.Get2DGeometryHandle();
    if (geometryHandle == 0) {
        CANDERA_DEVICE_LOG_WARN("Cannot be updated if not Uploaded.");
        return false;
    }

    GeometryAllocator2DOver3D::GeometryInfo* geometryInfo =
        GeometryAllocator2DOver3D::GetInstance().GetBase().GetDataFromGeometryHandle(geometryHandle);

    if (geometryInfo == 0) {
        CANDERA_DEVICE_LOG_WARN("GeometryHandle could not be resolved.");
        return false;
    }

    VertexBuffer::SharedPointer vb3D = geometryInfo->vertexBuffer;

    if (!vb3D.PointsToNull()) {
        CANDERA_DEVICE_LOG_WARN("Cannot be updated if not Uploaded.");
        return false;
    }

    return vb3D->Update(offset, size, data);
}

UInt32 RenderDevice2D::GetVertexBufferSize(const VertexBuffer2D* vb, UInt /*memoryPool*/)
{
    if (0 == vb) {
        return 0;
    }

    const Candera::VertexGeometry2D* vg = vb->GetVertexGeometry2D();

    if (0 == vg) {
        return 0;
    }

    return (vg->GetVertexCount() * sizeof(VertexGeometry2D::Vertex2D));
}


VertexGeometry::MemoryPool MapVertexGeometry2DMemoryPool(const VertexGeometry2D::MemoryPool memoryPool2D) {
    VertexGeometry::MemoryPool memoryPool = VertexGeometry::VideoMemory;

    switch (memoryPool2D) {
        case VertexGeometry2D::VideoMemory:
            memoryPool = VertexGeometry::VideoMemory;
            break;
        case VertexGeometry2D::ClientMemory:
            memoryPool = VertexGeometry::SystemMemory;
            break;
        default:
            memoryPool = VertexGeometry::VideoMemory;
    }

    return memoryPool;
}

VertexGeometry::BufferUsage MapVertexGeometry2DBufferUsage(const VertexGeometry2D::BufferUsage bufferUsage2D) {
    VertexGeometry::BufferUsage bufferUsage = VertexGeometry::StaticWrite;

    switch (bufferUsage2D) {
    case VertexGeometry2D::StaticWrite:
        bufferUsage = VertexGeometry::StaticWrite;
        break;
    case VertexGeometry2D::DynamicWrite:
        bufferUsage = VertexGeometry::DynamicWrite;
        break;
    case VertexGeometry2D::InfrequentDraw:
        bufferUsage = VertexGeometry::InfrequentDraw;
        break;
    default:
        bufferUsage = VertexGeometry::StaticWrite;
    }

    return bufferUsage;
}

VertexGeometry::BufferType MapVertexGeometry2DBufferType(const VertexGeometry2D::BufferType bufferType2D) {
    VertexGeometry::BufferType bufferType = VertexGeometry::ArrayBuffer;

    switch (bufferType2D) {
    case VertexGeometry2D::ArrayBuffer:
        bufferType = VertexGeometry::ArrayBuffer;
        break;
    case VertexGeometry2D::UInt16IndexedArrayBuffer:
        bufferType = VertexGeometry::UInt16IndexedArrayBuffer;
        break;
    default:
        bufferType = VertexGeometry::ArrayBuffer;
    }

    return bufferType;
}

VertexBuffer::PrimitiveType MapPrimitiveType2D(const VertexBuffer2D::PrimitiveType primitiveType2D) {
    VertexBuffer::PrimitiveType primitiveType = VertexBuffer::Triangles;

    switch (primitiveType2D) {
    case VertexBuffer2D::Triangles:
        primitiveType = VertexBuffer::Triangles;
        break;
    case VertexBuffer2D::TriangleStrip:
        primitiveType = VertexBuffer::TriangleStrip;
        break;
    case VertexBuffer2D::TriangleFan:
        primitiveType = VertexBuffer::TriangleFan;
        break;
    case VertexBuffer2D::Lines:
        primitiveType = VertexBuffer::Lines;
        break;
    case VertexBuffer2D::LineStrip:
        primitiveType = VertexBuffer::LineStrip;
        break;
    case VertexBuffer2D::LineLoop:
        primitiveType = VertexBuffer::LineLoop;
        break;
    case VertexBuffer2D::Points:
        primitiveType = VertexBuffer::Points;
        break;
    default:
        primitiveType = VertexBuffer::Triangles;
    }

    return primitiveType;
}

static const VertexGeometry::VertexElementFormat vertex2DFormat[] = {
    { 0, VertexGeometry::Float32_2, VertexGeometry::Position, 0 },
    { 8, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 }
};
ResourceDataHandle GetVertexElementFormatHandle() {
    static ResourceDataHandle formatArrayHandle = VertexGeometry::FormatArrayResource::CreateHandle(&vertex2DFormat[0], 0, 2 * sizeof(VertexGeometry::VertexElementFormat));
    return formatArrayHandle;
}

static VertexBuffer::SharedPointer CreateVertexBuffer(const VertexGeometry2D* vertexGeometry, VertexBuffer2D::PrimitiveType primitiveType)
{
    VertexBuffer::SharedPointer createdVB = VertexBuffer::SharedPointer(0);
    if (0 != vertexGeometry) {

        //VertexGeometry shall not take ownership, so no disposer functions shall be set.

        VertexGeometry* vg = FEATSTD_NEW(VertexGeometry)(
            vertexGeometry->GetVertexArray2DResourceHandle(),
            vertexGeometry->GetIndexArrayResourceHandle(),
            GetVertexElementFormatHandle(),
            sizeof(VertexGeometry2D::Vertex2D),
            MapVertexGeometry2DMemoryPool(vertexGeometry->GetMemoryPool()),
            MapVertexGeometry2DBufferType(vertexGeometry->GetBufferType()),
            MapVertexGeometry2DBufferUsage(vertexGeometry->GetBufferUsage())
            );

        createdVB = VertexBuffer::Create();

        if ((!createdVB.PointsToNull()) && (0 != vg)) {
            static_cast<void>(createdVB->SetVertexGeometry(vg, VertexBuffer::VertexGeometryDisposer::Dispose));
            createdVB->SetPrimitiveType(MapPrimitiveType2D(primitiveType));
        }

    }
    return createdVB;
}

bool RenderDevice2D::UploadVertexBuffer2D(VertexBuffer2D& vb)
{

    if (0 != vb.Get2DGeometryHandle()) {
        return true;
    }

    if (!IsPrimitiveTypeSupported(vb.GetPrimitiveType())) {
        CANDERA_DEVICE_LOG_WARN("Primitive type not supported.");
        return false;
    }

    if (0 == vb.GetVertexGeometry2D()) {
        CANDERA_DEVICE_LOG_WARN("No VertexGeoemetry attached to VertexBuffer2D.");
        return false;
    }

    const VertexGeometry2D* vg2D = vb.GetVertexGeometry2D();
    GeometryHandle geometryHandle = 0;

    if (0 == vg2D) {
        CANDERA_DEVICE_LOG_WARN("VertexGeometry2D is 0.");
        return false;
    }

    if ((vg2D->GetIndexCount() > 0) && (!IsIndexBufferSupported())) {
        CANDERA_DEVICE_LOG_WARN("Index Buffers are not supported.");
        return false;
    }

    if (!GeometryAllocator2DOver3D::GetInstance().GetBase().CreateGeometries(1, &geometryHandle)) {
        CANDERA_DEVICE_LOG_WARN("Failed to create geometry handle.");
        return false;
    }

    GeometryAllocator2DOver3D::GeometryInfo* geometryInfo =
        GeometryAllocator2DOver3D::GetInstance().GetBase().GetDataFromGeometryHandle(geometryHandle);
    if (geometryInfo == 0) {
        CANDERA_DEVICE_LOG_WARN("Invalid geometry handle.");
        return false;
    }

    VertexBuffer::SharedPointer vertexBuffer = CreateVertexBuffer(vg2D, vb.GetPrimitiveType());
    if (vertexBuffer.PointsToNull()) {
        CANDERA_DEVICE_LOG_WARN("Failed to create VertexBuffer.");
        return false;
    }

    if (!vertexBuffer->Upload()) {
        CANDERA_DEVICE_LOG_WARN("Failed to Upload VertexBuffer.");
        return false;
    }

    geometryInfo->vertexBuffer = vertexBuffer;
    vb.SetGeometryHandle(geometryHandle);

    return true;
}

bool RenderDevice2D::UnloadVertexBuffer2D(VertexBuffer2D& vb)
{
    GeometryHandle geometryHandle = vb.Get2DGeometryHandle();
    if (geometryHandle == 0) {
        return true;
    }

    bool success = GeometryAllocator2DOver3D::GetInstance().GetBase().DestroyGeometries(1, &geometryHandle);

    if (success) {
        vb.SetGeometryHandle(0);
        return true;
    }
    else {
        return false;
    }
}


bool RenderDevice2D::IsPrimitiveTypeSupported(const Int primitiveType)
{
    return ((primitiveType >= VertexBuffer2D::Triangles) && (primitiveType <= VertexBuffer2D::Points));
}

bool RenderDevice2D::IsIndexBufferSupported()
{
    return true;
}

}
