//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlOutlineTextBrushBlend.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    CANDERA_RTTI_DEFINITION(GlOutlineTextBrushBlend)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GlOutlineTextBrushBlend::GlOutlineTextBrushBlend()
    {
       //make TextBrushCacheType::BitmapCache default value
       m_textBrush.CacheType().Set(BitmapCache);
    }

    GlOutlineTextBrushBlend::GlOutlineTextBrushBlend(const GlOutlineTextBrushBlend& rhs) : 
        Base(rhs),
        m_textBrush(rhs.m_textBrush),
        m_blendEffect(rhs.m_blendEffect),
        m_outlineEffect(rhs.m_outlineEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlOutlineTextBrushBlend::~GlOutlineTextBrushBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlOutlineTextBrushBlend::SharedPointer GlOutlineTextBrushBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlOutlineTextBrushBlend* brush = FEATSTD_NEW(GlOutlineTextBrushBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlOutlineTextBrushBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlOutlineTextBrushBlend::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                                     const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "GlOutlineTextBrushBlend"));

        if (m_textBrush.Text() != 0) {
            m_blendEffect.ActivateBlend(output);
            m_outlineEffect.ColorFactor() = 1;
            m_outlineEffect.ActivateFilter(output);
            m_textBrush.ActivateColor(node, output);
            m_textBrush.RenderText(transform, output, outputArea);
            m_textBrush.DeactivateColor(output);
            m_outlineEffect.DeactivateFilter(output);
            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool GlOutlineTextBrushBlend::Upload()
    {
        bool success = m_textBrush.Upload();
        success = m_blendEffect.Upload() && success;
        return success;
    }
    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool GlOutlineTextBrushBlend::Unload()
    {
        bool success = m_textBrush.Unload();
        success = m_blendEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool GlOutlineTextBrushBlend::Update()
    {
        bool success = m_textBrush.Update();
        success = m_blendEffect.Update() && success;
        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlOutlineTextBrushBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlOutlineTextBrushBlend)(*this));
    }

    /******************************************************************************
    *  GetInPlaceEffect2D
    ******************************************************************************/
    Candera::InPlaceEffect2D* GlOutlineTextBrushBlend::GetInPlaceEffect2D(UInt8 index)
    {
        FEATSTD_DEBUG_ASSERT(index < 1);
        return (index == 0) ? &m_outlineEffect : 0;
    }

    /******************************************************************************
     *  GetBoundingRectangle
     ******************************************************************************/
    void GlOutlineTextBrushBlend::GetBoundingRectangle(Rectangle& boundingRectangle) const
    {
        //retrieve the outline width from the effect. Do not change following code without checking if const casts are still ok.
        const FeatStd::UInt8 width = const_cast<SimpleProperty<UInt8> &>(const_cast<GlOutlineEffectWrapper&>(m_outlineEffect).OutlineWidth());
        m_textBrush.GetBoundingRectangle(boundingRectangle);
        boundingRectangle.Set(boundingRectangle.GetLeft() - static_cast<Float>(width),
                              boundingRectangle.GetTop() - static_cast<Float>(width),
                              boundingRectangle.GetWidth() + 2.0F * static_cast<Float>(width),
                              boundingRectangle.GetHeight() + 2.0F * static_cast<Float>(width));
    }

}   // namespace Candera
