//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOUTLINE_BITMAP_BRUSH_COLOR_BLEND_H)
#define CANDERA_GLOUTLINE_BITMAP_BRUSH_COLOR_BLEND_H

#include <Candera/Engine2D/Effects/CombinedEffect2D.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <CanderaPlatform/Device/Common/Effects/ColorEffect.h>
#include <CanderaPlatform/Device/Common/Effects/BlendEffect.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlOutlineEffect.h>

namespace Candera {
    /**
    *  @brief  Output a bitmap image that has an alpha channel, outlined using GlOutlineEffect, apply a ColorEffect and blend it with the store buffer.
    *          Same as chaining (BitmapBrush  + ColorEffect + BlendEffect + GlOutlineEffect)
    *          @see Candera::BitmapBrush
    *          @see Candera::GlOutlineEffect
    *          @see Candera::BlendEffect
    */
    class GlOutlineBitmapBrushColorBlend : public CombinedEffect2D {
        FEATSTD_TYPEDEF_BASE(CombinedEffect2D);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(GlOutlineBitmapBrushColorBlend);

        CANDERA_RTTI_DECLARATION(Candera::GlOutlineBitmapBrushColorBlend);

        /**
        *  Creates an instance of this class
        *  @return A pointer to the created object.
        */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
        *  Destructor
        */
        virtual ~GlOutlineBitmapBrushColorBlend();

        /**
        *  Gets a valid BrushEffect2D pointer if the effect is a brush effect or
        *  if it is a combined effect and has a brush effect, otherwise 0.
        *  @return     A valid BrushEffect2D pointer if the effect is a brush effect or
        *              if it is a combined effect and has a brush effect, otherwise 0.
        */
        virtual BrushEffect2D* GetBrushEffect2D() { return &m_bitmapBrush; }

        // overrides Effect2D::GetInplaceEffect2DCount
        virtual UInt8 GetInPlaceEffect2DCount() const { return 2; }

        // overrides Effect2D::GetInPlaceEffect2D
        virtual InPlaceEffect2D* GetInPlaceEffect2D(UInt8 index);

        /**
        *  Gets a valid BlendEffect2D pointer if the effect is a blend effect or
        *  if it is a combined effect and has a blend effect, otherwise 0.
        *  @return     A valid BlendEffect2D pointer if the effect is a blend effect or
        *              if it is a combined effect and has a blend effect, otherwise 0.
        */
        virtual BlendEffect2D* GetBlendEffect2D() { return &m_blendEffect; }

        /**
        *  Uploads all associated resources.
        *  @return true if successful.
        */
        virtual bool Upload();

        /**
        *  Unloads all associated resources.
        *  @return true if successful.
        */
        virtual bool Unload();

        /**
        *  Uploads all resources that changed since last Upload/Update.
        *  Unloads resources removed from the effect, that were previously uploaded
        *  by an Upload/Update.
        *  This should only be called on Uploaded effects. (otherwise, call Upload.)
        *  @return true if successful.
        */
        virtual bool Update();

        /**
        *  Retrieves the bitmap brush.
        *  @return     The bitmap brush.
        */
        const BitmapBrush& GetBitmapBrush() const { return m_bitmapBrush; }
        BitmapBrush& GetBitmapBrush() { return m_bitmapBrush; }

        /**
        *  Retrieves the outline effect.
        *  @return     The outline effect.
        */
        const GlOutlineEffect& GetOutlineEffect() const { return m_outlineEffect; }
        GlOutlineEffect& GetOutlineEffect() { return m_outlineEffect; }

        /**
        *  Retrieves the color effect.
        *  @return     The color effect.
        */
        const ColorEffect& GetColorEffect() const { return m_colorEffect; }
        ColorEffect& GetColorEffect() { return m_colorEffect; }

        /**
        *  Retrieves the blend effect.
        *  @return     The blend effect.
        */
        const BlendEffect& GetBlendEffect() const { return m_blendEffect; }
        BlendEffect& GetBlendEffect() { return m_blendEffect; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DCombinedDef(Candera::GlOutlineBitmapBrushColorBlend, GlOutlineBitmapBrushColorBlend, EFFECT2D_TYPE_BRUSH, EFFECT2D_TYPE_BLEND)
           CdaDescription("This combined effect is designed for use with TextNode2D! Do not use with any text cache type other than BitmapCache! It adds an outline to an image depending on the data in the alpha channel of an image, modulates the color, and blends it with the store buffer.")
            CdaEffect2DProperties()

            CdaEffect2DImportProperties(BitmapBrush, Candera::BitmapBrush, m_bitmapBrush)
            CdaEffect2DImportProperties(GlOutlineEffect, Candera::GlOutlineEffect, m_outlineEffect)
            CdaEffect2DImportProperties(ColorEffect, Candera::ColorEffect, m_colorEffect)
            CdaEffect2DImportProperties(BlendEffect, Candera::BlendEffect, m_blendEffect)

            CdaEffect2DPropertiesEnd()
            CdaEffect2DDefEnd()
            /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GlOutlineBitmapBrushColorBlend();
        explicit GlOutlineBitmapBrushColorBlend(const GlOutlineBitmapBrushColorBlend& rhs);
        GlOutlineBitmapBrushColorBlend& operator = (const GlOutlineBitmapBrushColorBlend& rhs);

        /**
        *  Triggers the rendering for the effect.
        *  @param  input       Defines the input for the effect. The first effect inside the effect list NO input. The input for the second
        *                      effect in the list must be set to the output of the first effect and so on.
        *  @param  inputArea   Area of the input that should be used as input. This should by supplied through the ouputArea parameter of the last effect.
        *  @param  transform   Defines the transformation matrix for the effect.
        *  @param  node        Node this effect is referenced to, containing transformation matrix for the effect.
        *  @param  output      Defines the output for the effect. The last effect inside the effect list must be the context of the render target.
        *  @param  outputArea  Output parameter through which the effect supplies the area of the output that was modified.
        */
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        class GlOutlineEffectWrapper : public GlOutlineEffect {} m_outlineEffect;
        class BitmapBrushWrapper : public BitmapBrush {
        public:
            using BitmapBrush::Render;
        } m_bitmapBrush;
        class ColorEffectWrapper : public ColorEffect {} m_colorEffect;
        class BlendEffectWrapper : public BlendEffect {} m_blendEffect;
    };
}
#endif  // CANDERA_GLFLIP_BITMAP_BRUSH_COLOR_BLEND_H

