//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlOutlineBitmapBrushColorBlend.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/System/Mathematics/Matrix3.h>
#include <Candera/Engine2D/Core/TextNode2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    CANDERA_RTTI_DEFINITION(GlOutlineBitmapBrushColorBlend)
        /******************************************************************************
        *  Constructor
        ******************************************************************************/
        GlOutlineBitmapBrushColorBlend::GlOutlineBitmapBrushColorBlend() :
        Base()
    {
    }

    GlOutlineBitmapBrushColorBlend::GlOutlineBitmapBrushColorBlend(const GlOutlineBitmapBrushColorBlend& rhs) :
        Base(rhs),
        m_outlineEffect(rhs.m_outlineEffect),
        m_bitmapBrush(rhs.m_bitmapBrush),
        m_colorEffect(rhs.m_colorEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
    *  Destructor
    ******************************************************************************/
    GlOutlineBitmapBrushColorBlend::~GlOutlineBitmapBrushColorBlend()
    {
    }

    /******************************************************************************
    *  Create
    ******************************************************************************/
    GlOutlineBitmapBrushColorBlend::SharedPointer GlOutlineBitmapBrushColorBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER);

        GlOutlineBitmapBrushColorBlend* brush = FEATSTD_NEW(GlOutlineBitmapBrushColorBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlOutlineBitmapBrushColorBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
    *  GetInPlaceEffect2D
    ******************************************************************************/
    InPlaceEffect2D* GlOutlineBitmapBrushColorBlend::GetInPlaceEffect2D(UInt8 index)
    {
        FEATSTD_DEBUG_ASSERT(index < 2);
        if (index == 0) {
            return &m_outlineEffect;
        }
        else {
            return &m_colorEffect;
        }
    }

    /******************************************************************************
    *  Render
    ******************************************************************************/
    void GlOutlineBitmapBrushColorBlend::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node,
        ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "GlOutlineBitmapBrushColorBlend"));

        if (m_bitmapBrush.Image() != 0) {

            m_blendEffect.ActivateBlend(output);

            if (Dynamic_Cast<const TextNode2D*>(&node) != 0) {
                m_outlineEffect.ColorFactor() = 1;
            }

            m_outlineEffect.ActivateFilter(output);
            Color::Data& color = m_colorEffect.Color().Get();
            Float cameraAlpha = 1.0F;
            const Camera2D* camera2D = Renderer2D::GetActiveCamera();
            if (0 != camera2D) {
                if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                    cameraAlpha = camera2D->GetEffectiveAlphaValue();
                }
            }
            bool result = RenderDevice2D::SetSurfaceConstColor(
                output,
                RenderDevice2D::SourceSurface,
                color.red,
                color.green,
                color.blue,
                color.alpha * node.GetEffectiveAlphaValue() * cameraAlpha
                );
            if (result) {
                m_bitmapBrush.Render(input, inputArea, transform, node, output, outputArea);
                static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));
            }
            m_outlineEffect.ColorFactor() = 0;
            m_outlineEffect.DeactivateFilter(output);
            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
    *  Upload
    ******************************************************************************/
    bool GlOutlineBitmapBrushColorBlend::Upload()
    {
        bool success = m_bitmapBrush.Upload();
        success = m_outlineEffect.Upload() && success;
        success = m_colorEffect.Upload() && success;
        success = m_blendEffect.Upload() && success;
        return success;
    }
    /******************************************************************************
    *  Unload
    ******************************************************************************/
    bool GlOutlineBitmapBrushColorBlend::Unload()
    {
        bool success = m_bitmapBrush.Unload();
        success = m_outlineEffect.Unload() && success;
        success = m_colorEffect.Unload() && success;
        success = m_blendEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
    *  Update
    ******************************************************************************/
    bool GlOutlineBitmapBrushColorBlend::Update()
    {
        bool success = m_bitmapBrush.Update();
        success = m_outlineEffect.Update() && success;
        success = m_colorEffect.Update() && success;
        success = m_blendEffect.Update() && success;
        return success;
    }

    /******************************************************************************
    *  Clone
    ******************************************************************************/
    Effect2D::SharedPointer GlOutlineBitmapBrushColorBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlOutlineBitmapBrushColorBlend)(*this));
    }

}

