//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GL_LINEAR_GRADIENT_BRUSH_H)
#define CANDERA_GL_LINEAR_GRADIENT_BRUSH_H

#include <Candera/Engine2D/Effects/BrushEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Vector2Property.h>
#include <Candera/Engine2D/Property/ColorProperty.h>

namespace Candera {


    /**
     *  @brief Output a rectangle filled with a linear gradient effect between two colors.
     * 
     * The brush fills a rectangle of a defined Size() with generated linear gradient between a
     * NegativeColor() and a PositiveColor(). The GradientCenter(), GradientDirection() and the
     * GradientMagnitude() define the gradient layout.
     */
    class GlLinearGradientBrush : public BrushEffect2D {
        FEATSTD_TYPEDEF_BASE(BrushEffect2D);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(GlLinearGradientBrush);
        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Access the size of the fill area. This is a two component vector
         *  of which the first component represents the width and the second the height
         *  in pixels of the filled rectangle, in local space.
         *  @return A property that can be used to modify the rectangle size.
         */
        Vector2Property& Size() { return m_size; }

        /**
         *  Defines a point on the line where the gradient effect is centered (both colors are half opaque).
         * @return Property that can be used to modify the gradient center.
         */
        Vector2Property& GradientCenter() { return m_gradientCenter; }

        /**
         *  Defines the direction of the gradient effect. The direction is perpendicular to the centered gradient line.
         * The opacity of the PositiveColor() increases on the defined direction and decreases in the opposite direction.
         * The opacity of the NegativeColor() decreases on the defined direction and increases in the opposite direction.
         * @return Property that can be used to modify the gradient direction.
         */
        Vector2Property& GradientDirection() { return m_gradientDirection; }

        /**
         *  Defines the magnitude of the gradient effect. 
         * The magnitude is the distance from the center line to the closest parallel line where the PositiveColor()
         * is fully opaque on the GradientDirection() and NegativeColor() is fully opaque on the opposite direction.
         * @return Property that can be used to modify the gradient magnitude.
         */
        FloatProperty& GradientMagnitude() { return m_gradientMagnitude; }

        /**
         *  Defines the positive color.
         *  @return Property that can be used to modify the positive color.
         */
        ColorProperty& PositiveColor() { return m_positiveColor; }

        /**
         *  Defines the negative color.
         *  @return Property that can be used to modify the negative color.
         */
        ColorProperty& NegativeColor() { return m_negativeColor; }

        // overrides Effect2D::GetBoundingRectangle
        virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const override;

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const override;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::GlLinearGradientBrush, GlLinearGradientBrush, EFFECT2D_TYPE_BRUSH)

            CdaDescription("The brush generates a linear gradient between two colors. The size of the generated rectangle"
                " is equal to the effective scale factor of the Node2D onto which it is applied.")
            CdaEffect2DProperties()
                CdaEffect2DImportBaseClassProperties(BrushEffect2D, Candera::BrushEffect2D)
                
                CdaEffect2DProperty(Size, Vector2, m_size)
                    CdaDescription("The filled size (width and height).")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(GradientCenter, Vector2, m_gradientCenter)
                    CdaDescription("Defines a point on the line where the gradient effect is centered (both colors are half opaque).")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(GradientDirection, Vector2, m_gradientDirection)
                    CdaDescription(" Defines the direction of the gradient effect. The direction is normal to the centered gradient line."
                    " The opacity of the 'positive' color increases on the defined direction and decreases in the opposite direction."
                    " The opacity of the 'negative' color decreases on the defined direction and increases in the opposite direction.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(GradientMagnitude, Float, m_gradientMagnitude)
                    CdaDescription(" Defines the magnitude of the gradient effect."
                    " The magnitude is the distance from the center line to the closest parallel line where the PositiveColor()"
                    " is fully opaque on the GradientDirection() and NegativeColor() is fully opaque on the opposite direction.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(PositiveColor, Color, m_positiveColor)
                    CdaDescription("Defines the positive color.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(NegativeColor, Color, m_negativeColor)
                    CdaDescription("Defines the negative color.")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

        /**
         *  Activate the parameters used for gradient generation.
         *  @param node The effective alpha value of the Node2D is relevant.
         *  @param output The parameters are activated for this 2D context.
         */
        void Activate(const Node2D& node, ContextHandle2D output);

        /**
         *  Deactivate the color used for for gradient generation.
         *  @param output The parameters are deactivated for this 2D context.
         */
        void Deactivate(ContextHandle2D output) const;

        /**
         *  Render the gradient without setting the colors.
         *  @param transform    3x2 transform matrix.
         *  @param output       The 2D context handle.
         *  @param outputArea   The output area as a rectangle.
         */
        void RenderGradient(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea);

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GlLinearGradientBrush();
        explicit GlLinearGradientBrush(const GlLinearGradientBrush& rhs);
        virtual ~GlLinearGradientBrush();
        GlLinearGradientBrush& operator = (const GlLinearGradientBrush& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea) override;

    private:
        Vector2Property m_size;
        Vector2Property m_gradientCenter;
        Vector2Property m_gradientDirection;
        FloatProperty m_gradientMagnitude;
        ColorProperty m_positiveColor;
        ColorProperty m_negativeColor;
    };

}   // namespace Candera

#endif  // CANDERA_GL_LINEAR_GRADIENT_BRUSH_H
