//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlFlipEffect.h"

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/RenderDevice2DOver3D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/RenderDirectionHelper.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(GlFlipEffect, InPlaceEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GlFlipEffect::GlFlipEffect() :
        Base(),
        m_flipH(false),
        m_flipV(false)
    {
    }

    GlFlipEffect::GlFlipEffect(const GlFlipEffect& rhs) :
        Base(rhs),
        m_flipH(rhs.m_flipH.Get()),
        m_flipV(rhs.m_flipV.Get())
    {
        SetUseNodeLayoutDirectionEnabled(rhs.IsUseNodeLayoutDirectionEnabled());
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlFlipEffect::~GlFlipEffect()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlFlipEffect::SharedPointer GlFlipEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlFlipEffect* brush = FEATSTD_NEW(GlFlipEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlFlipEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlFlipEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& /*node*/,
                                 ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "GlFlipEffect"));
        
        bool result = Renderer2D::SetTransformationMatrix(
            output,
            RenderDevice2D::SourceSurface,
            transform);

        result = result && RenderDevice2D::SetSurface(
            output,
            RenderDevice2D::SourceSurface,
            input);

        result = result && RenderDevice2D::SetActiveArea(
            output,
            RenderDevice2D::SourceSurface,
            inputArea.GetLeft(),
            inputArea.GetTop(),
            inputArea.GetWidth(),
            inputArea.GetHeight());

        if (result) {
            ActivateFlip(output);
            result = Renderer2D::Blit(output);
            DeactivateFlip(output);
        }

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            static_cast<void>(RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height));
            outputArea = Rectangle(left, top, width, height);
        }
    }

    /******************************************************************************
     *  Activate
     ******************************************************************************/
    void GlFlipEffect::ActivateFlip(ContextHandle2D output)
    {
        bool flip = m_flipH;
        if (IsUseNodeLayoutDirectionEnabled() && Internal::RenderDirectionHelper::IsCurrentRightToLeft())
        {
            flip = !m_flipH;
        }
        static_cast<void>(Internal::RenderDevice2DOver3D::SetFlipMode(output, flip, m_flipV));
    }
    /******************************************************************************
     *  Deactivate
     ******************************************************************************/
    void GlFlipEffect::DeactivateFlip(ContextHandle2D output) const
    {
        static_cast<void>(Internal::RenderDevice2DOver3D::SetFlipMode(output, false, false));
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlFlipEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlFlipEffect)(*this));
    }

    void GlFlipEffect::SetUseNodeLayoutDirectionEnabled(bool enable)
    {
        Internal::RenderDirectionHelper::SetUseNodeLayoutDirection(*this, enable);
    }

    bool GlFlipEffect::IsUseNodeLayoutDirectionEnabled() const
    {
        return Internal::RenderDirectionHelper::GetUseNodeLayoutDirection(*this);
    }

}   // namespace Candera

