//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlDropShadowTextBrushBlend.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    CANDERA_RTTI_DEFINITION(GlDropShadowTextBrushBlend)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GlDropShadowTextBrushBlend::GlDropShadowTextBrushBlend()
    {
       //make TextBrushCacheType::BitmapCache default value
       m_textBrush.CacheType().Set(BitmapCache);
    }

    GlDropShadowTextBrushBlend::GlDropShadowTextBrushBlend(const GlDropShadowTextBrushBlend& rhs) : 
        Base(rhs),
        m_textBrush(rhs.m_textBrush),
        m_blendEffect(rhs.m_blendEffect),
        m_dropShadowEffect(rhs.m_dropShadowEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlDropShadowTextBrushBlend::~GlDropShadowTextBrushBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlDropShadowTextBrushBlend::SharedPointer GlDropShadowTextBrushBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlDropShadowTextBrushBlend* brush = FEATSTD_NEW(GlDropShadowTextBrushBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlDropShadowTextBrushBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlDropShadowTextBrushBlend::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                                     const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "GlDropShadowTextBrushBlend"));

        if (m_textBrush.Text() != 0) {
            m_blendEffect.ActivateBlend(output);

            // Drop Shadow Effect
            if (m_dropShadowEffect.ShadowEnabled().Get()) {
                m_dropShadowEffect.ActivateFilter(output);
                m_textBrush.RenderText(transform, output, outputArea);
                m_dropShadowEffect.DeactivateFilter(output);
            }

            // Text
            m_textBrush.ActivateColor(node, output);
            m_textBrush.RenderText(transform, output, outputArea);
            m_textBrush.DeactivateColor(output);

            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool GlDropShadowTextBrushBlend::Upload()
    {
        bool success = m_textBrush.Upload();
        success = m_blendEffect.Upload() && success;
        return success;
    }
    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool GlDropShadowTextBrushBlend::Unload()
    {
        bool success = m_textBrush.Unload();
        success = m_blendEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool GlDropShadowTextBrushBlend::Update()
    {
        bool success = m_textBrush.Update();
        success = m_blendEffect.Update() && success;
        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlDropShadowTextBrushBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlDropShadowTextBrushBlend)(*this));
    }

    /******************************************************************************
    *  GetInPlaceEffect2D
    ******************************************************************************/
    Candera::InPlaceEffect2D* GlDropShadowTextBrushBlend::GetInPlaceEffect2D(UInt8 index)
    {
        FEATSTD_DEBUG_ASSERT(index < 1);
        return (index == 0) ? &m_dropShadowEffect : 0;
    }

    /******************************************************************************
     *  GetBoundingRectangle
     ******************************************************************************/
    void GlDropShadowTextBrushBlend::GetBoundingRectangle(Rectangle& boundingRectangle) const
    {
        const Rectangle& shadowSize = m_dropShadowEffect.GetBorderSize();
        m_textBrush.GetBoundingRectangle(boundingRectangle);
        boundingRectangle.Set(boundingRectangle.GetLeft() + shadowSize.GetLeft(),
            boundingRectangle.GetTop() + shadowSize.GetTop(),
            boundingRectangle.GetWidth() + shadowSize.GetWidth(),
            boundingRectangle.GetHeight() + shadowSize.GetHeight());
    }

}   // namespace Candera
