//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GL_DROPSHADOW_EFFECT_H)
#define CANDERA_GL_DROPSHADOW_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/Properties/Property.h>
#include <Candera/Engine2D/Property/ColorProperty.h>

#define ENUM_DATA_TYPE \
    ENUM_DATA_TYPE_BEGIN(DropShadowBlurFilter)          \
        ENUM_DATA_TYPE_ITEM(HardShadow)                 \
        ENUM_DATA_TYPE_ITEM(SoftShadow3x3)              \
    ENUM_DATA_TYPE_END(DropShadowBlurFilter)

#include <Candera/System/MetaInfo/EnumDataType.h>

namespace Candera {

    /**
     * @brief InPlace effect that casts a blurred, offset shadow of the source.
     * The effect can be combined for now just with a TextBrush with BitmapCache.
     */
    class GlDropShadowEffect : public InPlaceEffect2D {
         FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(GlDropShadowEffect);
        CANDERA_RTTI_DECLARATION(Candera::GlDropShadowEffect);

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
        * Property for shadow enabled.
        * @return bool value that represents, if the shadow effect is enabled or not.
        */
        SimpleProperty<bool>& ShadowEnabled() { return m_shadowEnabled; }

        /**
         * Property for the shadow color.
         * @return Color of the casted shadow. Use alpha channel to set transparency.
         */
        ColorProperty& ShadowColor() { return m_shadowColor; }

        /**
         * Property for the light angle.
         * @return Angle of virtual light that creates the shadow effect.
         */
        SimpleProperty<Int>& LightAngle() { return m_angle; }

        /**
         * Property for the shadow distance.
         * @return Distance in pixels at which the shadow is casted at the given angle.
         */
        SimpleProperty<UInt>& Distance() { return m_distance; }

        /**
         * Property for the shadow scale.
         * @return Size of the shadow in pixels.
         */
        SimpleProperty<UInt>& ShadowScale() { return m_scale; }

        /**
         * Property for the blur filter.
         * @return Blur filter.
         */
        SimpleProperty<DropShadowBlurFilter>& BlurFilter() { return m_blurFilter; }

        /**
        * Property for the color factor.
        * @return color factor.
        */
        Int& ColorFactor() { return m_colorFactor; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const override;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::GlDropShadowEffect, GlDropShadowEffect, EFFECT2D_TYPE_INPLACE)

            CdaDescription("The effect casts a blurred, offset shadow of the source.")
            CdaEffect2DProperties()

                CdaEffect2DProperty(ShadowEnabled, bool, m_shadowEnabled)
                    CdaDescription("Enables or disables the shadow effect.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(ShadowColor, Color, m_shadowColor)
                    CdaDescription("Color of the casted shadow. Use alpha channel to set transparency.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(LightAngle, Int, m_angle)
                    CdaDescription("Angle of virtual light that creates the shadow effect.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(Distance, UInt, m_distance)
                    CdaDescription("Distance in pixels at which the shadow is casted at the given angle.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(ShadowScale, UInt, m_scale)
                    CdaDescription("Increases the shadow's target area in the respective direction by n pixels. " \
                    "For an optimal soft shadow effect, values from 1 to 3 are recommended.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(BlurFilter, DropShadowBlurFilter, m_blurFilter)
                    CdaDescription("Blur filter.")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

        /**
         *  Activate the parameters used for dropshadow.
         *  @param output The parameters are activated for this 2D context.
         */
         void ActivateFilter(ContextHandle2D output);

        /**
         *  Deactivate the parameters used for dropshadow.
         *  @param output The parameters are deactivated for this 2D context.
         */
        void DeactivateFilter(ContextHandle2D output) const;
        
        /**
         *  Retrieve a Rectangle containing the additional bounding border caused by the effect.
         *  @return Size of border.
         */
        Rectangle GetBorderSize() const;

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GlDropShadowEffect();
        explicit GlDropShadowEffect(const GlDropShadowEffect& rhs);
        virtual ~GlDropShadowEffect();
        GlDropShadowEffect& operator = (const GlDropShadowEffect& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea) override;

    private:
        SimpleProperty<bool> m_shadowEnabled;
        ColorProperty m_shadowColor;
        SimpleProperty<Int> m_angle;
        SimpleProperty<UInt> m_distance;
        SimpleProperty<UInt> m_scale;
        SimpleProperty<DropShadowBlurFilter> m_blurFilter;

        Int m_colorFactor;
    };

}   // namespace Candera

#endif  // CANDERA_GL_DROPSHADOW_EFFECT_H

