//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlDropShadowBitmapBrushColorBlend.h"
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/Engine2D/Core/TextNode2D.h>
#include <Candera/Engine2D/Core/Mesh2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    // FEATSTD_RTTI_DEFINITION(GlDropShadowBitmapBrushColorBlend, CombinedEffect2D)
    CANDERA_RTTI_DEFINITION(GlDropShadowBitmapBrushColorBlend)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GlDropShadowBitmapBrushColorBlend::GlDropShadowBitmapBrushColorBlend()
    {
    }

    GlDropShadowBitmapBrushColorBlend::GlDropShadowBitmapBrushColorBlend(const GlDropShadowBitmapBrushColorBlend& rhs) :
        Base(rhs),
        m_bitmapBrush(rhs.m_bitmapBrush),
        m_dropShadowEffect(rhs.m_dropShadowEffect),
        m_colorEffect(rhs.m_colorEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlDropShadowBitmapBrushColorBlend::~GlDropShadowBitmapBrushColorBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlDropShadowBitmapBrushColorBlend::SharedPointer GlDropShadowBitmapBrushColorBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlDropShadowBitmapBrushColorBlend* brush = FEATSTD_NEW(GlDropShadowBitmapBrushColorBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlDropShadowBitmapBrushColorBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
    *  GetInPlaceEffect2D
    ******************************************************************************/
    Candera::InPlaceEffect2D* GlDropShadowBitmapBrushColorBlend::GetInPlaceEffect2D(UInt8 index)
    {
        FEATSTD_DEBUG_ASSERT(index < 2);
        if (index == 0) {
            return &m_dropShadowEffect;
        }
        else {
            return &m_colorEffect;
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool GlDropShadowBitmapBrushColorBlend::Upload()
    {
        bool success = m_bitmapBrush.Upload();
        success = success && m_dropShadowEffect.Upload();
        success = success && m_colorEffect.Upload();
        success = success && m_blendEffect.Upload();
        return success;
    }

    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool GlDropShadowBitmapBrushColorBlend::Unload()
    {
        bool success = m_bitmapBrush.Unload();
        success = success && m_dropShadowEffect.Unload();
        success = success && m_colorEffect.Unload();
        success = success && m_blendEffect.Unload();
        return success;
    }

    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool GlDropShadowBitmapBrushColorBlend::Update()
    {
        bool success = m_bitmapBrush.Update();
        success = success && m_dropShadowEffect.Update();
        success = success && m_colorEffect.Update();
        success = success && m_blendEffect.Update();
        return success;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlDropShadowBitmapBrushColorBlend::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                                           const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "GlDropShadowBitmapBrushColorBlend"));

        Image2D* image = m_bitmapBrush.Image();

        if (image != 0) {
            m_blendEffect.ActivateBlend(output);

            if (Dynamic_Cast<const TextNode2D*>(&node) != 0) {
                m_dropShadowEffect.ColorFactor() = 1;
            }

            Color::Data& color = m_colorEffect.Color().Get();
            Float cameraAlpha = 1.0F;
            const Camera2D* camera2D = Renderer2D::GetActiveCamera();
            if (0 != camera2D) {
                if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                    cameraAlpha = camera2D->GetEffectiveAlphaValue();
                }
            }
            bool result = RenderDevice2D::SetSurfaceConstColor(
                output,
                RenderDevice2D::SourceSurface,
                color.red,
                color.green,
                color.blue,
                color.alpha * node.GetEffectiveAlphaValue() * cameraAlpha
                );
            if (result) {
                static_cast<void>(image->Activate(output));

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
                if ((m_bitmapBrush.IsNinePatch()) && (!node.IsTypeOf(Mesh2D::GetTypeId()))) {
                    m_bitmapBrush.ActivateNinePatch(transform, output, node);
                }
#endif

                Rectangle rect;
                m_bitmapBrush.GetBoundingRectangle(rect);

                m_dropShadowEffect.Render(image->Get2DSurfaceHandle(), rect, transform, node, output, outputArea);
                static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
                if ((m_bitmapBrush.IsNinePatch()) && (!node.IsTypeOf(Mesh2D::GetTypeId()))) {
                    m_bitmapBrush.DeactivateNinePatch();
                }
#endif
            }
            m_dropShadowEffect.ColorFactor() = 0;
            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlDropShadowBitmapBrushColorBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlDropShadowBitmapBrushColorBlend)(*this));
    }

    /******************************************************************************
    *  GetBoundingRectangle
    ******************************************************************************/
    void GlDropShadowBitmapBrushColorBlend::GetBoundingRectangle(Rectangle& boundingRectangle) const
    {
        const Rectangle& shadowSize = m_dropShadowEffect.GetBorderSize();
        m_bitmapBrush.GetBoundingRectangle(boundingRectangle);
        boundingRectangle.Set(boundingRectangle.GetLeft() + shadowSize.GetLeft(),
            boundingRectangle.GetTop() + shadowSize.GetTop(),
            boundingRectangle.GetWidth() + shadowSize.GetWidth(),
            boundingRectangle.GetHeight() + shadowSize.GetHeight());
    }

    /******************************************************************************
    *  GetLayoutingRectangle
    ******************************************************************************/
    void GlDropShadowBitmapBrushColorBlend::GetLayoutingRectangle(Rectangle& rectangle) const
    {
        const Rectangle& shadowSize = m_dropShadowEffect.GetBorderSize();
        m_bitmapBrush.GetLayoutingRectangle(rectangle);
        rectangle.Set(rectangle.GetLeft() + shadowSize.GetLeft(),
            rectangle.GetTop() + shadowSize.GetTop(),
            rectangle.GetWidth() + shadowSize.GetWidth(),
            rectangle.GetHeight() + shadowSize.GetHeight());
    }

}   // namespace Candera

