//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlDropShadowBitmapBrushBlend.h"
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/Engine2D/Core/Mesh2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(GlDropShadowBitmapBrushBlend, CombinedEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GlDropShadowBitmapBrushBlend::GlDropShadowBitmapBrushBlend()
    {
    }

    GlDropShadowBitmapBrushBlend::GlDropShadowBitmapBrushBlend(const GlDropShadowBitmapBrushBlend& rhs) :
        Base(rhs),
        m_bitmapBrush(rhs.m_bitmapBrush),
        m_dropShadowEffect(rhs.m_dropShadowEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlDropShadowBitmapBrushBlend::~GlDropShadowBitmapBrushBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlDropShadowBitmapBrushBlend::SharedPointer GlDropShadowBitmapBrushBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlDropShadowBitmapBrushBlend* brush = FEATSTD_NEW(GlDropShadowBitmapBrushBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlDropShadowBitmapBrushBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
    *  GetInPlaceEffect2D
    ******************************************************************************/
    Candera::InPlaceEffect2D* GlDropShadowBitmapBrushBlend::GetInPlaceEffect2D(UInt8 index)
    {
        FEATSTD_DEBUG_ASSERT(index < 1);
        return (index == 0) ? &m_dropShadowEffect : 0;
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool GlDropShadowBitmapBrushBlend::Upload()
    {
        bool success = m_bitmapBrush.Upload();
        success = success && m_dropShadowEffect.Upload();
        success = success && m_blendEffect.Upload();
        return success;
    }

    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool GlDropShadowBitmapBrushBlend::Unload()
    {
        bool success = m_bitmapBrush.Unload();
        success = success && m_dropShadowEffect.Unload();
        success = success && m_blendEffect.Unload();
        return success;
    }

    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool GlDropShadowBitmapBrushBlend::Update()
    {
        bool success = m_bitmapBrush.Update();
        success = success && m_dropShadowEffect.Update();
        success = success && m_blendEffect.Update();
        return success;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlDropShadowBitmapBrushBlend::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                                           const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "GlDropShadowBitmapBrushBlend"));

        Image2D* image = m_bitmapBrush.Image();

        if (image != 0) {
            m_blendEffect.ActivateBlend(output);

            static_cast<void>(image->Activate(output));

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
            if ((m_bitmapBrush.IsNinePatch()) && (!node.IsTypeOf(Mesh2D::GetTypeId()))) {
                m_bitmapBrush.ActivateNinePatch(transform, output, node);
            }
#endif

            Rectangle rect;
            m_bitmapBrush.GetBoundingRectangle(rect);

            m_dropShadowEffect.Render(image->Get2DSurfaceHandle(), rect, transform, node, output, outputArea);

#if defined(CANDERA_3D_ENABLED) && defined(CANDERA_2D_OVER_3D_ENABLED)
            if ((m_bitmapBrush.IsNinePatch()) && (!node.IsTypeOf(Mesh2D::GetTypeId()))) {
                m_bitmapBrush.DeactivateNinePatch();
            }
#endif

            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlDropShadowBitmapBrushBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlDropShadowBitmapBrushBlend)(*this));
    }

    /******************************************************************************
    *  GetBoundingRectangle
    ******************************************************************************/
    void GlDropShadowBitmapBrushBlend::GetBoundingRectangle(Rectangle& boundingRectangle) const
    {
        const Rectangle& shadowSize = m_dropShadowEffect.GetBorderSize();
        m_bitmapBrush.GetBoundingRectangle(boundingRectangle);
        boundingRectangle.Set(boundingRectangle.GetLeft() + shadowSize.GetLeft(),
            boundingRectangle.GetTop() + shadowSize.GetTop(),
            boundingRectangle.GetWidth() + shadowSize.GetWidth(),
            boundingRectangle.GetHeight() + shadowSize.GetHeight());
    }

    /******************************************************************************
    *  GetLayoutingRectangle
    ******************************************************************************/
    void GlDropShadowBitmapBrushBlend::GetLayoutingRectangle(Rectangle& rectangle) const
    {
        const Rectangle& shadowSize = m_dropShadowEffect.GetBorderSize();
        m_bitmapBrush.GetLayoutingRectangle(rectangle);
        rectangle.Set(rectangle.GetLeft() + shadowSize.GetLeft(),
            rectangle.GetTop() + shadowSize.GetTop(),
            rectangle.GetWidth() + shadowSize.GetWidth(),
            rectangle.GetHeight() + shadowSize.GetHeight());
    }

}   // namespace Candera

