//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CONTEXT_2D_OVER_3D_DEVICE_PROGRAM_H)
#define CONTEXT_2D_OVER_3D_DEVICE_PROGRAM_H

#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Rectangle.h>

#include <Candera/EngineBase/Common/Color.h>

#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>

#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/Engine3D/Core/Texture.h>
#include <Candera/Engine3D/Core/RenderMode.h>

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DState.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DTarget.h>


namespace Candera {

    namespace Internal {

        /**
         * This class handles a concrete shader used for 2D rendering.
         */
        class Context2DOver3DDeviceProgram {
        public:

            enum ProgramType{
                Start = 0,
                RGBABlit = Start,
                CCCABlit = 1,
                CCCCBlit = 2,
                StartMask = 3,
                RGBABlitMaskMul = StartMask,
                CCCABlitMaskMul = 4,
                CCCCBlitMaskMul = 5,
                StartTransformed = 6,
                RGBABlitTransformed = StartTransformed,
                CCCABlitTransformed = 7,
                CCCCBlitTransformed = 8,
                RGBABlitMaskMulTransformed = 9,
                CCCABlitMaskMulTransformed = 10,
                CCCCBlitMaskMulTransformed = 11,
                StartExtended = 12,
                CCCABlitLinearGradient = StartExtended,
                CCCABlitRadialGradient = 13,
                StartOutline=14,
                RGBABlitOutline = StartOutline,
                CCCABlitOutline= 15,
                CCCCBlitOutline = 16,
                StartDropShadow = 17,
                CCCABlitDropShadowNoBlur = StartDropShadow,
                CCCABlitDropShadow3x3Blur = 18,
                ProgramsCount = 19               ///< Number of blit program types.
            };

            /**
             * Constructs a Context2DOver3DDeviceProgram. Shader programs must not be changed any more, therefore onyl in ctor.
             * @param vertexShader Vertex shader source code.
             * @param pixelShader Pixel shader source code.
             */
            Context2DOver3DDeviceProgram(const void* vertexShader, const void* pixelShader, ProgramType programType, FeatStd::Internal::Guid const & guid);

            /**
             * Destructor unloads the attached shader before destruction.
             */
            ~Context2DOver3DDeviceProgram();

            /**
             * Uploads the shader (context handling is done by the shader object itself). Determines
             * the needed uniform locations for the shader in the current EGL context.
             * @return True if Upload succeeded, false if not.
             */
            bool Upload();


            /**
             * Unloads the shader (context handling is done by the shader object itself).
             * @return True if Unload succeeded, false if not.
             */
            bool Unload();

            /**
             * Computes the viewport needed for rendering.
             * @param context     Holds Node2D centric data needed for viewport calculation.
             * @param target      Holds RenderTarget2D centric data needed for viewport calculation.
             * @param correction  Out parameter: Viewport correction that has to be applied when drawing.
             * @return Whether viewport correction could be computed (true) or not (false).
             */
            static bool Viewport(const Context2DOver3DState &context, const Context2DOver3DTarget& target, Rectangle &correction);

            /**
             * Activates the scissor that has been set via RenderDevice2D
             * @param context  The context to activate the scissor for.
             * @param target   The render target associated with the context to activate the scissor for.
             * @return         True, if the scissor was successfully activated. False, if the operation failed or
             *                 scissor is not supported.
             */
            static bool ActivateScissor(const Context2DOver3DState& context, const Context2DOver3DTarget& target);

            /**
             * Draws a node.
             * @param context Holds Node2D centric data needed for rendering.
             * @param context Holds RenderTarget2D centric data needed for rendering.
             * @param vertexBuffer VertexBuffer that's used to render a quad.
             * @param updatedArea Out parameter: Area of render target that got updated.
             * @return Whether rendering could be done (true) or not (false).
             */
            bool Draw(const Context2DOver3DState &context, const Context2DOver3DTarget& target,
                MemoryManagement::SharedPointer<VertexBuffer> vertexBuffer, Rectangle &updatedArea);

        private:

            struct TextureShaderParamName {

                TextureShaderParamName() : m_contextShaderParamName(0), m_shaderParamName(0) {}

                TextureShaderParamName(const Char* contextShaderParamName) :
                    m_contextShaderParamName(contextShaderParamName),
                    m_shaderParamName(0) { }

                const Char* m_contextShaderParamName;
                const Char* m_shaderParamName;
            };

            enum TextureShaderParamNames {
                TextureName = 0,
                TextureName1,
                TextureShaderParamCount
            };

            TextureShaderParamName m_textureShaderParamNames[TextureShaderParamCount];

            bool ActivateShaderAndGeometry(const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer);

            bool ActivateMaskTransform(const Rectangle& activeSourceArea, const Rectangle& activeMaskArea, const Context2DOver3DState &state);
            bool ActivateTexture(Int textureLocation, UInt textureId, const MemoryManagement::SharedPointer<Texture>& texture);

            Rectangle GetTextureRectangle(const MemoryManagement::SharedPointer<Texture>& texture) const;
            bool ActivateTransform(Int transformLocation, const Rectangle& active, const Rectangle& correction, const Matrix3x2& transform, Rectangle& updated) const;
            bool ActivateTextureTransform(Int transformLocation, const Rectangle& size, const Rectangle& active, RenderDevice2D::PackOrder packOrder, bool flipH, bool flipV) const;

            bool ActivateColor(Int colorLocation, const Color& color) const;
            bool ActivateMatrix(Int matrixLocation, const Matrix3x2& matrix) const;
            bool ActivateMatrix4(Int matrixLocation, const Matrix4& matrix) const;
            bool ActivateVec2(Int location, const Vector2& vector) const;
            bool ActivateFloat(Int location, const Float& val) const;

            bool ActivateRenderMode(const MemoryManagement::SharedPointer<RenderMode>& renderMode) const;

            void DrawArrays(const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer) const;

            bool IsGradientShader() const { return (m_programType == CCCABlitLinearGradient) || (m_programType == CCCABlitRadialGradient); }

            bool IsOutlineShader() const { return ((m_programType == RGBABlitOutline) || (m_programType == CCCABlitOutline) || (m_programType == CCCCBlitOutline)); }

            bool IsDropShadowShader() const { return ((m_programType == CCCABlitDropShadowNoBlur) || (m_programType == CCCABlitDropShadow3x3Blur)); }

            const void* m_vertexShader;
            const void* m_pixelShader;

            MemoryManagement::SharedPointer<Shader> m_shader;

            struct UniformTypes
            {
                enum Enum
                {
                    Transform,
                    Texture,
                    TextureTransform,
                    TextureColor,
                    MaskTexture,
                    MaskTransform,
                    MaskColor,
                    ColorTransform,
                    GradientColor,
                    GradientDirection,
                    GradientMagnitude,
                    GradientCenter,
                    OutlineColor,
                    OutlinePixelDimension,
                    OutlineWidth,
                    DropShadowColor,
                    DropShadowOffset,
                    DropShadowScale,
                    DropShadowColorFactor,

                    _LastEntry
                };
            };

            static const Candera::Char* s_uniformNames[UniformTypes::_LastEntry];

            Int m_uniformLocation[UniformTypes::_LastEntry][CANDERA_MAX_CONTEXT_COUNT];

            ProgramType m_programType;

            Int& UniformLocation(Int uniformType);
        };

    }
}

#endif


