//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(CONTEXT_2D_OVER_3D_DEVICE_POOL_H)
#define CONTEXT_2D_OVER_3D_DEVICE_POOL_H

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DDeviceProgram.h>
#include <Candera/Engine3D/Core/DeviceObject.h>


namespace Candera {

    namespace Internal {


        /**
        * This class manages the access to OpenGL device objects used for 2D rendering.
        * It takes care of shaders and vertex buffers only being created once per
        * EGL context. Using this class in Context2DOver3D prevents OpenGL objects from being created for every
        * render target. As this should be the single point to handle OpenGL objects it is implemented as singleton.
        */
        class Context2DOver3DDevicePool : public DeviceObject {

            FEATSTD_TYPEDEF_BASE(DeviceObject);

        public:

            /**
             * Creates or gets the instance of this singleton class.
             * @return Context2DOver2DDevicePool instance.
             */
            static Context2DOver3DDevicePool& GetInstance();

            /**
             * Destructor. Destroys accomodated shader and vertex buffer objects.
             */
            virtual ~Context2DOver3DDevicePool();

            /**
             * Gets the requested shader program.
             * @param type The concrete shader program to request.
             * @return The requested program, or 0 if unavailabel.
             */
            Context2DOver3DDeviceProgram* GetProgram(Context2DOver3DDeviceProgram::ProgramType type) const;

            /**
             * Gets the vertex buffer of this Context2DOver3DDevicePool.
             * @return The vertex buffer of this Context2DOver3DDevicePool.
             */
            VertexBuffer::SharedPointer GetVertexBuffer() const { return (m_customVertexBuffer == 0) ? m_vertexBuffer : m_customVertexBuffer; }

            /**
             *  Set a custom vertex buffer to be used instead of the default quad.
             *  The vertex buffer's positions are expected to be normalized (i.e. within the range [0..1]) if not specified otherwise.
             *  The vertex buffer's vertex layout is expected to be Context2DOver3DDevicePool::Vertex.
             *  @param vertexBuffer  The vertex buffer to be used instead of the default quad.
             *  @param rectangle     The bounding rectangle of the vertex buffer in local screen in pixels.
             */
            void SetCustomVertexBuffer(const VertexBuffer::SharedPointer& vertexBuffer, const Rectangle& rectangle = Rectangle(), bool normalized = true) {
                m_customVertexBuffer = vertexBuffer;
                m_customVertexBufferBoundingRectangle = rectangle;
                m_isCustomVertexBufferNormalized = normalized;
            }

            /**
             *  Check if a custom vertex buffer is set.
             *  @return  True, if a custom vertex buffer is set. False, otherwise.
             */
            bool HasCustomVertexBuffer() const { return (m_customVertexBuffer != 0); }

            /**
             *  Get the bounding rectangle that was set together with a custom vertex buffer.
             *  @return  The bounding rectangle that was set together with a custom vertex buffer.
             */
            const Rectangle& GetCustomVertexBufferBoundingRectangle() const { return m_customVertexBufferBoundingRectangle; }

            /**
             * Returns whether the customized vertex buffer is normalized or not.
             * An unnormalized vertex buffer can occur if e.g. a VertexBuffer2D is used.
             * @return True if the customized vertex buffer is normalized or False if not.
             */
            bool IsCustomVertexBufferNormalized() const {  return m_isCustomVertexBufferNormalized; }

            /**
             *  The vertex layout that is used by the default quad. Custom vertex buffers must use the same layout.
             */
            struct Vertex {
                Float x;
                Float y;
                Float u;
                Float v;
            };

            /**
             *  The vertex format that is used by the default quad. Custom vertex buffers must use the same format.
             */
            static const VertexGeometry::VertexElementFormat VertexFormat[];

        protected:
            /**
            *  overridden from DeviceObject
            *  @return True if uploading of device pool succeeded. False otherwise.
            */
            virtual bool UploadInternal(LoadingHint loadingHint) override;
            /**
            *  overridden from DeviceObject
            *  @return True if unloading of device pool succeeded. False otherwise.
            */
            virtual bool UnloadInternal(LoadingHint loadingHint) override;

            // override DeviceObject::DisposeInternal
            virtual void DisposeInternal() override;

        private:

            /**
             * Constructs the Context2DOver3DDevicePool instance. Instantiates the needed device objects.
             */
            Context2DOver3DDevicePool();
            FEATSTD_MAKE_CLASS_UNCOPYABLE(Context2DOver3DDevicePool);

            Context2DOver3DDeviceProgram* m_programPool[Context2DOver3DDeviceProgram::ProgramsCount]; ///< shader programs used for effects.
            mutable bool m_programLoaded[Context2DOver3DDeviceProgram::ProgramsCount]; ///< flag whether shader program is uploaded.

            VertexBuffer::SharedPointer m_vertexBuffer; ///< Vertex buffer used for rendering content to the target.

            VertexBuffer::SharedPointer m_customVertexBuffer; ///< Custom vertex buffer used for rendering content to the target.

            Rectangle m_customVertexBufferBoundingRectangle;

            bool m_isCustomVertexBufferNormalized;
        };

    }

}

#endif
