//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Context2DOver3D.h"

#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/ShaderDefinitionsFor2DOver3D.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/SurfaceAllocator2DOver3D.h>

namespace Candera {
    namespace Internal {

Context2DOver3D::Context2DOver3D() :
    m_isInitialized(false),
    m_programType(Context2DOver3DDeviceProgram::Start),
    m_customGeometry(0)
{
}

Context2DOver3D::~Context2DOver3D()
{
    if (!Unload()) {
        CANDERA_DEVICE_LOG_WARN("Unload failed.");
    }
}

bool Context2DOver3D::Upload()
{
    if (m_isInitialized) {
        return true;
    }


    if(!Context2DOver3DDevicePool::GetInstance().Upload()) {
        return false;
    }

    m_isInitialized = true;
    return true;
}

bool Context2DOver3D::Unload()
{
    if (!m_isInitialized) {
        return true;
    }

    bool success = Context2DOver3DDevicePool::GetInstance().Unload();

    // release blit textures
    m_state.srcTexture->SetTextureImage(MemoryManagement::SharedPointer<TextureImage>(0));
    m_state.mskTexture->SetTextureImage(MemoryManagement::SharedPointer<TextureImage>(0));

    if (success) {
        m_isInitialized = false;
    }
    return success;
}

void Context2DOver3D::SetSize(Int width, Int height, Int actualWidth, Int actualHeight)
{
    m_target.width = static_cast<Float>(width);
    m_target.height = static_cast<Float>(height);
    m_target.actualHeight = static_cast<Float>(actualHeight);
    m_target.actualWidth = static_cast<Float>(actualWidth);
}

void Context2DOver3D::ChooseProgram(SurfaceHandle surfaceHandle)
{
    if (SurfaceAllocator2DOver3D::HasTextureImage(surfaceHandle)) {
        // check for outline programs
        if (m_state.outlineWidth != 0) {
            m_programType = Context2DOver3DDeviceProgram::RGBABlitOutline;
        }
        else if (m_state.enableShadow) {
            m_programType = Context2DOver3DDeviceProgram::StartDropShadow;
        }
        else  if (RenderDevice2D::GetPixelFormat(surfaceHandle) == Bitmap::AlphaUnsignedBytePixelFormat) {
            m_programType = Context2DOver3DDeviceProgram::CCCABlit;
        }
        else {
            m_programType = Context2DOver3DDeviceProgram::RGBABlit;
        }
    }
    else {
        if (!Math::FloatAlmostEqual(m_state.gradientDirection.GetLength(), 0.0F)) {
            m_programType = Context2DOver3DDeviceProgram::CCCABlitLinearGradient;
        }
        else if (!Math::FloatAlmostEqual(m_state.gradientMagnitude, 0.0F)) {
            m_programType = Context2DOver3DDeviceProgram::CCCABlitRadialGradient;
        }
        else {
            m_programType = Context2DOver3DDeviceProgram::CCCCBlit;
        }
    }
}

bool Context2DOver3D::Draw() {
    if (m_isInitialized) {
        Rectangle updatedArea;
        Context2DOver3DDeviceProgram::ProgramType programType = m_programType;

        if (m_state.outlineWidth > 0){
            ImageSource3D *source = m_state.srcTexture->GetTextureImage()->ToImageSource3D();
            if (0 != source){
                //get pixel height and width in texture coordinate system.

               if ((source->GetWidth() == 0) || (source->GetHeight() == 0)){
                   // nothing to render here
                   return false;
               }

               m_state.outlinePixelDimensions.SetX(1.0F / static_cast<Float>(source->GetWidth()));
               m_state.outlinePixelDimensions.SetY(1.0F / static_cast<Float>(source->GetHeight()));
               Float doubleOutlineWidth = static_cast<Float>(m_state.outlineWidth * 2);

                //translate the target area up and left half the amount we will scale it. this works because we are already in world coordinate system
                m_state.srcMatrix.Translate(-1.0F * static_cast<Float>(m_state.outlineWidth), -1.0F * static_cast<Float>(m_state.outlineWidth));
                //adjust src area if necessary
                if ((m_state.srcArea.GetHeight() < (static_cast<Float>(source->GetHeight()) + doubleOutlineWidth)) &&
                    (m_state.srcArea.GetWidth() < (static_cast<Float>(source->GetWidth()) + doubleOutlineWidth))) {
                    m_state.srcArea.SetHeight(static_cast<Float>(source->GetHeight()) + doubleOutlineWidth);
                    m_state.srcArea.SetWidth(static_cast<Float>(source->GetWidth()) + doubleOutlineWidth);
                }
                //scale the output area by two times the outline width to get enough space for the outline.
                m_state.srcMatrix.Scale(1.0F + (m_state.outlinePixelDimensions.GetX() * doubleOutlineWidth), 1.0F + (m_state.outlinePixelDimensions.GetY() * doubleOutlineWidth));
            }
         }
        else if (m_state.enableShadow) {
            programType = m_state.blurFilter ? Context2DOver3DDeviceProgram::CCCABlitDropShadow3x3Blur : Context2DOver3DDeviceProgram::CCCABlitDropShadowNoBlur;

            ImageSource3D *source = m_state.srcTexture->GetTextureImage()->ToImageSource3D();
            if (0 != source) {
                //get pixel height and width in texture coordinate system.

                if ((source->GetWidth() == 0) || (source->GetHeight() == 0)) {
                    // nothing to render here
                    return false;
                }

                m_state.outlinePixelDimensions.SetX(1.0F / static_cast<Float>(source->GetWidth()));
                m_state.outlinePixelDimensions.SetY(1.0F / static_cast<Float>(source->GetHeight()));
                Float radians = Math::DegreeToRadian(180.0F - static_cast<Float>(m_state.angle));
                m_state.offset.SetX(Math::Cosine(radians) * static_cast<Float>(m_state.distance));
                m_state.offset.SetY(Math::Sine(radians) * static_cast<Float>(m_state.distance));

                Float left = Math::Minimum(m_state.offset.GetX() - static_cast<Float>(m_state.scale), 0.0F);
                Float top = Math::Minimum(m_state.offset.GetY() - static_cast<Float>(m_state.scale), 0.0F);

                Float right = Math::Maximum(0.0F, m_state.offset.GetX() + static_cast<Float>(m_state.scale));
                Float bottom = Math::Maximum(0.0F, m_state.offset.GetY() + static_cast<Float>(m_state.scale));

                //translate the target area up and left with the negative offset + 1 scaled pixel.
                m_state.srcMatrix.Translate(left, top);
                //adjust src area if necessary
                if ((m_state.srcArea.GetHeight() < (static_cast<Float>(source->GetHeight()) + bottom - top))) {
                    m_state.srcArea.SetHeight(static_cast<Float>(source->GetHeight()) + bottom - top);
                }
                if (m_state.srcArea.GetWidth() < (static_cast<Float>(source->GetWidth()) + right - left)) {
                    m_state.srcArea.SetWidth(static_cast<Float>(source->GetWidth()) + right - left);
                }
                //scale the output area by two times the outline width to get enough space for the outline.
                m_state.srcMatrix.Scale(1.0F + (m_state.outlinePixelDimensions.GetX() * (-left + right)), 1.0F + (m_state.outlinePixelDimensions.GetY() * (-top + bottom)));
            }
        }
        else {
            if (m_state.mskTexture->GetTextureImage() != 0) {
                programType = static_cast<Context2DOver3DDeviceProgram::ProgramType>(programType + Context2DOver3DDeviceProgram::StartMask);
            }
            if (m_state.srcColorMatrixEnabled) {
                programType = static_cast<Context2DOver3DDeviceProgram::ProgramType>(programType + Context2DOver3DDeviceProgram::StartTransformed);
            }
        }

        Context2DOver3DDeviceProgram* program = Context2DOver3DDevicePool::GetInstance().GetProgram(programType);
        bool success = false;

        if (program != 0) {
            success = program->Draw(m_state, m_target, Context2DOver3DDevicePool::GetInstance().GetVertexBuffer(), updatedArea);
        }
        m_state.lastUpdatedArea = updatedArea;
        return success;
    }
    return false;
}

bool Context2DOver3D::Clear(const Color& color) {
    // Get the actual dimensions - resolves negative dimensions.
    Float width = m_state.dstArea.GetWidth() < 0.0F ? m_target.width : m_state.dstArea.GetWidth();
    Float height = m_state.dstArea.GetHeight() < 0.0F ? m_target.height : m_state.dstArea.GetHeight();

    m_state.lastUpdatedArea = Rectangle(
        0.0F,
        0.0F,
        width,
        height);

    // Can we clear the entire color buffer, or do we need to blit only a specific area?
    bool can_clear =
        (m_state.dstArea.GetLeft() <= 0.0F) &&
        (m_state.dstArea.GetTop() <= 0.0F) &&
        ((width + m_state.dstArea.GetLeft()) >= m_target.width) &&
        ((height + m_state.dstArea.GetTop()) >= m_target.height);

    bool result = true;
    if (can_clear) {
        if (!Context2DOver3DDeviceProgram::ActivateScissor(m_stateForClearing, m_target)) {
            return false;
        }

        result = RenderDevice::ClearColorBuffer(color);
    }
    else {
        m_stateForClearing.srcConstColor = color;
        m_stateForClearing.srcArea.Set(0.0F, 0.0F, m_target.width, m_target.height);
        m_stateForClearing.dstArea = m_state.dstArea;

        Rectangle updatedArea;

        Context2DOver3DDeviceProgram* program = Context2DOver3DDevicePool::GetInstance().GetProgram(Context2DOver3DDeviceProgram::CCCCBlit);
        if (program != 0) {
            result = program->Draw(m_stateForClearing, m_target, Context2DOver3DDevicePool::GetInstance().GetVertexBuffer(), updatedArea);
        }
        else {
            result = false;
        }

    }

    return result;
}

}}
