//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GRAPHIC_DEVICE_UNIT_CONFIGURATION_H)
#define GRAPHIC_DEVICE_UNIT_CONFIGURATION_H

#include <Candera/Environment.h>

#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnitAttributes.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

    /**
     * @brief Template class for parsing and querying the GraphicDeviceUnit configuration attributes.
     *
     * GraphicDeviceUnitConfiguration parses an array of attribute and value pairs and selects relevant attribute values for later retrieval.
     *  The relevant attributes are selected by the template parameters:
     * @param FirstAttribute The first relevant attribute, for example GduRedSizeAttr for a ColorFormat configuration.
     * @param AttributeCount The number of relevant attributes, for example 4 for a ColorFormat configuration.
     * @param DefaultValue The default value that will be set if some attributes are not specified in the configuration.
     *
     * An attribute is relevant if attribute >= FirstAttribute and attribute < FirstAttribute + AttributeCount.
     */
    template <Int FirstAttribute, Int AttributeCount = 1, Int DefaultValue = -1>
    class GraphicDeviceUnitConfiguration {
    public:
        /**
         * Constructor that receives the configuration as parameter.
         *
         * The constructor will parse the given configuration and select relevant attributes.
         * @param configuration The configuration that is set.
         */
        GraphicDeviceUnitConfiguration(const Int* configuration);

        /**
         * Check if the configuration contained any relevant attributes.
         * @return True if the configuration contained any relevant attributes, otherwise false.
         */
        bool IsDefault() const { return m_isDefault; }

        /**
         * Retrieve the value attached to a given attribute.
         * @param attribute The queried attribute.
         * @return
         *  - 0 if the attribute is not relevant for this class specification, or
         *  - DefaultValue if the queried attribute is relevant but not specified in the configuration list, or
         *  - attribute value if attribute is relevant and it is found in the configuration list.
         */
        template<typename T>
        Int operator [] (T attribute) const { return IsAttributeRelevant(static_cast<Int>(attribute)) ? m_values[static_cast<Int>(attribute) - FirstAttribute] : 0; }

    protected:
        bool m_isDefault;
        Int m_values[AttributeCount];

        bool IsAttributeRelevant(Int attribute) const { return (attribute >= FirstAttribute) && (attribute < (FirstAttribute + AttributeCount)); }
    };

    template <Int FirstAttribute, Int AttributeCount, Int DefaultValue>
    GraphicDeviceUnitConfiguration<FirstAttribute, AttributeCount, DefaultValue>::GraphicDeviceUnitConfiguration(const Int* configuration)
        :m_isDefault(true)
    {
        for (Int index = 0; index < AttributeCount; ++index) {
            m_values[index] = DefaultValue;
        }

        if (configuration != 0) {
            while (*configuration != 0) {
                if (IsAttributeRelevant(*configuration)) {
                    m_values[*(configuration) - FirstAttribute] = *(configuration + 1);
                    m_isDefault = false;
                }
                configuration += 2;
            }
        }
    }

    /**
     * GraphicDeviceUnitConfiguration extension for color configuration.
     */
    class GraphicDeviceUnitColorFormatConfiguration: public GraphicDeviceUnitConfiguration<GduColorFormatBegin, GduColorFormatEnd - GduColorFormatBegin>
    {
        typedef GraphicDeviceUnitConfiguration<GduColorFormatBegin, GduColorFormatEnd - GduColorFormatBegin> Base;

    public:
        /**
         * Constructor that receives the configuration as parameter.
         * The constructor will parse the given configuration and select relevant attributes.
         * @param configuration The configuration that is set.
         */
        GraphicDeviceUnitColorFormatConfiguration(const Int* configuration) : Base(configuration) {}

        /**
         * Check if given color format is valid for this configuration (each channel size is greater than or equal to the configured one).
         * @param redSize   The size of Red.
         * @param greenSize The size of Green.
         * @param blueSize  The size of Blue.
         * @param alphaSize The size of Alpha.
         * @return True if color format is valid, otherwise false.
         */
        bool IsColorFormatValid(Int redSize, Int greenSize, Int blueSize, Int alphaSize) const;
    };
    typedef GraphicDeviceUnitConfiguration<GduWidthAttr> GraphicDeviceUnitWidthConfiguration;
    typedef GraphicDeviceUnitConfiguration<GduHeightAttr> GraphicDeviceUnitHeightConfiguration;

/** @}*/ //end of CommonDevice

}

#endif // GRAPHIC_DEVICE_UNIT_CONFIGURATION_H
