//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(GL_WARPING_PROGRAM_H)
#define GL_WARPING_PROGRAM_H

#include <Candera/Environment.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlInclude.h>
#include <CanderaPlatform/Device/Common/Base/DisplayOrientation.h>

#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/Core/RenderMode.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GlWarpingProgram is a class that sets up a shader used in screen space rendering.
 */
    class GlWarpingProgram
    {
        public:
            /**
             * Constructor
             */
            GlWarpingProgram();

            /**
             * Destructor
             */
            ~GlWarpingProgram();

            /**
             * Upload the shader.
             * @param positionAttributeCount Number of position attributes.
             * @return True if the upload was successful. False otherwise.
             */
            bool Upload(Int positionAttributeCount);

            /**
             * Unload the shader.
             */
            void Unload();

            /**
             * Activate the shader an other Screen space related Render States.
             */
            void Activate() const;

            /**
             * Check whether the shader has been uploaded.
             * @return True if the shader has been successfully uploaded.
             */
            bool IsUploaded() const { return m_shader->IsUploaded(); }

            /**
             * Specifies the orientation of the display.
             * @param orientation The orientation of the display.
             */
            void SetDisplayOrientation(DisplayOrientation::Enum orientation) { m_displayOrientation = orientation; }

            /**
             * Retrieves the orientation of the display.
             * @Return The orientation of the display.
             */
            DisplayOrientation::Enum GetDisplayOrientation() const { return m_displayOrientation; }

            /**
            * Retrieves the generated shader.
            * @Return The generated shader
            */
            const MemoryManagement::SharedPointer<Shader> GetShader() const { return m_shader; }

            /**
            * Sets the warp weight uniform
            * @param warpWeight Pointer to warpWeight data
            * @param warpWeightCount Number of warp weights
            */
            void UpdateWarpWeightUniform(const Float* warpWeight, Int warpWeightCount) const;

        private:

            MemoryManagement::SharedPointer<Shader> m_shader;
            MemoryManagement::SharedPointer<RenderMode> m_renderMode;

            DisplayOrientation::Enum m_displayOrientation;

            GLint m_textureUniformLocation;
            GLint m_invertUniformLocation;
            GLint m_warpWeightUniformLocation;

            /**
             * Updates the 'invert' uniform according to the currently set display orientation.
             */
            void UpdateInvertUniform() const;
    };

/** @}*/ //end of CommonDevice

}

#endif // !defined(GL_WARPING_PROGRAM_H)
