//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlWarpingMesh.h"

#include <CanderaPlatform/Device/Common/OpenGLES/GlTrace.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Diagnostics/Log.h>

#include <FeatStd/Util/StaticObject.h>


namespace Candera
{
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaPlatformDevice);

struct GlWarpingMeshElementData {
    Float x;
    Float y;
};

GlWarpingMesh::GlWarpingMesh() :
    m_vertexBuffer(0)
{
}

GlWarpingMesh::~GlWarpingMesh()
{
    Unload();
}

bool GlWarpingMesh::AreWarpMatricesCompatible(const WarpMatrix* src, UInt count, Int width, Int height) const
{
    for (UInt i = 0; i < count; ++i) {
        if (0 != src[i].GetGenericData()) {
            if (src[i].GetWidth() != width) {
                return false;
            }
            if (src[i].GetHeight() != height) {
                return false;
            }
        }
    }
    return true;
}


bool GlWarpingMesh::Upload(const WarpMatrix* src, UInt matrixCount, const Rectangle& imageBounds)
{
    if (IsUploaded()) {
        CANDERA_DEVICE_LOG_WARN("already uploaded");
        return false;
    }

    UInt actualMatrixCount = 0;
    Int width = 0;
    Int height = 0;

    for (UInt i = 0; i < matrixCount; i++) {
        if (0 != src[i].GetGenericData()) {
            actualMatrixCount++;
        }
    }

    for (UInt i = 0; i < matrixCount; ++i) {
        if (0 != src[i].GetGenericData()) {
            width = src[i].GetWidth();
            height = src[i].GetHeight();
            break;
        }
    }

    // check warp matrices compatibility
    if (actualMatrixCount > 1) {
        if (!AreWarpMatricesCompatible(&src[0], matrixCount, width, height)) {
            FEATSTD_LOG_ERROR("Failed to create warping mesh spliced vertex buffer, warp matrices are incompatible.");
            return false;
        }
    }

    UInt elementCount = actualMatrixCount + 1;
    VertexGeometry::VertexElementFormat* const elementFormat = FEATSTD_NEW_ARRAY(VertexGeometry::VertexElementFormat, static_cast<UInt32>(elementCount));
    if (elementFormat == 0) {
        FEATSTD_LOG_ERROR("Warping mesh create spliced vertex buffer failed, element format allocation failed.");
        FEATSTD_DELETE_ARRAY(elementFormat);
        return false;
    }

    UInt32 offset = 2 * sizeof(Float);
    for (UInt index = 0; index < actualMatrixCount; ++index) {
        elementFormat[index].Offset = static_cast<UInt16>(index * offset);
        elementFormat[index].Type = VertexGeometry::Float32_2;
        elementFormat[index].Usage = VertexGeometry::Position;
        elementFormat[index].UsageIndex = static_cast<UInt8>(index);
    }

    elementFormat[actualMatrixCount].Offset = static_cast<UInt16>(actualMatrixCount * offset);
    elementFormat[actualMatrixCount].Type = VertexGeometry::Float32_2;
    elementFormat[actualMatrixCount].Usage = VertexGeometry::TextureCoordinate;
    elementFormat[actualMatrixCount].UsageIndex = 0;

    Int drawCount = 2 * width * (height - 1);

    GlWarpingMeshElementData *data = FEATSTD_NEW_ARRAY(GlWarpingMeshElementData, drawCount * elementCount);
    if (data == 0) {
        FEATSTD_LOG_ERROR("Failed to allocate data for GlWarpingMeshElementData");
        FEATSTD_DELETE_ARRAY(data);
        FEATSTD_DELETE_ARRAY(elementFormat);
        return false;
    }
    GlWarpingMeshElementData* element = data;

    Float xTextureStepWidth = imageBounds.GetWidth() / static_cast<Float>(width - 1);
    Float yTextureStepWidth = imageBounds.GetHeight() / static_cast<Float>(height - 1);
    for (Int y = 0; y < (height - 1); y++) {
        for (Int i = 0; i < width; i++) {
            Int x = ((static_cast<UInt>(y)& 1U) != 0) ? ((width - 1) - i) : i;

            for (UInt count = 0; count < matrixCount; ++count) {
                if (0 != src[count].GetGenericData()) {
                    element->x = src[count].GetVertex(x, y).GetX();
                    element->y = src[count].GetVertex(x, y).GetY();
                    (element + elementCount)->x = src[count].GetVertex(x, y + 1).GetX();
                    (element + elementCount)->y = src[count].GetVertex(x, y + 1).GetY();
                    ++element;
                }
            }
            element->x = imageBounds.GetPosition().GetX() + static_cast<Float>(x)* xTextureStepWidth;
            element->y = imageBounds.GetPosition().GetY() + static_cast<Float>(y)* yTextureStepWidth;
            (element + elementCount)->x = imageBounds.GetPosition().GetX() + static_cast<Float>(x)* xTextureStepWidth;
            (element + elementCount)->y = imageBounds.GetPosition().GetY() + static_cast<Float>(y + 1) * yTextureStepWidth;
            element += elementCount + 1;
        }
    }

   VertexGeometry* vertexGeometry = FEATSTD_NEW(VertexGeometry)(
        data, MemoryManagement::AdaptedArrayDisposer<const void*, const GlWarpingMeshElementData*>::Dispose,
        elementFormat, VertexGeometry::VertexElementFormatDisposer::Dispose,
        0, 0,
        drawCount, static_cast<UInt16>(elementCount * sizeof(GlWarpingMeshElementData)), static_cast<UInt16>(elementCount), 0,
        VertexGeometry::VideoMemory,
        VertexGeometry::ArrayBuffer,
        VertexGeometry::StaticWrite);

    m_vertexBuffer = VertexBuffer::Create();
    if (m_vertexBuffer == 0) {
        FEATSTD_LOG_ERROR("Failed to to create vertex buffer for GlWarpingMesh");
        FEATSTD_DELETE(vertexGeometry);
        return false;
    }

    if (!m_vertexBuffer->SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose))
    {
        FEATSTD_LOG_ERROR("Failed to associate vertex geometry for GlWarpingMesh");
        FEATSTD_DELETE(vertexGeometry);
        return false;
    }

    m_vertexBuffer->SetPrimitiveType(VertexBuffer::TriangleStrip);

    if (!m_vertexBuffer->Upload()) {
        FEATSTD_LOG_ERROR("Upload vertex buffer failed for GlWarpingMesh.");
        return false;
    }

    return true;
}

void GlWarpingMesh::Unload()
{
    if (m_vertexBuffer != 0) {
        if (!m_vertexBuffer->Unload()) {
            FEATSTD_LOG_ERROR("Unload vertex buffer failed for GlWarpingMesh.");
        }
        m_vertexBuffer = SharedPointer<VertexBuffer>(0);
    }
}

void GlWarpingMesh::Activate(const MemoryManagement::SharedPointer<Shader>& shader) const
{
    if (m_vertexBuffer != 0) {
        static_cast<void>(m_vertexBuffer->Activate());

        if (!shader.PointsToNull()) {
            if (!shader->BindAttributes(m_vertexBuffer)) {
                FEATSTD_LOG_ERROR("Failed to bind attributes for GlWarpingMesh.");
            }
        }
    }
}

void GlWarpingMesh::Draw() const
{
    if (m_vertexBuffer != 0) {
        m_vertexBuffer->Render();
    }
}
}
