//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GL_TEXTURE_FRAME_BUFFER_H)
#define GL_TEXTURE_FRAME_BUFFER_H

#include <CanderaPlatform/Device/Common/OpenGLES/GlInclude.h>

#include <CanderaPlatform/Device/Common/Internal/GL/GlFrameBufferTypes.h>
#include <CanderaPlatform/Device/Common/Internal/GL/GlFrameBufferProperties.h>

#include <CanderaPlatform/Device/Common/Internal/GDU/GduSupport.h>

#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

class EglContext;

/**
 * @brief GlTextureFrameBuffer represents a frame buffer object with a texture attached as Color.
 *
 * This object may be used as a frame buffer in a GduBaseRenderTarget, with a 2D and/or 3D attachment.
 */
class GlTextureFrameBuffer : public FeatStd::Diagnostics::Measurable
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GlTextureFrameBuffer::Unload)

    public:
        static const Int c_maxBufferCount = 1;

        /**
         *  Constructs a GlTextureFrameBuffer object.
         */
        GlTextureFrameBuffer();

        /**
         *  Destructs a GlTextureFrameBuffer object.
         */
        ~GlTextureFrameBuffer();

        /**
         *  Activates the frame buffer object. Required by GduBaseRenderTarget.
         *  @param contextProvider Object providing the Egl context in which this frame buffer resides.
         *  @return True, if no errors occurred, false otherwise.
         */
        bool Activate(EglContext& contextProvider);

        /**
         *  Returns current height of the surface. Required by GduBaseRenderTarget.
         *  @return The current height of the window surface.
         */
        Int GetHeight() const;

        /**
         *  Returns current width of the surface. Required by GduBaseRenderTarget.
         *  @return The current width of the window surface.
         */
        Int GetWidth() const;

        /**
         *  Uploads this object to video memory. Required by GduBaseRenderTarget.
         *  @param contextProvider  The associated context provider.
         *  @param support          Unused.
         *  @param properties       Properties to which this frame buffer should be set.
         *  @return True if successful, false otherwise.
         */
        bool Upload(EglContext& contextProvider, const GduSupport& support, const GlFrameBufferProperties& properties);

        /**
         *  Unloads this object from video memory. Required by GduBaseRenderTarget.
         *  @param contextProvider Object providing the Egl context in which this frame buffer resides.
         */
        void Unload(EglContext& contextProvider);

        /**
         *  Does nothing. Required by GduBaseRenderTarget.
         *  @param contextProvider  Unused.
         *  @param properties       Unused.
         */
        void ApplyChanges(EglContext& contextProvider, const GlFrameBufferProperties& properties) const;

        /**
         *  If multisampling is enabled the main (multisampled) framebuffer is resolved using a blit operation into the
         *  second (target) framebuffer.
         *
         *  If multisampling is disabled, this function does nothing. Required by GduBaseRenderTarget.
         *  @param contextProvider  Unused.
         */
        void SwapBuffers(EglContext& contextProvider) const;

        /**
         *  Returns the image handle associated with the given index. Required by GenericImageSource.
         *  @param bufferId         Must be 0.
         *  @param imageId          Channel of the associated image. 0 = color, 1 = depth.
         *  @return Image handle associated with this object.
         */
        Handle GetImageHandle(Int bufferId, Int imageId) const;

        /**
         *  Returns the buffer that should be used as image source.
         *  @return 0.
         */
        Int GetPreviousBufferIndex() const { return 0; }

        /**
         *  Allow use as 2D frame buffer.
         *  @return Same as GetHeight.
         */
        Int GetHeight2D() const { return GetHeight(); }

        /**
         *  Allow use as 2D frame buffer.
         *  @return Same as GetWidth.
         */
        Int GetWidth2D() const { return GetWidth(); }

        /**
         *  Allow use as 2D frame buffer.
         *  @param contextProvider Forwarded to Activate.
         *  @return Same as Activate.
         */
        bool Activate2D(EglContext& contextProvider) { return Activate(contextProvider); }

        /**
         *  Initialize drawing to this frame buffer.
         *  @param contextProvider Object providing the Egl context in which this frame buffer resides.
         */
        void BeginDraw(EglContext& contextProvider) { Activate(contextProvider); }

        /**
         *  Ends drawing to this frame buffer.
         *  @param contextProvider Unused.
         */
        void EndDraw(EglContext& /*contextProvider*/) {}

        /**
         *  Initialize 2D drawing to this frame buffer.
         *  @param contextProvider Object providing the Egl context in which this frame buffer resides.
         */
        void BeginDraw2D(EglContext& contextProvider) { Activate2D(contextProvider); }

        /**
         *  Ends 2D drawing to this frame buffer.
         *  @param contextProvider Unused.
         */
        void EndDraw2D(EglContext& /*contextProvider*/) {}

         /**
          *  Overridden function of Measurable.
          *  Note: This method only gives the correct result after uploading.
          *  @return estimate of the video memory used by the OpenGL frame buffer (in bytes).
          */
        virtual UInt GetSize() const;

    private:
#ifdef CGIDEVICE_OPENGLES_30
        bool UploadMsaaColorAttachment();
        bool UploadMsaaDepthAttachment();
        bool UploadMsaaStencilAttachment();
#endif

        bool UploadColorAttachment();
        bool UploadDepthAttachment();
        bool UploadStencilAttachment();

        /**
         *  Unloads the specified renderBuffer attachment and sets the id to 0. A check for 0 is made.
         */
        void UnloadRenderBufferAttachment(GLuint& renderBuffer);

        /**
        *  Unloads the specified texture attachment and sets the id to 0. A check for 0 is made.
        */
        void UnloadTextureAttachment(GLuint& texture);


#ifdef CGIDEVICE_OPENGLES_30
        GLuint m_msaaColorRenderBuffer;
        GLuint m_msaaDepthRenderBuffer;
        GLuint m_msaaStencilRenderBuffer;
#endif

        GLuint m_colorTexture;
        GLuint m_depthRenderBuffer;
        GLuint m_stencilRenderBuffer;

        /**
        *  @brief Frame buffer object handles.
        *
        *  For offscreen render targets without multisampling the first framebuffer object is rendered to. The second framebuffer will not be generated.
        *
        *  For multisampled offscreen render targets, a second frame buffer object is required.
        *  The first framebuffer object then serves as an intermediate framebuffer with multisample attachments.
        *  The second frame buffer object will hold the resolved framebuffer.
        */
        GLuint m_frameBuffer[2];

        GlFrameBufferProperties m_properties;

        static const GLenum c_allBitsSet = 0xffffffff;
};

/** @}*/ //end of CommonDevice

}

#endif
