//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <CanderaPlatform/Device/Common/Internal/GL/GlTextureFrameBuffer.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglContext.h>

#include <CanderaPlatform/Device/Common/OpenGLES/GlTrace.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>

#include <CanderaPlatform/Device/Common/Base/ImageSource3D.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/OpenGLES20/GlTypeMapper.h>

namespace Candera
{

    GlTextureFrameBuffer::GlTextureFrameBuffer() :
#ifdef CGIDEVICE_OPENGLES_30
        m_msaaColorRenderBuffer(0),
        m_msaaDepthRenderBuffer(0),
        m_msaaStencilRenderBuffer(0),
#endif
        m_colorTexture(0),
        m_depthRenderBuffer(0),
        m_stencilRenderBuffer(0)
{
    for (UInt32 i = 0; i < (sizeof(m_frameBuffer) / sizeof(GLuint)); ++i)
    {
        m_frameBuffer[i] = 0;
    }
}

GlTextureFrameBuffer::~GlTextureFrameBuffer()
{
}

bool GlTextureFrameBuffer::Activate(EglContext& /*contextProvider*/)
{
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::GlTextureFrameBuffer::Activate, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects (OpenGL) and is therefore considered non-const")
    if (m_frameBuffer[0] != 0) {
        // We always render to the first (main) frame buffer.
        glBindFramebuffer(GL_FRAMEBUFFER, m_frameBuffer[0]);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindFrameBuffer");
        return !CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Gles)::WasLastErrorSet();
    }

    return false;
}

Int GlTextureFrameBuffer::GetHeight() const
{
    // CGI1-16247: actual height (not msaa) required
    return m_properties.GetHeight();
}

Int GlTextureFrameBuffer::GetWidth() const
{
    // CGI1-16247: actual width (not msaa) required
    return m_properties.GetWidth();
}

bool GlTextureFrameBuffer::Upload(EglContext& contextProvider, const GduSupport& /*support*/, const GlFrameBufferProperties& properties)
{
    if (m_frameBuffer[0] != 0) {
        return false;
    }

    m_properties = properties;

    bool success = contextProvider.ActivateContext();
    if (success) {
#ifdef CGIDEVICE_OPENGLES_30
        bool msaaEnabled = m_properties.GetMsaaSamples() > 1;
        if (msaaEnabled) {
            // For MSAA we need two frame buffer objects.
            // The first (main) frame buffer object will be the multisample framebuffer to render to.
            // The second frame buffer object will hold the resolved framebuffer - the target texture.
            glGenFramebuffers(2, m_frameBuffer);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenFramebuffers");
            glBindFramebuffer(GL_FRAMEBUFFER, m_frameBuffer[1]);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindFramebuffer");
        }
        else
#endif
        {
            glGenFramebuffers(1, m_frameBuffer);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenFramebuffers");
            glBindFramebuffer(GL_FRAMEBUFFER, m_frameBuffer[0]);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindFramebuffer");
        }

        success = UploadColorAttachment();
        success = success && UploadDepthAttachment();
        success = success && UploadStencilAttachment();

        if (success) {
            GLenum error = glCheckFramebufferStatus(GL_FRAMEBUFFER);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glCheckFramebufferStatus");

            success = (GL_FRAMEBUFFER_COMPLETE == error);

            if (!success) {
                // make sure the image is mipmap complete...
                glBindTexture(GL_TEXTURE_2D, m_colorTexture);
                glGenerateMipmap(GL_TEXTURE_2D);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenerateMipmap");

                error = glCheckFramebufferStatus(GL_FRAMEBUFFER);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glCheckFramebufferStatus");

                success = (GL_FRAMEBUFFER_COMPLETE == error);
            }
        }

#ifdef CGIDEVICE_OPENGLES_30
        if (msaaEnabled) {
            // MSAA is enabled, attach multisample attachments to main framebuffer object.

            glBindFramebuffer(GL_FRAMEBUFFER, m_frameBuffer[0]);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindFramebuffer");

            success = UploadMsaaColorAttachment();
            success = success && UploadMsaaDepthAttachment();
            success = success && UploadMsaaStencilAttachment();

            if (success) {
                GLenum error = glCheckFramebufferStatus(GL_FRAMEBUFFER);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glCheckFramebufferStatus");
                success = (GL_FRAMEBUFFER_COMPLETE == error);
            }
        }
#endif

        if (!success) {
            Unload(contextProvider);
        }
        RenderDevice::ResyncCurrentRenderStateCache(RenderDevice::ResyncActiveTextureImagesOption);

    }

    return success;
}

void GlTextureFrameBuffer::ApplyChanges(EglContext& /*contextProvider*/, const GlFrameBufferProperties& /*properties*/) const
{
}

void GlTextureFrameBuffer::SwapBuffers(EglContext& /*contextProvider*/) const
{
#ifdef CGIDEVICE_OPENGLES_30
    if ((m_properties.GetMsaaSamples() > 1) && (m_frameBuffer[0] != 0) && (m_frameBuffer[1] != 0)) {
        // Need blitting operation to resolve multisampling from render buffer to target texture.

        glBindFramebuffer(GL_READ_FRAMEBUFFER, m_frameBuffer[0]);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindFrameBuffer");
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER, m_frameBuffer[1]);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindFrameBuffer");

        glReadBuffer(GL_COLOR_ATTACHMENT0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glReadBuffer");

        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1960, OpenGL flags are not unsigned);
        glBlitFramebuffer(
            0, 0, m_properties.GetWidth(), m_properties.GetHeight(),
            0, 0, m_properties.GetWidth(), m_properties.GetHeight(),
            GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT, GL_NEAREST); //GLES3 Specification, pg 198: If a buffer is specified in mask and does not exist in both the read and draw framebuffers, the corresponding bit is silently ignored.
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBlitFramebuffer");
    }
#endif
}

void GlTextureFrameBuffer::Unload(EglContext& contextProvider)
{
    if (contextProvider.ActivateContext()){
        UnloadTextureAttachment(m_colorTexture);
        UnloadRenderBufferAttachment(m_depthRenderBuffer);
        UnloadRenderBufferAttachment(m_stencilRenderBuffer);

        if (m_frameBuffer[0] != 0) {
            glDeleteFramebuffers(1, &m_frameBuffer[0]);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glDeleteFramebuffers");
            m_frameBuffer[0] = 0;
        }
#ifdef CGIDEVICE_OPENGLES_30
        UnloadRenderBufferAttachment(m_msaaColorRenderBuffer);
        UnloadRenderBufferAttachment(m_msaaDepthRenderBuffer);
        UnloadRenderBufferAttachment(m_msaaStencilRenderBuffer);

        if (m_frameBuffer[1] != 0) {
            glDeleteFramebuffers(1, &m_frameBuffer[1]);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glDeleteFramebuffers");
            m_frameBuffer[1] = 0;
        }
#endif

        RenderDevice::ResyncCurrentRenderStateCache(RenderDevice::ResyncActiveTextureImagesOption);
    }
}

Handle GlTextureFrameBuffer::GetImageHandle(Int bufferId, Int imageId) const
{
    Handle ret = 0;
    if (bufferId == 0) {
        ImageSource3D* externalColor = m_properties.GetExternalColorTexture();

        switch(imageId) {
        case 0:
            if (externalColor == 0) {
                ret = m_colorTexture;
            }
            else {
                ret = externalColor->GetVideoMemoryHandle();
            }
            break;
        case 1: ret = m_depthRenderBuffer;
            break;
        default:
            break;
        }
    }
    return ret;
}

void GlTextureFrameBuffer::UnloadRenderBufferAttachment(GLuint& renderBuffer) {
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::GlTextureFrameBuffer::UnloadRenderBufferAttachment, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects (OpenGL) and is therefore considered non-const")
    if (renderBuffer != 0) {
        glDeleteRenderbuffers(1, &renderBuffer);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glDeleteRenderbuffers");
        renderBuffer = 0;
    }
}

void GlTextureFrameBuffer::UnloadTextureAttachment(GLuint& texture) {
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::GlTextureFrameBuffer::UnloadTextureAttachment, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects (OpenGL) and is therefore considered non-const")
    if (texture != 0) {
        glDeleteTextures(1, &texture);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glDeleteTextures");
        texture = 0;
    }
}

#ifdef CGIDEVICE_OPENGLES_30

bool GlTextureFrameBuffer::UploadMsaaColorAttachment()
{
    FrameBufferColorFormat colorFormat = m_properties.GetColorFormat();
    if (colorFormat != DisabledColorFormat) {
        if (m_msaaColorRenderBuffer == 0) {
            glGenRenderbuffers(1, &m_msaaColorRenderBuffer);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenRenderbuffers");
            if (m_msaaColorRenderBuffer == 0) {
                return false;
            }

            glBindRenderbuffer(GL_RENDERBUFFER, m_msaaColorRenderBuffer);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindRenderbuffer");

            glRenderbufferStorageMultisample(GL_RENDERBUFFER,
                m_properties.GetMsaaSamples(),
                FrameBufferColorFormatToGlesFormat(colorFormat),
                m_properties.GetWidth(),
                m_properties.GetHeight());
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glRenderbufferStorageMultisample");
            if (CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Gles)::WasLastErrorSet()) {
                return false;
            }
        }

        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_RENDERBUFFER, m_msaaColorRenderBuffer);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }
    else {
        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_RENDERBUFFER, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }

    return true;
}

bool GlTextureFrameBuffer::UploadMsaaDepthAttachment()
{
    FrameBufferDepthFormat depthFormat = m_properties.GetDepthFormat();
    if (depthFormat != DisabledDepthFormat) {
        GLuint buffer = 0;
        // Determine if stencil and depth buffer can be merged.
        GLenum format = FrameBufferDepthStencilFormatToGlesFormat(depthFormat, m_properties.GetStencilFormat());
        if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
            format = FrameBufferDepthFormatToGlesFormat(depthFormat);
            if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
                return false;
            }
            buffer = m_msaaDepthRenderBuffer;
        }
        else {
            buffer = m_msaaStencilRenderBuffer;
        }

        if (buffer == 0) {
            glGenRenderbuffers(1, &m_msaaDepthRenderBuffer);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenRenderbuffers");
            if (m_msaaDepthRenderBuffer == 0) {
                return false;
            }
            glBindRenderbuffer(GL_RENDERBUFFER, m_msaaDepthRenderBuffer);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindRenderbuffer");

            glRenderbufferStorageMultisample(GL_RENDERBUFFER, m_properties.GetMsaaSamples(), format, m_properties.GetWidth(), m_properties.GetHeight());
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glRenderbufferStorageMultisample");
            if (CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Gles)::WasLastErrorSet()) {
                return false;
            }
            buffer = m_msaaDepthRenderBuffer;
        }

        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, buffer);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }
    else {
        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }

    return true;
}

bool GlTextureFrameBuffer::UploadMsaaStencilAttachment()
{
    FrameBufferStencilFormat stencilFormat = m_properties.GetStencilFormat();

    if (stencilFormat == Stencil8Format) {
        GLuint buffer = 0;
        // Determine if stencil and depth buffer can be merged.
        GLenum format = FrameBufferDepthStencilFormatToGlesFormat(m_properties.GetDepthFormat(), stencilFormat);
        if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
            format = FrameBufferStencilFormatToGlesFormat(stencilFormat);
            if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
                return false;
            }
            buffer = m_msaaStencilRenderBuffer;
        }
        else {
            buffer = m_msaaDepthRenderBuffer;
        }

        if (buffer == 0) {
            glGenRenderbuffers(1, &m_msaaStencilRenderBuffer);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenRenderbuffers");
            if (m_stencilRenderBuffer == 0) {
                return false;
            }
            glBindRenderbuffer(GL_RENDERBUFFER, m_msaaStencilRenderBuffer);
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindRenderbuffer");

            glRenderbufferStorageMultisample(GL_RENDERBUFFER, m_properties.GetMsaaSamples(), format, m_properties.GetWidth(), m_properties.GetHeight());
            CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glRenderbufferStorageMultisample");
            if (CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Gles)::WasLastErrorSet()) {
                return false;
            }

            buffer = m_msaaStencilRenderBuffer;
        }
        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_STENCIL_ATTACHMENT, GL_RENDERBUFFER, buffer);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }
    else {
        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_STENCIL_ATTACHMENT, GL_RENDERBUFFER, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }

    return true;
}

#endif

bool GlTextureFrameBuffer::UploadColorAttachment()
{
    GLuint textureType = GL_TEXTURE_2D; //default
    FrameBufferColorFormat colorFormat = m_properties.GetColorFormat();
    if (colorFormat != DisabledColorFormat) {
        ImageSource3D* externalColor = m_properties.GetExternalColorTexture();
        GLuint colorTexture = 0;
        if (externalColor == 0) {
            if (m_colorTexture == 0) {
                glGenTextures(1, &m_colorTexture);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenTextures");
                if (m_colorTexture == 0) {
                    return false;
                }
                glBindTexture(GL_TEXTURE_2D, m_colorTexture);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindTexture");

                glTexImage2D(
                    GL_TEXTURE_2D,
                    0,
                    FrameBufferColorFormatToGlesInternalFormat(colorFormat),
                    m_properties.GetWidth(),
                    m_properties.GetHeight(),
                    0,
                    FrameBufferColorFormatToGlesFormat(colorFormat),
                    FrameBufferColorFormatToGlesType(colorFormat),
                    0);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glTexImage2D");
                if (CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Gles)::WasLastErrorSet()) {
                    return false;
                }
            }
            colorTexture = m_colorTexture;
        }
        else {
            colorTexture = static_cast<GLuint>(externalColor->GetVideoMemoryHandle());
            textureType = Candera::TextureTargetTypeMapping[externalColor->GetTextureTarget()];
        }

        glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, textureType, colorTexture, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferTexture2D");
    }
    else {
        glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, textureType, 0, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferTexture2D");
    }

    return true;
}

bool GlTextureFrameBuffer::UploadDepthAttachment()
{
    FrameBufferDepthFormat depthFormat = m_properties.GetDepthFormat();
    if (depthFormat != DisabledDepthFormat) {
        ImageSource3D* externalDepth = m_properties.GetExternalDepthBuffer();
        GLuint buffer = 0;
        if (externalDepth == 0) {
            // Determine if stencil and depth buffer can be merged.
            GLenum format = FrameBufferDepthStencilFormatToGlesFormat(depthFormat, m_properties.GetStencilFormat());
            if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
                format = FrameBufferDepthFormatToGlesFormat(depthFormat);
                if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
                    return false;
                }
                buffer = m_depthRenderBuffer;
            }
            else {
                buffer = m_stencilRenderBuffer;
            }

            if (buffer == 0) {
                glGenRenderbuffers(1, &m_depthRenderBuffer);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenRenderbuffers");
                if (m_depthRenderBuffer == 0) {
                    return false;
                }
                glBindRenderbuffer(GL_RENDERBUFFER, m_depthRenderBuffer);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindRenderbuffer");

                glRenderbufferStorage(GL_RENDERBUFFER, format, m_properties.GetWidth(), m_properties.GetHeight());
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glRenderbufferStorage");
                if (CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Gles)::WasLastErrorSet()) {
                    return false;
                }

                buffer = m_depthRenderBuffer;
            }
        }
        else {
            buffer = static_cast<GLuint>(externalDepth->GetVideoMemoryHandle());
        }

        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, buffer);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }
    else {
        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }

    return true;
}

bool GlTextureFrameBuffer::UploadStencilAttachment()
{
    FrameBufferStencilFormat stencilFormat = m_properties.GetStencilFormat();

    if (stencilFormat == Stencil8Format) {
        ImageSource3D* externalStencil = m_properties.GetExternalStencilBuffer();
        GLuint buffer = 0;
        if (externalStencil == 0) {
            // Determine if stencil and depth buffer can be merged.
            GLenum format = FrameBufferDepthStencilFormatToGlesFormat(m_properties.GetDepthFormat(), stencilFormat);
            if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
                format = FrameBufferStencilFormatToGlesFormat(stencilFormat);
                if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
                    return false;
                }
                buffer = m_stencilRenderBuffer;
            }
            else {
                buffer = m_depthRenderBuffer;
            }

            if (buffer == 0) {
                glGenRenderbuffers(1, &m_stencilRenderBuffer);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glGenRenderbuffers");
                if (m_stencilRenderBuffer == 0) {
                    return false;
                }
                glBindRenderbuffer(GL_RENDERBUFFER, m_stencilRenderBuffer);
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindRenderbuffer");

                glRenderbufferStorage(GL_RENDERBUFFER, format, m_properties.GetWidth(), m_properties.GetHeight());
                CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glRenderbufferStorage");
                if (CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Gles)::WasLastErrorSet()) {
                    return false;
                }

                buffer = m_stencilRenderBuffer;
            }
        }
        else {
            buffer = static_cast<GLuint>(externalStencil->GetVideoMemoryHandle()); //The user has to ensure only render buffers are passed here as textures cannot be set as stencil attachments.
        }

        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_STENCIL_ATTACHMENT, GL_RENDERBUFFER, buffer);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }
    else {
        glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_STENCIL_ATTACHMENT, GL_RENDERBUFFER, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glFramebufferRenderbuffer");
    }

    return true;
}

UInt GlTextureFrameBuffer::GetSize() const
{
    UInt size = 0;
    UInt stride = 0;
    UInt frameSize = static_cast<UInt>(m_properties.GetWidth() * m_properties.GetHeight());
    FrameBufferColorFormat colorFormat = m_properties.GetColorFormat();
    if (colorFormat != DisabledColorFormat) {
        ImageSource3D* externalColor = m_properties.GetExternalColorTexture();
        if (externalColor == 0) {
            GLenum glType = FrameBufferColorFormatToGlesType(colorFormat);
            if (glType != GL_INVALID_ENUM) {
                if (glType == GL_UNSIGNED_BYTE) {
                    stride = sizeof(UInt32);
                }
                else {
                    stride = sizeof(UInt16);
                }
            }
            size += stride * frameSize;
        }
    }

    stride = 0;
    FrameBufferDepthFormat depthFormat = m_properties.GetDepthFormat();
    bool stencil = true;
    if (depthFormat != DisabledDepthFormat) {
        ImageSource3D* externalDepth = m_properties.GetExternalDepthBuffer();
        if (externalDepth == 0) {
            GLenum format = FrameBufferDepthStencilFormatToGlesFormat(depthFormat, m_properties.GetStencilFormat());
            if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
                if (depthFormat == Depth16Format) {
                    stride = 2;
                }
                else if (depthFormat == Depth24Format) {
                    stride = 3;
                }
                else {
                    if(depthFormat == Depth32Format) {
                        stride = 4;
                    }
                }
            }
            else {
                stride = 4;
                stencil = false;    //Stencil buffer is already considered.
            }

            size += stride * frameSize;
        }
    }

    if (stencil) {
        stride = 0;
        FrameBufferStencilFormat stencilFormat = m_properties.GetStencilFormat();

        if (stencilFormat == Stencil8Format) {
            ImageSource3D* externalStencil = m_properties.GetExternalStencilBuffer();
            if (externalStencil == 0) {
                stride = 1;
            }
            size += stride * frameSize;
        }
    }

#ifdef CGIDEVICE_OPENGLES_30
    if (m_properties.GetMsaaSamples() > 1) {
        frameSize *= m_properties.GetMsaaSamples();
        // additional space used by MSAA frame buffer object.
        stride = 0;
        if (colorFormat != DisabledColorFormat) {
            GLenum glType = FrameBufferColorFormatToGlesType(colorFormat);
            if (glType != GL_INVALID_ENUM) {
                if (glType == GL_UNSIGNED_BYTE) {
                    stride = sizeof(UInt32);
                }
                else {
                    stride = sizeof(UInt16);
                }
            }
            size += stride * frameSize;
        }

        stride = 0;
        if (depthFormat != DisabledDepthFormat) {
            GLenum format = FrameBufferDepthStencilFormatToGlesFormat(depthFormat, m_properties.GetStencilFormat());
            if ((format == c_allBitsSet) || (format == GL_INVALID_ENUM)) {
                if (depthFormat == Depth16Format) {
                    stride = 2;
                }
                else if (depthFormat == Depth24Format) {
                    stride = 3;
                }
                else {
                    if (depthFormat == Depth32Format) {
                        stride = 4;
                    }
                }
            }
            else {
                stride = 4;
                stencil = false;    //Stencil buffer is already considered.
            }

            size += stride * frameSize;
        }

        if (stencil) {
            stride = 0;
            FrameBufferStencilFormat stencilFormat = m_properties.GetStencilFormat();

            if (stencilFormat == Stencil8Format) {
                ImageSource3D* externalStencil = m_properties.GetExternalStencilBuffer();
                if (externalStencil == 0) {
                    stride = 1;
                }
                size += stride * frameSize;
            }
        }
    }
#endif

    return size;
}

}
