//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <CanderaPlatform/Device/Common/Internal/GL/GlFrameBufferTypes.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlInclude.h>
#include CANDERA_GLES_INCLUDE(GlTypeMapper.h)

namespace Candera
{

static const Bitmap::PixelFormat c_FrameBufferColorFormatToBitmapPixelFormat[] =
{
    Bitmap::PixelFormat(-1),                       // DisabledColorFormat
    Bitmap::RgbaUnsignedBytePixelFormat,           // RgbaUnpackedColorFormat
    Bitmap::RgbUnsignedBytePixelFormat,            // RgbUnpackedColorFormat
    Bitmap::RgbaUnsignedShort4444PixelFormat,      // Rgba4ColorFormat
    Bitmap::RgbaUnsignedShort5551PixelFormat,      // Rgba5A1ColorFormat
    Bitmap::RgbUnsignedShort565PixelFormat,        // Rgb565ColorFormat
    Bitmap::LuminanceAlphaUnsignedBytePixelFormat, // LuminanceAlphaColorFormat
    Bitmap::LuminanceUnsignedBytePixelFormat,      // LuminanceColorFormat
    Bitmap::AlphaUnsignedBytePixelFormat,          // AlphaColorFormat
    Bitmap::SrgbaUnsignedBytePixelFormat,          // SrgbaColorFormat
    Bitmap::RgbaUnsignedIntRev10A2PixelFormat      // Rgba10A2ColorFormat
};

GLenum FrameBufferColorFormatToGlesType(FrameBufferColorFormat format)
{
    if (DisabledColorFormat == format) {
        return GL_INVALID_ENUM;
    }

    Bitmap::PixelFormat pixelFormat = FrameBufferColorFormatToBitmapPixelFormat(format);
    return GlTypeMapper::MapBitmapPixelFormatToGLType(pixelFormat);
}

GLenum FrameBufferColorFormatToGlesFormat(FrameBufferColorFormat format)
{
    if (DisabledColorFormat == format) {
        return GL_INVALID_ENUM;
    }
    
    Bitmap::PixelFormat pixelFormat = FrameBufferColorFormatToBitmapPixelFormat(format);
    return GlTypeMapper::MapBitmapPixelFormatToGLFormat(pixelFormat);
}

GLenum FrameBufferColorFormatToGlesInternalFormat(FrameBufferColorFormat format)
{
    if (DisabledColorFormat == format) {
        return GL_INVALID_ENUM;
    }

    Bitmap::PixelFormat pixelFormat = FrameBufferColorFormatToBitmapPixelFormat(format);
    return GlTypeMapper::MapBitmapPixelFormatToGLInternalFormat(pixelFormat);
}

Bitmap::Type FrameBufferColorFormatToBitmapType(FrameBufferColorFormat format)
{
    if (DisabledColorFormat == format) {
        return Bitmap::Type(-1);
    }

    Bitmap::PixelFormat pixelFormat = FrameBufferColorFormatToBitmapPixelFormat(format);
    return Bitmap::MapPixelFormatToDeprecatedType(pixelFormat);
}

Int FrameBufferColorFormatToBitmapFormat(FrameBufferColorFormat format)
{
    if (DisabledColorFormat == format) {
        return static_cast<Int>(-1);
    }

    Bitmap::PixelFormat pixelFormat = FrameBufferColorFormatToBitmapPixelFormat(format);
    return Bitmap::MapPixelFormatToDeprecatedFormat(pixelFormat);
}

Bitmap::PixelFormat FrameBufferColorFormatToBitmapPixelFormat(FrameBufferColorFormat format)
{
    FEATSTD_DEBUG_ASSERT((format >= static_cast<FrameBufferColorFormat>(0)) &&
        (format < static_cast<FrameBufferColorFormat>((sizeof(c_FrameBufferColorFormatToBitmapPixelFormat) /
        sizeof(Bitmap::PixelFormat)))));
    return c_FrameBufferColorFormatToBitmapPixelFormat[format];
}

UInt8 FrameBufferColorFormatBitsPerPixel(FrameBufferColorFormat format)
{
    if (DisabledColorFormat == format) {
        return 0;
    }

    Bitmap::PixelFormat pixelFormat = FrameBufferColorFormatToBitmapPixelFormat(format);
    return GlTypeMapper::MapBitmapPixelFormatToGLBitsPerPixel(pixelFormat);
}

GLenum FrameBufferDepthFormatToGlesFormat(FrameBufferDepthFormat format)
{
    GLenum ret;
    switch (format) {
        case Depth16Format:
            ret = GL_DEPTH_COMPONENT16; break;
#if defined(CGIDEVICE_OPENGLES_30)
        case Depth24Format:
            ret = GL_DEPTH_COMPONENT24; break;
        case Depth32Format:
            ret = GL_DEPTH_COMPONENT32F; break;
#else
        case Depth24Format:
            ret = GL_DEPTH_COMPONENT24_OES; break;
        case Depth32Format:
            ret = GL_DEPTH_COMPONENT32_OES; break;
#endif
        case DisabledDepthFormat:
        default:
            ret = GL_INVALID_ENUM; break;
    }
    return ret;
}

Int FrameBufferDepthFormatToBitmapFormat(FrameBufferDepthFormat format)
{
    GLenum ret = FrameBufferDepthFormatToBitmapPixelFormat(format);

    if (format != static_cast<Candera::FrameBufferDepthFormat>(c_glInvalidParameter)) {
        ret = Bitmap::MapPixelFormatToDeprecatedFormat(format);
    }

    return ret;
}

Int FrameBufferDepthFormatToBitmapPixelFormat(FrameBufferDepthFormat format)
{
    GLenum ret;
    switch (format) {
    case Depth16Format:
    case Depth24Format:
    case Depth32Format:
        ret = static_cast<Int>(Bitmap::DepthUnsignedShortPixelFormat); break;

    case DisabledDepthFormat:
    default:
        ret = c_glInvalidParameter; break;
    }
    return ret;
}

GLenum FrameBufferStencilFormatToGlesFormat(FrameBufferStencilFormat format)
{
    GLenum ret;
    switch (format) {
        case Stencil8Format:
            ret = GL_STENCIL_INDEX8; break;
        case DisabledStencilFormat:
        default:
            ret = GL_INVALID_ENUM; break;
    }
    return ret;
}

Int FrameBufferStencilFormatToBitmapFormat(FrameBufferStencilFormat format)
{
    GLenum ret = FrameBufferStencilFormatToBitmapPixelFormat(format);

    if (format != static_cast<Candera::FrameBufferStencilFormat>(c_glInvalidParameter)) {
        ret = Bitmap::MapPixelFormatToDeprecatedFormat(format);
    }

    return ret;
}

Int FrameBufferStencilFormatToBitmapPixelFormat(FrameBufferStencilFormat format)
{
    GLenum ret;
    switch (format) {
    case Stencil8Format:
    case DisabledStencilFormat:
    default:
        ret = c_glInvalidParameter; break;
    }
    return ret;
}

GLenum FrameBufferDepthStencilFormatToGlesFormat(FrameBufferDepthFormat dformat, FrameBufferStencilFormat sformat)
{
    if ((sformat == Stencil8Format) && (dformat >= Depth16Format) && (dformat <= Depth24Format)) {
#if defined(CGIDEVICE_OPENGLES_30)
        return GL_DEPTH24_STENCIL8;
#else
        return GL_DEPTH24_STENCIL8_OES;
#endif
    }
    return GL_INVALID_ENUM;
}

static bool ColorFormatMatches(
    Int redSize, Int greenSize, Int blueSize, Int alphaSize,
    Int redRef, Int greenRef, Int blueRef, Int alphaRef)
{
    return (redSize <= redRef) &&
        (greenSize <= greenRef) &&
        (blueSize <= blueRef) &&
        (alphaSize <= alphaRef);
}

FrameBufferColorFormat RgbaToFrameBufferColorFormat(Int red, Int green, Int blue, Int alpha)
{
    FrameBufferColorFormat format = DisabledColorFormat;
    if (!ColorFormatMatches(red, green, blue, alpha, 0, 0, 0, 0)) {
        if (ColorFormatMatches(red, green, blue, alpha, 5, 6, 5, 0)) {
            format = Rgb565ColorFormat;
        }
        else if (ColorFormatMatches(red, green, blue, alpha, 8, 8, 8, 0)) {
            format = RgbUnpackedColorFormat;
        }
        else if (ColorFormatMatches(red, green, blue, alpha, 5, 5, 5, 1)) {
            format = Rgba5A1ColorFormat;
        }
        else if (ColorFormatMatches(red, green, blue, alpha, 4, 4, 4, 4)) {
            format = Rgba4ColorFormat;
        }
        else if (ColorFormatMatches(red, green, blue, alpha, 8, 8, 8, 8)) {
            format = RgbaUnpackedColorFormat;
        }
        else {
            // do nothing
        }
    }
    return format;
}

FrameBufferDepthFormat SizeToFrameBufferDepthFormat(Int size)
{
    FrameBufferDepthFormat format = DisabledDepthFormat;
    if (size > 0) {
        if (size <= 16) {
            format = Depth16Format;
        }
        else if (size <= 24) {
            format = Depth24Format;
        }
        else if (size <= 32) {
            format = Depth32Format;
        }
        else {
            // do nothing
        }
    }
    return format;
}

FrameBufferStencilFormat SizeToFrameBufferStencilFormat(Int size)
{
    FrameBufferStencilFormat format = DisabledStencilFormat;
    if (size > 0) {
        if (size <= 8) {
            format = Stencil8Format;
        }
    }
    return format;
}

}
