//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GL_FLUSH_SYNC_H)
#define GL_FLUSH_SYNC_H

#include <Candera/Environment.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

class EglContext;

/**
 * @brief GlFlushSync represents class used for synchronizing Open GL operations.
 * It uses glFinalize to sync the GPU and the CPU.
 *
 * This object may be used as a sync object for GduBaseRenderTarget, with 2D and/or 3D attachments.
 */
class GlFlushSync
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GlFlushSync::Unload)

    public:

        GlFlushSync();

        virtual ~GlFlushSync();

        /**
         * Brings this object to a synchronized state, so that the
         * Cpu can access freely the resources associated with this context.
         * @param contextProvider Object that provides the context for which the synchronization takes place.
         */
        void WaitSync(EglContext& contextProvider) const;
        /**
         * Does nothing. The Open GL objects area always in synchronous mode in relation to other Open GL
         * operations executed from the same thread.
         * @param contextProvider Unused.
         */
        void Sync(EglContext& contextProvider) const;
        /**
         * Does nothing. @see Sync.
         * @param contextProvider Unused.
         */
        void WaitBeginDraw(EglContext& contextProvider) const;
        /**
         * Does nothing. This object does not synchronize to a marker.
         * @param contextProvider Unused.
         */
        void MarkEndDraw(EglContext& contextProvider) const;

        /**
         * Same as WaitSync. Used by 2D graphic device units base on Open GL.
         * @param contextProvider Same as WaitSync.
         */
        void WaitSync2D(EglContext& contextProvider) { return WaitSync(contextProvider); }
        /**
         * Does nothing. Sync.
         * @param contextProvider Unused.
         */
        void Sync2D(EglContext& contextProvider) { return Sync(contextProvider); }
        /**
         * Does nothing. WaitBeginDraw.
         * @param contextProvider Unused.
         */
        void WaitBeginDraw2D(EglContext& contextProvider) { return WaitBeginDraw(contextProvider); }
        /**
         * Does nothing. MarkEndDraw.
         * @param contextProvider Unused.
         */
        void MarkEndDraw2D(EglContext& contextProvider) { return MarkEndDraw(contextProvider); }
};

/** @}*/ //end of CommonDevice

}

#endif
