//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_WRAPPER_RENDER_TARGET_2D_H)
#define GDU_WRAPPER_RENDER_TARGET_2D_H

#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduWrapper.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduProperties.h>

#include <FeatStd/Event/EventListener.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief Class GduWrapperRenderTarget2D provides dedicated interfaces to access a specific surface
 * as a 2D render target.
 *
 * The class has one template parameter representing the specific type of surface which is
 * wrapped by this class. The specific type must implement the folowing methods:
 *   Int GetHeight2D() const;
 *   Int GetWidth2D() const;
 *   void Sync2D()
 *   void WaitSync2D()
 *   ContextHandle2D Get2DContextHandle()
 *   bool Activate2D()
 *   void BeginDraw2D()
 *   void EndDraw2D()
 *   void SwapBuffers2D()
 * @param TWrappedType type of the wrapped surface
 */
template <typename TWrappedType>
class GduWrapperRenderTarget2D : public RenderTarget2D, public GduWrapper<TWrappedType>, public FeatStd::EventListener
{
    typedef GduWrapper<TWrappedType> Wrapper;
    public:
        /**
         *  Constructs a GduWrapperRenderTarget2D object.
         *  @param surface      The surface to which this interface is attached.
         *  @param index        The index used to retrieve the ImageHandle using GetImageHandle.
         */
        GduWrapperRenderTarget2D();

        /**
         *  Destructs a GduWrapperRenderTarget2D object.
         */
        virtual ~GduWrapperRenderTarget2D();

        /**
        * Initilize the wrapper. Called when creating a GduAttachment.
        * @param wrappedObject     Base object of the gdu, wrapped by this wrapper.
        * @param index             Index of the wrapper within the Gdu.
        */
        void SetBaseObject(TWrappedType& wrappedObject, Int index);

        /**
         *  Returns current height of the surface. Overrides pure virtual function of class WrappedSurface.
         *  @return The current height of the surface.
         */
        virtual Int GetHeight() const;

        /**
         *  Returns current width of the surface. Overrides pure virtual function of class WrappedSurface.
         *  @return the current width of the surface.
         */
        virtual Int GetWidth() const;

        /**
         *  Overrides pure virtual function of class Synchronizable.
         */
        virtual void Sync();

        /**
         *  Overrides pure virtual function of class Synchronizable.
         */
        virtual void WaitSync();

        /**
         *  Overrides pure virtual function of class Synchronizable.
         */
        virtual void OnAccess();

        /**
         *  Returns the 2D context. The context can be used in the 2D API.
         *  @return the 2D context handle
         */
        virtual ContextHandle2D Get2DContextHandle();

        /**
         *  Makes the render target active by setting the context to the associated surface.
         *  @return true if activation was successful
         */
        virtual bool Activate();

        /**
         *  A call to BeginDraw makes the object leave the synchronized state. Within this state neither calls to another
         *  BeginDraw of the same object nor any Sync methods are allowed and would result in undefined behavior (i.e. most
         *  certainly unwanted behavior).
         */
        virtual void BeginDraw();

        /**
         *  EndDraw tells the object that the client's definition of operations have finished. Note, that at this time
         *  the state is still not necessarily synchronized as just the completeness specification is present but the
         *  GPU might still work to finish the specified task.
         *  At this point also calls to any Sync methods and BeginDraw are allowed.
         */
        virtual void EndDraw();

        /**
         *  If the draw target operates on multiple buffers the presentation buffer is exchanged with the current background
         *  buffer. Derived objects might provide additional interfaces about when buffer swapping is executed.
         */
        virtual void SwapBuffers();

        virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event& event) override;

        typedef typename TWrappedType::EventListenerCollection EventListenerCollection;
        typedef typename EventListenerCollection::template TEvent<EventListenerCollection::BeforeSwap> BeforeSwapEvent;
        typedef typename EventListenerCollection::template TEvent<EventListenerCollection::AfterSwap> AfterSwapEvent;
};

/** @}*/ //end of CommonDevice

template <typename TWrappedType>
GduWrapperRenderTarget2D<TWrappedType>::GduWrapperRenderTarget2D()
{

}

template <typename TWrappedType>
GduWrapperRenderTarget2D<TWrappedType>::~GduWrapperRenderTarget2D()
{
    if (Wrapper::GetWrappedObject() != 0) {
        Wrapper::GetWrappedObject()->RemoveEventListener(this);
    }
}


template <typename TWrappedType>
void GduWrapperRenderTarget2D<TWrappedType>::SetBaseObject(TWrappedType& wrappedObject, Int index)
{
    if (Wrapper::GetWrappedObject() != 0) {
        Wrapper::GetWrappedObject()->RemoveEventListener(this);
    }
    Wrapper::SetBaseObject(wrappedObject, index); 
    
    Wrapper::GetWrappedObject()->AddEventListener(this);
}

template <typename TWrappedType>
Int GduWrapperRenderTarget2D<TWrappedType>::GetHeight() const
{
    return (Wrapper::GetWrappedObject() == 0) ? 0 : Wrapper::GetWrappedObject()->GetHeight2D();
}

template <typename TWrappedType>
Int GduWrapperRenderTarget2D<TWrappedType>::GetWidth() const
{
    return (Wrapper::GetWrappedObject() == 0) ? 0 : Wrapper::GetWrappedObject()->GetWidth2D();
}

template <typename TWrappedType>
void GduWrapperRenderTarget2D<TWrappedType>::Sync()
{
    (Wrapper::GetWrappedObject() == 0) ? (void)0 : Wrapper::GetWrappedObject()->Sync2D();
}

template <typename TWrappedType>
void GduWrapperRenderTarget2D<TWrappedType>::WaitSync()
{
    (Wrapper::GetWrappedObject() == 0) ? (void)0 : Wrapper::GetWrappedObject()->WaitSync2D();
}

template <typename TWrappedType>
void GduWrapperRenderTarget2D<TWrappedType>::OnAccess()
{
    (Wrapper::GetWrappedObject() == 0) ? (void)0 : Wrapper::GetWrappedObject()->OnAccess2D();
}

template <typename TWrappedType>
ContextHandle2D GduWrapperRenderTarget2D<TWrappedType>::Get2DContextHandle()
{
    return (Wrapper::GetWrappedObject() == 0) ? 0 : Wrapper::GetWrappedObject()->Get2DContextHandle();
}

template <typename TWrappedType>
bool GduWrapperRenderTarget2D<TWrappedType>::Activate()
{
    return (Wrapper::GetWrappedObject() == 0) ? false : Wrapper::GetWrappedObject()->Activate2D();
}

template <typename TWrappedType>
void GduWrapperRenderTarget2D<TWrappedType>::BeginDraw()
{
    (Wrapper::GetWrappedObject() == 0) ? (void)0 : Wrapper::GetWrappedObject()->BeginDraw2D();
}

template <typename TWrappedType>
void GduWrapperRenderTarget2D<TWrappedType>::EndDraw()
{
    (Wrapper::GetWrappedObject() == 0) ? (void)0 : Wrapper::GetWrappedObject()->EndDraw2D();
}

template <typename TWrappedType>
void GduWrapperRenderTarget2D<TWrappedType>::SwapBuffers()
{
    (Wrapper::GetWrappedObject() == 0) ? (void)0 : Wrapper::GetWrappedObject()->SwapBuffers2D();
}

template <typename TWrappedType>
FeatStd::EventResult::Enum GduWrapperRenderTarget2D<TWrappedType>::OnEvent(const FeatStd::Event& event)
{
    const BeforeSwapEvent* beforeSwapEvent = Dynamic_Cast<const BeforeSwapEvent*>(&event);
    if ((beforeSwapEvent != 0) && (Wrapper::GetWrappedObject() == beforeSwapEvent->GetRenderTarget())) {
        NotifyListenersOnBeforeSwapBuffersCalled();
    }

    const AfterSwapEvent* afterSwapEvent = Dynamic_Cast<const AfterSwapEvent*>(&event);
    if ((afterSwapEvent != 0) && (Wrapper::GetWrappedObject() == afterSwapEvent->GetRenderTarget())) {
        NotifyListenersOnAfterSwapBuffersExecuted();
    }

    return FeatStd::EventResult::Proceed;
}

}

#endif
