//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_WRAPPER_H)
#define GDU_WRAPPER_H

#include <CanderaPlatform/Device/Common/Internal/GDU/GduProperties.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief Class GduWrapper is a base class used for easily writting wrappers
 * used with GduAttachments.
 */
template <typename TWrappedType>
class GduWrapper
{
public:
    typedef TWrappedType WrappedType;

    /**
     * Constructor.
     */
    GduWrapper() : m_object(0), m_index(0) {}

    /**
     * Initilize the wrapper. Called when creating a GduAttachment.
     * @param wrappedObject     Base object of the gdu, wrapped by this wrapper.
     * @param index             Index of the wrapper within the Gdu.
     */
    void SetBaseObject(WrappedType& wrappedObject, Int index);

    /**
     * Does nothing. Called when uploading a GduAttachment.
     * @param displayId         Unused.
     * @param properties        Unused.
     * @return true;
     */
    bool Upload(Int displayId, GduProperties& properties);

    /**
     * Does nothing. Called when unloading a GduAttachment.
     */
    void Unload();

    /**
     * Does nothing. Called when updating a GduAttachment.
     * @param properties        Unused.
     */
    void ApplyChanges(GduProperties& properties);

    /**
     * Retrieve the wrapped object.
     * @return pointer to the Wrapped object, 0 if this object is uninitialized.
     */
    WrappedType* GetWrappedObject() const { return m_object; }

    /**
     * Retrieve the index of the wrapped object.
     * @return index of the Wrapped object, 0 if this object is uninitialized.
     */
    Int GetIndex() const { return m_index; }

protected:
    /**
     * Non-virtual destructor. Objects should never be destroyed by pointer to this object.
     */
    ~GduWrapper() { m_object = 0; m_index = 0; }

private:
    WrappedType* m_object;
    Int m_index;
};

/** @}*/ //end of CommonDevice
template <typename TWrappedType>
void GduWrapper<TWrappedType>::SetBaseObject(WrappedType& wrappedObject, Int index)
{
    m_object = &wrappedObject;
    m_index = index;
}

template <typename TWrappedType>
bool GduWrapper<TWrappedType>::Upload(Int /*displayId*/, GduProperties& /*properties*/)
{
    return true;
}

template <typename TWrappedType>
void GduWrapper<TWrappedType>::Unload()
{
}

template <typename TWrappedType>
void GduWrapper<TWrappedType>::ApplyChanges(GduProperties& /*properties*/)
{
}


}

#endif
