//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_RENDERTARGET_2D_ATTACHMENT_H)
#define GDU_RENDERTARGET_2D_ATTACHMENT_H

#include <Candera/Environment.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduRenderTarget2DAttachment should be attached to a GduRenderTarget2DAttachment,
 * to make a complete RenderTarget2D, and interface for GduWrapperImageSource2D.
 *
 * It introduces a set of additional restrictions on the aggregated object:
 * - the Context needs to be an 2D ContextProvider. Must implement:
 *   bool Activate2D();
 *   ContextHandle2D Get2DContextHandle() const;
 * - the FrameBuffer needs to implement:
 *   Int GetHeight2D() const;
 *   Int GetWidth2D() const;
 *   bool Activate2D(TBase& );
 *  - SyncObject must implement:
 *   void WaitBeginDraw(2DTBase& );
 *   void MarkEndDraw2D(TBase& )
 *   void WaitSync2D(TBase& )
 *   void Sync2D(TBase& );
 *
 * @param TBase GduRenderTarget2DAttachment or class that extends GduRenderTarget2DAttachment
 */
template <typename TBase>
class GduRenderTarget2DAttachment : public TBase
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GduRenderTarget2DAttachment::Unload)

    public:
        FEATSTD_TYPEDEF_BASE(TBase);
        typedef typename Base::FrameBuffer FrameBuffer;
        typedef typename Base::Properties Properties;
        typedef typename Base::SyncObject SyncObject;
        typedef typename Base::Support Support;

        /**
         *  Constructs a GduRenderTarget2DAttachment object.
         */
        GduRenderTarget2DAttachment() {}

        /**
         *  Destructs a GduRenderTarget2DAttachment object.
         */
        ~GduRenderTarget2DAttachment() {}

        /**
         *  Retrieves the context associated with this RenderTarget.
         *  @return handle of the associated EGLContext.
         */
        ContextHandle2D Get2DContextHandle() const;

        /**
         *  Activates this render target and makes it the current default render target.
         *  @return True, if no errors occurred, false otherwise.
         */
        bool Activate2D();

        /**
         *  Returns current height of the surface. Implements function of class Surface.
         *  @return The current height of the surface.
         */
        Int GetHeight2D() const;

        /**
         *  Returns current width of the surface. Implements function of class Surface.
         *  @return The current width of the surface.
         */
        Int GetWidth2D() const;

        /**
         *  Implements function of class RenderTarget.
         */
        void BeginDraw2D();

        /**
         *  Implements function of class RenderTarget.
         */
        void EndDraw2D();

        /**
         *  Implements function of class Synchronizable.
         */
        void WaitSync2D();

        /**
         *  Implements function of class Synchronizable.
         */
        void Sync2D();

        /**
         *  Implements function of class Synchronizable.
         */
        void OnAccess2D();

        /**
         *  Implements function of class RenderTarget.
         */
        void SwapBuffers2D();

};

/** @}*/ //end of CommonDevice

template <typename TBase>
ContextHandle2D GduRenderTarget2DAttachment<TBase>::Get2DContextHandle() const
{
    return Base::Get2DContextHandle();
}

template <typename TBase>
bool GduRenderTarget2DAttachment<TBase>::Activate2D()
{
    if (Base::Activate2D()) {
        return Base::GetFrameBuffer().Activate2D(*this);
    }
    return false;
}

template <typename TBase>
Int GduRenderTarget2DAttachment<TBase>::GetHeight2D() const
{
    return Base::GetFrameBuffer().GetHeight2D();
}

template <typename TBase>
Int GduRenderTarget2DAttachment<TBase>::GetWidth2D() const
{
    return Base::GetFrameBuffer().GetWidth2D();
}

template <typename TBase>
void GduRenderTarget2DAttachment<TBase>::BeginDraw2D()
{
    if (Base::IsDrawing()){
        CANDERA_DEVICE_LOG_WARN("already drawing");
        return;
    }
    Base::SetDrawing(true);
    if (Base::Activate2D()) {
        Base::GetFrameBuffer().BeginDraw2D(*this);
    }
}

template <typename TBase>
void GduRenderTarget2DAttachment<TBase>::EndDraw2D()
{
    if (!Base::IsDrawing()){
        CANDERA_DEVICE_LOG_WARN("not drawing");
        return;
    }
    Base::GetFrameBuffer().EndDraw2D(*this);
    Base::GetSyncObject().MarkEndDraw2D(*this);
    Base::SetDrawing(false);
}

template <typename TBase>
void GduRenderTarget2DAttachment<TBase>::WaitSync2D()
{
    CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::WaitSync, "RenderTarget2D"));
    return Base::GetSyncObject().WaitSync2D(*this);
}

template <typename TBase>
void GduRenderTarget2DAttachment<TBase>::Sync2D()
{
    return Base::GetSyncObject().Sync2D(*this);
}

template <typename TBase>
void GduRenderTarget2DAttachment<TBase>::OnAccess2D()
{
    return Base::GetSyncObject().MarkEndDraw2D(*this);
}

template <typename TBase>
void GduRenderTarget2DAttachment<TBase>::SwapBuffers2D()
{
    return Base::SwapBuffers();
}

}

#endif
