//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_IMAGESOURCE_3D_ATTACHMENT_H)
#define GDU_IMAGESOURCE_3D_ATTACHMENT_H

#include <Candera/Environment.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduImageSource3DAttachment should be attached to a GduImageSource3DAttachment,
 * to make a complete ImageSource3D, and interface for GduWrapperImageSource3D.
 *
 * It introduces a set of additional restrictions on the aggregated object:
 * - the FrameBuffer needs to implement:
 *   Int GetHeight() const;
 *   Int GetWidth() const;
 *   Handle GetImageHandle(Int bufferId, Int imageId);
 *   Int GetPreviousBufferIndex();
 *   void WaitSync(TBase& )
 *   void Sync(TBase& );
 *
 * @param TBase GduImageSource3DAttachment or class that extends GduImageSource3DAttachment
 */
template <typename TBase>
class GduImageSource3DAttachment : public TBase
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GduImageSource3DAttachment::Unload)

    public:
        FEATSTD_TYPEDEF_BASE(TBase);
        typedef typename Base::FrameBuffer FrameBuffer;
        typedef typename Base::Properties Properties;
        typedef typename Base::SyncObject SyncObject;
        typedef typename Base::Support Support;

        /**
         *  Constructs a GduImageSource3DAttachment object.
         */
        GduImageSource3DAttachment() {}

        /**
         *  Destructs a GduImageSource3DAttachment object.
         */
        ~GduImageSource3DAttachment() {}

        /**
         *  Returns current height of the surface. Implements function of class Surface.
         *  @return The current height of the surface.
         */
        Int GetHeight() const;

        /**
         *  Returns current width of the surface. Implements function of class Surface.
         *  @return The current width of the surface.
         */
        Int GetWidth() const;

        /**
         *  Implements function of class Synchronizable.
         */
        void WaitSync();

        /**
         *  Implements function of class Synchronizable.
         */
        void Sync();

        /**
         *  Implements function of class Synchronizable.
         */
        void OnAccess();

        /**
         *  Returns the image handle associated with the given index. Required by GduWrapperImageSource3D.
         *  @param imageId          Channel of the associated image. 0 = color, 1 = depth.
         *  @return Image handle associated with this object.
         */
        Handle GetImageHandle(Int imageId);
};

/** @}*/ //end of CommonDevice

template <typename TBase>
Int GduImageSource3DAttachment<TBase>::GetHeight() const
{
    return Base::GetFrameBuffer().GetHeight();
}

template <typename TBase>
Int GduImageSource3DAttachment<TBase>::GetWidth() const
{
    return Base::GetFrameBuffer().GetWidth();
}

template <typename TBase>
void GduImageSource3DAttachment<TBase>::WaitSync()
{
    CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::WaitSync, "ImageSource3D"));
    return Base::GetSyncObject().WaitSync(*this);
}

template <typename TBase>
void GduImageSource3DAttachment<TBase>::Sync()
{
    return Base::GetSyncObject().Sync(*this);
}

template <typename TBase>
void GduImageSource3DAttachment<TBase>::OnAccess()
{
    return Base::GetSyncObject().MarkEndDraw(*this);
}

template <typename TBase>
Handle GduImageSource3DAttachment<TBase>::GetImageHandle(Int imageId)
{
    return Base::GetFrameBuffer().GetImageHandle(Base::GetFrameBuffer().GetPreviousBufferIndex(), imageId);
}


}

#endif
