//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_IMAGESOURCE_2D_ATTACHMENT_H)
#define GDU_IMAGESOURCE_2D_ATTACHMENT_H

#include <Candera/Environment.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduImageSource2DAttachment should be attached to a GduImageSource2DAttachment,
 * to make a complete ImageSource2D, and interface for GduWrapperImageSource2D.
 *
 * It introduces a set of additional restrictions on the aggregated object:
 * - the FrameBuffer needs to implement:
 *   Int GetHeight2D() const;
 *   Int GetWidth2D() const;
 *   Handle GetImage2DHandle(Int bufferId, Int imageId);
 *   Int GetPreviousBufferIndex();
 *  - SyncObject must implement:
 *   void WaitSync2D(TBase& )
 *   void Sync2D(TBase& );
 *
 * @param TBase GduImageSource2DAttachment or class that extends GduImageSource2DAttachment
 */
template <typename TBase>
class GduImageSource2DAttachment : public TBase
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GduImageSource2DAttachment::Unload)

    public:
        FEATSTD_TYPEDEF_BASE(TBase);
        typedef typename Base::FrameBuffer FrameBuffer;
        typedef typename Base::Properties Properties;
        typedef typename Base::SyncObject SyncObject;
        typedef typename Base::Support Support;

        /**
         *  Constructs a GduImageSource2DAttachment object.
         */
        GduImageSource2DAttachment() {}

        /**
         *  Destructs a GduImageSource2DAttachment object.
         */
        ~GduImageSource2DAttachment() {}

        /**
         *  Returns current height of the surface. Implements function of class Surface.
         *  @return The current height of the surface.
         */
        Int GetHeight2D() const;

        /**
         *  Returns current width of the surface. Implements function of class Surface.
         *  @return The current width of the surface.
         */
        Int GetWidth2D() const;

        /**
         *  Implements function of class Synchronizable.
         */
        void WaitSync2D();

        /**
         *  Implements function of class Synchronizable.
         */
        void Sync2D();

        /**
         *  Implements function of class Synchronizable.
         */
        void OnAccess2D();

        /**
         *  Returns the image handle associated with the given index. Required by GduWrapperImageSource2D.
         *  @param imageId          0.
         *  @return Image handle associated with this object.
         */
        Handle GetImage2DHandle(Int index) const;
};

/** @}*/ //end of CommonDevice

template <typename TBase>
Int GduImageSource2DAttachment<TBase>::GetHeight2D() const
{
    return Base::GetFrameBuffer().GetHeight2D();
}

template <typename TBase>
Int GduImageSource2DAttachment<TBase>::GetWidth2D() const
{
    return Base::GetFrameBuffer().GetWidth2D();
}

template <typename TBase>
void GduImageSource2DAttachment<TBase>::WaitSync2D()
{
    CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::WaitSync, "ImageSource2D"));
    return Base::GetSyncObject().WaitSync2D(*this);
}

template <typename TBase>
void GduImageSource2DAttachment<TBase>::Sync2D()
{
    return Base::GetSyncObject().Sync2D(*this);
}

template <typename TBase>
void GduImageSource2DAttachment<TBase>::OnAccess2D()
{
    return Base::GetSyncObject().MarkEndDraw2D(*this);
}

template <typename TBase>
Handle GduImageSource2DAttachment<TBase>::GetImage2DHandle(Int /*imageId*/) const
{
    return Base::GetFrameBuffer().GetImage2DHandle(Base::GetFrameBuffer().GetPreviousBufferIndex());
}


}

#endif
