//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_CONTEXT_2D_ATTACHMENT_H)
#define GDU_CONTEXT_2D_ATTACHMENT_H

#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Internal/Util/Select.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduContext2DAttachment represents a ContextProvider with 2D destination capabilities.
 */
template <
    typename TBase,
    typename TSupport,
    typename TProperties,
    typename TContext2D,
    typename TSizeProvider = TProperties
    >
class GduContext2DAttachment : public TBase
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GduContext2DAttachment::Unload)

    public:

        FEATSTD_TYPEDEF_BASE(TBase);
        typedef TSupport Support;
        typedef TProperties Properties;
        typedef TContext2D Context2D;
        typedef TSizeProvider SizeProvider;

        using Base::ActivateContext;

        /**
         *  Constructs a GduContext2DAttachment object.
         */
        GduContext2DAttachment() : m_uploaded(false), m_context2D() {}

        /**
         *  Uploads this object to video memory. Required by GduBaseRenderTarget.
         *  @param displayId        Forwarded to Base.
         *  @param support          Forwarded to Base.
         *  @param properties       Forwarded to Base.
         *  @return True if successful, false otherwise.
         */
        bool Upload(Int displayId, Support& support, Properties& properties);

        /**
         * Applies properties that can be changed after upload has been conducted.
         * Required by GduBaseRenderTarget.
         * @param properties    Forwarded to Base.
         */
        void ApplyChanges(Properties& properties);

        /**
         *  Unloads this object from video memory. Required by GduBaseRenderTarget.
         */
        void Unload();

        /**
         *  Retrieves the associated context.
         *  @return reference to the 2D context used internaly, 0 if not uploaded.
         */
        Context2D* Get2DContext() { return m_uploaded ? &m_context2D : 0; }
        const Context2D* Get2DContext() const { return m_uploaded ? &m_context2D : 0; }

        /**
         *  Retrieves a handle for the associated context.
         *  @return handle of the 2D context for use with RenderDevice2D.
         */
        ContextHandle2D Get2DContextHandle() const { return reinterpret_cast<ContextHandle2D>(Get2DContext()); }

        /**
         *  Activate the 2D context.
         *  This activates the context of the base. The attached 2D contexts shouldn't require any activation.
         *  Binding of the frame buffer to the context is handled by the activation.
         *  @return True if the activation is successfull.
         */
        bool Activate2D() { return m_uploaded ? Base::Activate() : false; }

    protected:
        /*
         *  Non-virtual. An object should never be distroyed from a pointer to this class.
         */
        ~GduContext2DAttachment() { Base::Activate(); Unload(); }

    private:
        bool m_uploaded;
        Context2D m_context2D;
};

/** @}*/ //end of CommonDevice

template <typename TBase, typename TSupport, typename TProperties, typename TContext2D, typename TSizeProvider>
bool GduContext2DAttachment<TBase, TSupport, TProperties, TContext2D, TSizeProvider>::Upload(Int displayId, Support& support, Properties& properties)
{
    if (m_uploaded) {
        return true;
    }

    if (!Base::Upload(displayId, support, properties)) {
        return false;
    }

    Int width = Util::Select<Support, Properties, SizeProvider>::Get(support, properties).GetMsaaWidth();
    Int height = Util::Select<Support, Properties, SizeProvider>::Get(support, properties).GetMsaaHeight();

    m_context2D.SetSize(width, height);
    if(!m_context2D.Upload()) {
        Base::Unload();
        return false;
    }

    bool createdOffscreenSurfaces =
        RenderDevice2D::UploadOffscreenSurfaces(
            Get2DContextHandle(),
            (UInt32)width,
            (UInt32)height);
    if (!createdOffscreenSurfaces) {
        m_context2D.Unload();
        Base::Unload();

        return false;
    }

    m_uploaded = true;

    return true;
}

template <typename TBase, typename TSupport, typename TProperties, typename TContext2D, typename TSizeProvider>
void GduContext2DAttachment<TBase, TSupport, TProperties, TContext2D, TSizeProvider>::ApplyChanges(Properties& properties)
{
    if (!m_uploaded) {
        return;
    }
    Base::ApplyChanges(properties);
}

template <typename TBase, typename TSupport, typename TProperties, typename TContext2D, typename TSizeProvider>
void GduContext2DAttachment<TBase, TSupport, TProperties, TContext2D, TSizeProvider>::Unload()
{
    if (!m_uploaded){
        return;
    }

    if (!ActivateContext()){
        return;
    }

    RenderDevice2D::UnloadOffscreenSurfaces(Get2DContextHandle());
    m_context2D.Unload();

    Base::Unload();

    m_uploaded = false;
}

}

#endif
