//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_RENDERTARGET_BASE_H)
#define GDU_RENDERTARGET_BASE_H

#include <CanderaPlatform/Device/Common/Base/ContextDelegator3D.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>

#include <CanderaPlatform/Device/Common/Base/RenderTargetListenerCollection.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduBaseRenderTarget represents a proper object to use as in a GduBase class.
 *
 * This class aggregates a few objects and implements the interfaces requried to turn non-abstract
 * the folowing classes: RenderTarget, Surface, Synchronizable and the common part of RenderTarget2D and RenderTarget3D.
 *
 * I satisfies part of the base requirements for GduWrapperImageSource3D and is the base for GduRenderTarget*Attachment.
 *
 * The aggregate parts must be:
 *  - TBase an object that acts as Context for the rest. Must extend GduContext. Must implement.
 *      bool Upload(Int displayId, TSupport& support, TProperties& properties);
 *      void Unload();
 *      void ApplyChanges(TProperties& properties);
 *  - TFrameBuffer an object that acts as support for drawing. Must implement:
 *       bool Upload(TBase& , TSupport& support, TProperties& properties))
 *       void Unload(TBase& );
 *       void ApplyChanges(TBase& , TProperties& properties);
 *       void SwapBuffers(TBase& );
 *  - TProperties properties used to initialize the frame buffer and the context.
 *  - TSyncObject object used to implement the synchronized interface.
 *  - TSupport support object as defined by GduBase. Must extend GduSupport.
 */
template <
    typename TBase,
    typename TFrameBuffer,
    typename TProperties,
    typename TSyncObject,
    typename TSupport>
class GduBaseRenderTarget : public TBase
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GduBaseRenderTarget::Unload)
    public:

        FEATSTD_TYPEDEF_BASE(TBase);
        typedef TFrameBuffer FrameBuffer;
        typedef TProperties Properties;
        typedef TSyncObject SyncObject;
        typedef TSupport Support;
        typedef Internal::RenderTargetEventListenerCollection<GduBaseRenderTarget> EventListenerCollection;

        /**
         *  Constructs a GduBaseRenderTarget object.
         */
        GduBaseRenderTarget() : m_drawing(false), m_support(0) {}

        /**
         *  Destructs a GduBaseRenderTarget object.
         */
        ~GduBaseRenderTarget() { Unload(); }

        /**
         *  Implements function of class RenderTarget2D or RenderTarget3D.
         */
        void SwapBuffers();

        /**
         *  Uploads this object to video memory.
         *  @param displayId         Display to which this render target is attached.
         *  @param support           Object used as support for this render target.
         *  @param properties        Properties to associate to the current RenderTarget.
         *  @return True if successful, false otherwise.
         */
        bool Upload(Int displayId, Support& support, Properties& properties);

        /**
         *  Unloads this object from video memory.
         */
        void Unload();

        /**
         * Applies properties that can be changed after upload has been conducted.
         * Note: In order to apply changes, this function activates the RenderTarget
         * @param properties    Properties to associate to the current RenderTarget.
         */
        void ApplyChanges(Properties& properties);

        /**
         *  Returns whether the render target is uploaded or not.
         *  @return true if the render target has been uploaded successfully.
         */
        bool IsUploaded() const;

        /**
         *  Returns the associated support object.
         *  @return pointer to the support object. 0 if the object is not uploaded.
         */
        Support* GetSupport() { return m_support; }
        const Support* GetSupport() const { return m_support; }

        /**
         *  Returns the associated framebuffer object.
         *  @return reference to the framebuffer object.
         */
        FrameBuffer& GetFrameBuffer() { return m_frameBuffer; }
        const FrameBuffer& GetFrameBuffer() const { return m_frameBuffer; }

        /**
         *  Returns the associated synchronization object.
         *  @return reference to the synchronization object.
         */
        SyncObject& GetSyncObject() { return m_syncObject; }
        const SyncObject& GetSyncObject() const { return m_syncObject; }

        bool IsDrawing() const { return m_drawing; }
        void SetDrawing(bool enabled) { m_drawing = enabled; }


        /**
         * Add a listener for RenderTarget generated events.
         *
         * Each listener is added only once, and it will be removed after the first call to RemoveRenderTargetEventListener.
         *
         * @param listener Listener for RenderTarget events.
         * @see Candera::RenderTarget::Event
         */
        void AddEventListener(FeatStd::EventListener* listener);

        /**
         * Remove existing listener for RenderTarget generated events.
         *
         * Removing a listener will delete if from the listener list, regardless of the number of times it was previously added.
         *
         * @param listener Listener for RenderTarget events.
         */
        void RemoveEventListener(FeatStd::EventListener* listener);

    protected:
        void NotifyListenersOnBeforeSwapBuffersCalled();
        void NotifyListenersOnAfterSwapBuffersExecuted();

    private:
        bool m_drawing;

        Support* m_support;
        FrameBuffer m_frameBuffer;
        SyncObject m_syncObject;
};

/** @}*/ //end of CommonDevice


template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
bool GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::Upload(Int displayId, Support& support, Properties& properties)
{
    if (IsUploaded()) {
        CANDERA_DEVICE_LOG_WARN("already uploaded");
        return true;
    }
    m_support = &support;
    if(!Base::Upload(displayId, *m_support, properties)) {
        CANDERA_DEVICE_LOG_WARN("base upload failed");
        m_support = 0;
        return false;
    }
    if (!m_frameBuffer.Upload(*this, *m_support, properties)) {
        CANDERA_DEVICE_LOG_WARN("frame buffer upload failed");
        Base::Unload();
        m_support = 0;
        return false;
    }

    return true;
}

template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
void GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::Unload()
{
    if (IsUploaded()) {
        m_frameBuffer.Unload(*this);
        Base::Unload();

        m_support = 0;
        SetDrawing(false);
    }
}

template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
void GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::ApplyChanges(Properties& properties)
{
    if (IsUploaded()) {
        Base::ApplyChanges(properties);
        m_frameBuffer.ApplyChanges(*this, properties);
    }
}

template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
void GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::SwapBuffers()
{
    if (IsUploaded() && !IsDrawing()) {
        m_frameBuffer.SwapBuffers(*this);
    }
}

template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
bool GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::IsUploaded() const
{
    return m_support != 0;
}

template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
void GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::AddEventListener(FeatStd::EventListener* listener)
{
    Candera::Internal::InternalEventSourceAccess<Candera::RenderTarget>::GetEventSource().AddEventListener(listener);
}

template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
void GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::RemoveEventListener(FeatStd::EventListener* listener)
{
    Candera::Internal::InternalEventSourceAccess<Candera::RenderTarget>::GetEventSource().RemoveEventListener(listener);
}

template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
void GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::NotifyListenersOnBeforeSwapBuffersCalled()
{
    Internal::RenderTargetEventListenerCollection<RenderTarget>::TEvent<Internal::RenderTargetEventListenerCollection<RenderTarget>::BeforeSwap> ev(static_cast<RenderTarget*>(this));
    Candera::Internal::InternalEventSourceAccess<Candera::RenderTarget>::GetEventSource().DispatchEvent(ev);
}

template <typename TBase, typename TFrameBuffer, typename TProperties, typename TSyncObject, typename TSupport>
void GduBaseRenderTarget<TBase, TFrameBuffer, TProperties, TSyncObject, TSupport>::NotifyListenersOnAfterSwapBuffersExecuted()
{
    Internal::RenderTargetEventListenerCollection<RenderTarget>::TEvent<Internal::RenderTargetEventListenerCollection<RenderTarget>::AfterSwap> ev(static_cast<RenderTarget*>(this));
    Candera::Internal::InternalEventSourceAccess<Candera::RenderTarget>::GetEventSource().DispatchEvent(ev);
}

}

#endif
