//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_BASE_H)
#define GDU_BASE_H

#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnitMetaInfo.h>

#include <Candera/System/MemoryManagement/TypeTraits.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief The class GduBase a generic GraphicDeviceUnit, that wraps a class providing the required
 * functionality. Conversion to Render Targets and Image Sources is done via GduAttachment.
 *
 * This template class wraps an object that needs to contain the following methods:
 *  bool Upload(Int displayId, TSupport& support, TProperties& properties);
 *  void ApplyChanges(TProperties&);
 *  void Unload();
 * The support object needs to implement the following methods:
 *  bool Upload(Int displayId, TSupportProperties& properties)) {
 *  void ApplyChanges(TSupportProperties& properties);
 *  void Unload();
 * The base object is initialized after the support object, and it may use the base object
 * while uploading, to retrieve specific support properties. (e.g. an EglNativeWindowHandle)
 */
template <
    class TObject,
    class TProperties,
    class TSupport,
    class TSupportProperties,
    GraphicDeviceUnitTypeHandle TUnitType>
class GduBase : public GraphicDeviceUnit
{
public:
    typedef TObject Object;
    typedef TProperties Properties;
    typedef TSupport Support;
    typedef TSupportProperties SupportProperties;

    static const GraphicDeviceUnitTypeHandle c_unitType = TUnitType;

    /**
     * Constructor.
     */
    GduBase()
    {
    }

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * Applies properties that can be changed after upload has been conducted.
     * Note: In order to apply changes, this function might activate the RenderTarget associated.
     */
    virtual void ApplyChanges()
    {
        m_support.ApplyChanges(GetSupportProperties());
        m_object.ApplyChanges(GetProperties());
    }

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * @return true if upload was successful
     */
    virtual bool Upload()
    {
        if (!IsUploadPresequisiteFullfiled()) {
            return false;
        }

        if (!m_support.Upload(GetDisplay(), GetSupportProperties())) {
            return false;
        }

        if (!m_object.Upload(GetDisplay(), GetSupport(), GetProperties())) {
            m_support.Unload();
            return false;
        }

        ApplyChanges();
        return true;
    }

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     */
    virtual void Unload()
    {
        m_object.Unload();
        m_support.Unload();
    }

    /**
     *  Overrides virtual function of class GraphicDeviceUnit.
     *  @return   Pointer to the support object.
     */
    virtual Window* ToWindow() { return &m_support; }

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * @return handle of this Gdu.
     */
    virtual GraphicDeviceUnitTypeHandle GetUnitType() const { return c_unitType; }

    /**
     * Retrieves the object at the base of this Gdu.
     * @return reference of the object.
     */
    Object& GetObject() { return m_object; }
    const Object& GetObject() const { return m_object; }

    /**
     * Retrieves the main properties of the object at the base of this Gdu.
     * @return reference of the properties.
     */
    Properties& GetProperties() { return m_properties; }
    const Properties& GetProperties() const { return m_properties; }

    /**
     * Retrieves the object that plays support for the object at the base of this Gdu.
     * @return reference of the support object.
     */
    Support& GetSupport() { return m_support; }
    const Support& GetSupport() const { return m_support; }

    /**
     * Retrieves the properties of the support object.
     * @return reference of the support properties.
     */
    SupportProperties& GetSupportProperties() { return m_supportProperties; }
    const SupportProperties& GetSupportProperties() const { return m_supportProperties; }

protected:
    //parameters used by the GduAttachmentSelector
    static const Int c_Rt2DId = -1;
    static const Int c_Rt3DId = -1;
    static const Int c_Is2DId = -1;
    static const Int c_Is3DId = -1;


    virtual Internal::GraphicDeviceUnitOwnerAccess GetOwnerAccess() override
    {
        return Internal::GraphicDeviceUnitOwnerAccess::GetOwnerAccess(GetProperties());
    }

    virtual void SetOwner(RenderTarget3D* owner)
    {
        Internal::GraphicDeviceUnitOwnerAccess::SetOwner(GetProperties(), owner);
    }

private:
    SupportProperties m_supportProperties;
    Properties m_properties;
    Support m_support;
    Object m_object;
};

/** @}*/ //end of Device

}

#endif
