//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(EGL_WINDOW_SURFACE_PROVIDER_H)
#define EGL_WINDOW_SURFACE_PROVIDER_H

#include <CanderaPlatform/Device/Common/Internal/EGL/EglContextCreator.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglWindowSurfaceProperties.h>

#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>
#include <CanderaPlatform/Device/Common/EGL/EglTypeMapper.h>
#include <CanderaPlatform/Device/Common/EGL/EglWrapper.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief EglWindowSurfaceProvider represents a ContextProvider that doesn't have a surface and it gets its
 * surface and configuration from a surface provider.
 */
template <
    typename TDisplayProvider,
    typename TSupport
    >
class EglWindowSurfaceProvider : public EglContextCreator
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::EglWindowSurfaceProvider::Unload)

    FEATSTD_TYPEDEF_BASE(EglContextCreator);
    typedef TDisplayProvider DisplayProvider;
    typedef TSupport Support;
    typedef EglWindowSurfaceProperties Properties;

    public:
        /**
         *  Constructs a EglWindowSurfaceProvider object.
         */
        EglWindowSurfaceProvider();

        /**
         *  Uploads this object to video memory. Required by GduBaseRenderTarget.
         *  @param displayId        The displayId
         *  @param support          Unused
         *  @param properties       Unused
         *  @return True if successful, false otherwise.
         */
        bool Upload(Int displayId, Support& support, Properties& properties);

        /**
         *  Unloads this object from video memory. Required by GduBaseRenderTarget.
         */
        void Unload();

        /**
         *  Applies swap interval. Required by GduBaseRenderTarget.
         *  @param properties Used to retrieve new swap interval.
         */
        void ApplyChanges(const Properties& properties);

        /**
         *  Retrieves the associated surface config. Overwrites pure method from EglChain
         *  @return handle of the associated EGLConfig.
         */
        virtual EGLConfig GetConfig() const { return m_config; }

        /**
         *  Retrieves the surface used for reading operations. Overwrites pure method from EglChain
         *  @return handle of the associated EGLSurface.
         */
        virtual EGLSurface GetReadSurface() const { return m_surface; }

        /**
         *  Retrieves the surface used for writing operations. Overwrites pure method from EglChain
         *  @return handle of the associated EGLSurface.
         */
        virtual EGLSurface GetDrawSurface() const { return m_surface; }

        /**
         *  Retrieves the associated GetDisplay(). Overwrites pure method from EglChain
         *  @return handle of the associated EGLDisplay.
         */
        virtual EGLDisplay GetDisplay() const;

    protected:
        /*
         *  Non-virtual. An object should never be distroyed from a pointer to this class.
         */
        ~EglWindowSurfaceProvider() { Unload(); }

    private:
        EGLConfig m_config;
        EGLSurface m_surface;

        DisplayProvider m_displayProvider;
};

/** @}*/ //end of CommonDevice


template <typename TDisplayProvider, typename TSupport>
EglWindowSurfaceProvider<TDisplayProvider, TSupport>::EglWindowSurfaceProvider() :
    m_config(0),
    m_surface(EGL_NO_SURFACE)
{
}

template <typename TDisplayProvider, typename TSupport>
bool EglWindowSurfaceProvider<TDisplayProvider, TSupport>::Upload(Int displayId, Support& support, Properties& properties)
{
    if (GetDisplay() != EGL_NO_DISPLAY) {
        CANDERA_DEVICE_LOG_INFO("already uploaded");
        return true;
    }

    m_displayProvider.Upload(displayId, support);
    if (GetDisplay() == EGL_NO_DISPLAY) {
        CANDERA_DEVICE_LOG_WARN("wrong GetDisplay()");
        m_displayProvider.Unload();
        return false;
    }

    EGLNativeWindowType window = support.GetWindow();
    if (window == 0) {
        CANDERA_DEVICE_LOG_WARN("wrong support");
        m_displayProvider.Unload();
        return false;
    }

    EGLint windowAttributes[70];
    EglTypeMapper::MapEglConfiguration(windowAttributes, properties.GetConfiguration().GetConfigurationAttributes());

    EGLint windowSurfaceAttributes[] =
    {
        EGL_RENDER_BUFFER, EGL_BACK_BUFFER,
        EGL_NONE
    };

    EGLBoolean success;

    EGLint numOfConfigs = 0;
    success = eglChooseConfig(GetDisplay(), windowAttributes, &m_config, 1, &numOfConfigs);
    CANDERA_DEVICE_CHECK_AND_LOG_INFO(Egl, "eglChooseConfig");

    if ((success != EGL_TRUE) || (numOfConfigs <= 0)) {
        CANDERA_DEVICE_LOG_WARN("no valid config found");
        m_displayProvider.Unload();
        return false;
    }


    EGLDisplay dpy = GetDisplay();
    /* Show attributes of config[0]. */
    {
        EGLint value, r, g, b, a; 

        CANDERA_DEVICE_LOG_INFO("\nEGL config #%d:\n", 1);
        eglGetConfigAttrib(dpy, m_config, EGL_BUFFER_SIZE, &value);
        CANDERA_DEVICE_LOG_INFO("EGL_BUFFER_SIZE: %d\n", value);
        eglGetConfigAttrib(dpy, m_config, EGL_RED_SIZE,   &r);
        eglGetConfigAttrib(dpy, m_config, EGL_GREEN_SIZE, &g);
        eglGetConfigAttrib(dpy, m_config, EGL_BLUE_SIZE,  &b);
        eglGetConfigAttrib(dpy, m_config, EGL_ALPHA_SIZE, &a);
        CANDERA_DEVICE_LOG_INFO("EGL_R/G/B/A_SIZE: %d/%d/%d/%d\n", r, g, b, a);
        eglGetConfigAttrib(dpy, m_config, EGL_DEPTH_SIZE, &value);
        CANDERA_DEVICE_LOG_INFO("EGL_DEPTH_SIZE: %d\n", value);
        eglGetConfigAttrib(dpy, m_config, EGL_CONFIG_ID, &value);
        CANDERA_DEVICE_LOG_INFO("EGL_CONFIG_ID:  %d\n", value);
    }

    m_surface = eglCreateWindowSurface(GetDisplay(), m_config, support.GetWindow(), windowSurfaceAttributes);
    CANDERA_DEVICE_CHECK_AND_LOG_INFO(Egl, "eglCreateWindowSurface");

    if (m_surface == EGL_NO_SURFACE) {
        CANDERA_DEVICE_LOG_WARN("failed to create surface");
        m_displayProvider.Unload();
        return false;
    }

    success = Base::Upload();
    if (!success) {
        CANDERA_DEVICE_LOG_WARN("context creation failed");
        eglDestroySurface(GetDisplay(), m_surface);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Egl, "eglDestroySurface");
        m_surface = EGL_NO_SURFACE;
        m_displayProvider.Unload();
        return false;
    }
    return true;
}

template <typename TDisplayProvider, typename TSupport>
void EglWindowSurfaceProvider<TDisplayProvider, TSupport>::Unload()
{
    if (GetDisplay() == EGL_NO_DISPLAY) {
        return;
    }

    Base::Unload();

    if (m_surface != EGL_NO_SURFACE) {
        Internal::EGLWrapper::GetInstance().MakeCurrent(GetDisplay(), 0, 0 ,0);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Egl, "EGLWrapper::GetInstance().MakeCurrent");
        eglDestroySurface(GetDisplay(), m_surface);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Egl, "eglDestroySurface");
        m_surface = EGL_NO_SURFACE;
    }

    m_displayProvider.Unload();
}

template <typename TDisplayProvider, typename TSupport>
void EglWindowSurfaceProvider<TDisplayProvider, TSupport>::ApplyChanges(const Properties& properties)
{
    if (ActivateContext()) {
        eglSwapInterval(GetDisplay(), properties.GetSwapInterval());
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Egl, "eglSwapInterval");
    }
}

template <typename TDisplayProvider, typename TSupport>
EGLDisplay EglWindowSurfaceProvider<TDisplayProvider, TSupport>::GetDisplay() const
{
    return m_displayProvider.GetDisplay();
}

}

#endif
