//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(EGL_EXTENSIONS_H)
#define EGL_EXTENSIONS_H


#include <FeatStd/Container/Vector.h>
#include <FeatStd/Util/ConvenienceMacros.h>
#include <FeatStd/Util/String.h>

#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/Extensions/EglExtension.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/Extensions/EglExtensionProcedure.h>


namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */


/**
 * @brief Singleton class providing access to EGL extensions procedures.
 */
class EglExtensions
{
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);

public:
    // Types

#if defined(CANDERA_DEVICE_EMERALD)
    typedef EGLint ImageAttributeElement;
#else // defined(CANDERA_DEVICE_EMERALD)
    typedef const EGLint ImageAttributeElement;
#endif // !defined(CANDERA_DEVICE_EMERALD)


    // Interface

    /**
     *  Get the instance of this singleton class. Not thread safe.
     *  @return Reference to the instance of this singleton class.
     */
    static EglExtensions& GetInstance();

    /**
     * @return True if EGL extension with given name exists, false otherwise.
     */
    bool IsSupported(const FeatStd::Char* name) const;


#if EGL_KHR_image_base > 0
    /**
     * Wrapper for eglCreateImageKHR() from extension "EGL_KHR_image_base".
     */

    EGLImageKHR EglCreateImageKhr(EGLDisplay dpy, EGLContext ctx, EGLenum target, EGLClientBuffer buffer,
        ImageAttributeElement* attributeList);

    /**
     * Wrapper for eglDestroyImageKHR() from extension "EGL_KHR_image_base".
     */
    EGLBoolean EglDestroyImageKhr(EGLDisplay dpy, EGLImageKHR image);
#endif // EGL_KHR_image_base > 0

#if GL_OES_EGL_image_external > 0
    /**
     * Wrapper for glEGLImageTargetTexture2DOES() from extension "GL_OES_EGL_image".
     */
    void GlEglImageTargetTexture2Does(GLenum target, GLeglImageOES image);
#endif // GL_OES_EGL_image_external > 0


private:
    // Singleton class

    FEATSTD_MAKE_CLASS_UNCOPYABLE(EglExtensions);
    FEATSTD_MAKE_CLASS_STATIC(EglExtensions);


    // Types

#if EGL_KHR_image_base > 0
    class EglKhrImageBaseExtension : public EglExtension
    {
    public:
        FEATSTD_SUPPRESS_MSC_WARNING_BEGIN(4355, "Only reference is set, but not used inside constructor.")
        EglKhrImageBaseExtension()
        :   EglExtension("EGL_KHR_image_base"),
            eglCreateImageKhr(*this, "eglCreateImageKHR"),
            eglDestroyImageKhr(*this, "eglDestroyImageKHR")
        { }
        FEATSTD_SUPPRESS_MSC_WARNING_END()

        EglExtensionProcedure<PFNEGLCREATEIMAGEKHRPROC, EGLImageKHR> eglCreateImageKhr;
        EglExtensionProcedure<PFNEGLDESTROYIMAGEKHRPROC, EGLBoolean> eglDestroyImageKhr;
    };
#endif // EGL_KHR_image_base > 0

#if GL_OES_EGL_image_external > 0
    class GlOesEglImageExtension : public EglExtension
    {
    public:
        FEATSTD_SUPPRESS_MSC_WARNING_BEGIN(4355, "Only reference is set, but not used inside constructor.")
        GlOesEglImageExtension()
        :   EglExtension("GL_OES_EGL_image"),
            glEglImageTargetTexture2Does(*this, "glEGLImageTargetTexture2DOES")
        { }
        FEATSTD_SUPPRESS_MSC_WARNING_END()

        EglExtensionProcedure<PFNGLEGLIMAGETARGETTEXTURE2DOESPROC, void> glEglImageTargetTexture2Does;
    };
#endif // GL_OES_EGL_image_external > 0

    typedef FeatStd::Internal::Vector<FeatStd::String> SupportedExtensionsList;


    // Implementation

    void AddAvailableExtension(const FeatStd::Char*& beginPtr, const FeatStd::Char* endPtr);
    void ParseExtensions(const FeatStd::Char* extensions);
    void QueryAvailableExtensions();


    // Attributes

    static const FeatStd::Char c_extensionsNamesSeparator;

    SupportedExtensionsList m_supportedExtensions;

#if EGL_KHR_image_base > 0
    EglKhrImageBaseExtension m_eglKhrImageBaseExtension;
#endif // EGL_KHR_image_base > 0
#if GL_OES_EGL_image_external > 0
    GlOesEglImageExtension m_glOesEglImageExtension;
#endif // GL_OES_EGL_image_external > 0
};


/** @}*/ //end of CommonDevice

} // namespace Candera


#endif // defined(EGL_EXTENSIONS_H)
