//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/Util/PointerUtil.h>
#include <FeatStd/Util/StaticObject.h>

#include <CanderaPlatform/Device/Common/EGL/EglWrapper.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglExtensions.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/Extensions/EglExtensionProcedureCallDispatcher.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlTrace.h>


using namespace FeatStd;


namespace Candera {


using namespace Internal;


// Static attributes and constants


const Char EglExtensions::c_extensionsNamesSeparator = ' ';


// Construction


EglExtensions::EglExtensions()
{
    QueryAvailableExtensions();
}

EglExtensions& EglExtensions::GetInstance()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(EglExtensions, implementation);
    return implementation;
}


bool EglExtensions::IsSupported(const FeatStd::Char* name) const
{

    for (SupportedExtensionsList::ConstIterator it = m_supportedExtensions.ConstBegin(); it != m_supportedExtensions.ConstEnd(); ++it){

        Char* a = const_cast<Char*>((*it).GetCString());
        Char* b = const_cast<Char*>(name);
        while (a != 0 && *a == ' ')
            a++;
        while (b != 0 && *b == ' ')
            b++;

        if (0 == strcmp(a, b)){
            return true;
        }
    }
    return false; // m_supportedExtensions.Contains(FeatStd::String(name));
}

// Interface


#if EGL_KHR_image_base > 0

EGLImageKHR EglExtensions::EglCreateImageKhr(EGLDisplay dpy, EGLContext ctx, EGLenum target,
    EGLClientBuffer buffer, ImageAttributeElement* attributeList)
{
    FEATSTD_DEBUG_ASSERT(dpy != 0);
    FEATSTD_DEBUG_ASSERT(attributeList != 0);
    EglExtensionProcedureCallDispatcher5<EGLDisplay, EGLContext, EGLenum, EGLClientBuffer, ImageAttributeElement*,
        /*Result*/EGLImageKHR> arguments(dpy, ctx, target, buffer, attributeList);
    return m_eglKhrImageBaseExtension.eglCreateImageKhr.Call(arguments);
}

EGLBoolean EglExtensions::EglDestroyImageKhr(EGLDisplay dpy, EGLImageKHR image)
{
    FEATSTD_DEBUG_ASSERT(dpy != 0);
    FEATSTD_DEBUG_ASSERT(image != 0);
    EglExtensionProcedureCallDispatcher2<EGLDisplay, EGLImageKHR, /*Result*/EGLBoolean> arguments(dpy, image);
    return m_eglKhrImageBaseExtension.eglDestroyImageKhr.Call(arguments);
}

#endif // EGL_KHR_image_base > 0

#if GL_OES_EGL_image_external > 0

void EglExtensions::GlEglImageTargetTexture2Does(GLenum target, GLeglImageOES image)
{
    FEATSTD_DEBUG_ASSERT(image != 0);
    EglExtensionProcedureCallDispatcher2<GLenum, GLeglImageOES, /*Result*/void> arguments(target, image);
    static_cast<void>(m_glOesEglImageExtension.glEglImageTargetTexture2Does.Call(arguments));
}

#endif // GL_OES_EGL_image_external > 0


// Implementation


void EglExtensions::AddAvailableExtension(const Char*& beginPtr, const Char* endPtr)
{
    FEATSTD_DEBUG_ASSERT(beginPtr < endPtr);
    const size_t c_maxExtensionNameLength = 50;
    Char buffer[c_maxExtensionNameLength];
    FEATSTD_DEBUG_ASSERT(SizeType(endPtr - beginPtr + 1/*nul*/) < c_maxExtensionNameLength); // increase c_maxExtensionNameLength if this fails
    //string has to be trimmed as there may be leading and trailing spaces after tokenization.

    // copy begin pointer and move on until the first significant character (not space) is reached.
    // precondition: no null pointers between begin and end
    const Char* first = beginPtr;
    const Char* onePastLast = endPtr;
        FEATSTD_LINT_NEXT_EXPRESSION(946, "comparing addresses is intended here")
        while (first < onePastLast && *first == ' '){
            FEATSTD_LINT_NEXT_EXPRESSION(947, "pointer arithmetic is intended here")
            ++first;
        }
        FEATSTD_LINT_NEXT_EXPRESSION(946, "comparing addresses is intended here")
        while ( onePastLast > first && *(onePastLast -1) == ' ') {// if there is some string left trim trailing spaces
            FEATSTD_LINT_NEXT_EXPRESSION(947, "pointer arithmetic is intended here")
            --onePastLast;
            }


    const OffsetType length = FeatStd::Internal::PointerByteDiff(onePastLast, first);
    FeatStd::Internal::Memory::Copy(/*destination*/buffer, first, length);
    buffer[length] = '\0'; //forcefully terminate buffer
    // CANDERA_DEVICE_LOG_DEBUG("EGL extension: '%s'", buffer); // commented out - slows down MULTI, clogs logs with rarely changed info
    if (!m_supportedExtensions.Add(String(buffer))) {
        FEATSTD_DEBUG_FAIL();
    }

    beginPtr = endPtr; //token consumed
}

void EglExtensions::ParseExtensions(const Char* extensions)
{
    const Char* currentPtr = extensions;
    const Char* beginPtr = currentPtr;
    for (; *currentPtr != '\0'; ++currentPtr) {
        if (*currentPtr == c_extensionsNamesSeparator) {
            AddAvailableExtension(beginPtr, currentPtr);
        }
    }
    FEATSTD_DEBUG_ASSERT(beginPtr <= currentPtr);
    if (beginPtr != currentPtr) {
        AddAvailableExtension(beginPtr, currentPtr);
    }
}

void EglExtensions::QueryAvailableExtensions()
{
    // query GL extensions
    const GLubyte* extensions1 = glGetString(GL_EXTENSIONS);
    CANDERA_DEVICE_CHECK_AND_LOG_ERROR(Gles, "glGetString(GL_EXTENSIONS)");
    if (extensions1 != 0) {
        ParseExtensions(FeatStd::Internal::PointerToPointer<const Char*>(extensions1));
    }
    // query EGL extensions
    const Char* extensions2 = static_cast<const Char*>(eglQueryString(EGLWrapper::GetInstance().GetCurrentDisplay(), EGL_EXTENSIONS));
    CANDERA_DEVICE_CHECK_AND_LOG_ERROR(Egl, "eglQueryString(EGL_EXTENSIONS)");
    if (extensions2 != 0) {
        ParseExtensions(extensions2);
    }
}


} // namespace Candera
