//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CONTEXT_2D_PROVIDER_H)
#define CONTEXT_2D_PROVIDER_H

#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>

namespace Candera {
template <class _Context>
class Context2DFeeder{
    public:
    typedef _Context BaseContext;

    Context2DFeeder() : first(0) {}
    ~Context2DFeeder() {
        while(first) {
            first->Unload();
            ContextWrapper* next = first->next;
            FEATSTD_DELETE(first);
            first = next;
        }
    }

    /**
     *  Create a context handle for the specified RenderTarget2D, so that one can render
     *  2D context using RenderDevice2D. ContextHandle should be destroyed before destroying the
     *  render target.. Should most probably be called when uploading the render target, so that other
     *  external contexts (for instance EGL context) is properly set.
     *
     *  @param renderTarget Pointer to RenderTarget2D for which the context handle shall be created.
     *
     *  @return ContextHandle2D for rendering using RenderDevice2D.
     */
    ContextHandle2D CreateRenderContext2D(const RenderTarget2D* renderTarget) {
        ContextWrapper* wrapper = FEATSTD_NEW(ContextWrapper);
        if (wrapper == 0) {
            return 0;
        }

        if (renderTarget) {
            wrapper->SetSize(renderTarget->GetWidth(), renderTarget->GetHeight());
            wrapper->source = renderTarget;
        }
        if (!wrapper->Upload()) {
            FEATSTD_DELETE(wrapper);
            return 0;
        }

        wrapper->next = first;
        if (first != 0) {
            first->previous = wrapper;
        }
        first = wrapper;

        return reinterpret_cast<ContextHandle2D>(static_cast<BaseContext*>(wrapper));
    }

    /**
     *  Destroy the context associated with the handle.
     *
     *  @param context ContextHandle2D to be destroyed.
     */
    void DestroyRenderContext2D(ContextHandle2D context) {
        ContextWrapper* wrapper = static_cast<ContextWrapper*>(reinterpret_cast<BaseContext*>(context));
        if (context == 0 || wrapper == 0) {
            return;
        }

        wrapper->Unload();

        if (wrapper->previous) {
            FEATSTD_DEBUG_ASSERT(wrapper->previous->next == wrapper);
            wrapper->previous->next = wrapper->next;
        }
        else {
            FEATSTD_DEBUG_ASSERT(first == wrapper);
            first = wrapper->next;
        }
        if (wrapper->next) {
            FEATSTD_DEBUG_ASSERT(wrapper->next->previous == wrapper);
            wrapper->next->previous = wrapper->previous;
        }

        FEATSTD_DELETE(wrapper);
    }

    /**
     *  Refresh context properties, based on the new render target properties.
     *  Should be called each time the render target modifies its properties, like width and height.
     *
     *  @param context ContextHandle2D that should be updated.
     *
     *  @return         Boolean value indicating whether updating succeeded or not.
     *                  False if context is null or Update() failed (returned false).
     *                  True otherwise.
     */
    bool UpdateRenderContext2D(ContextHandle2D context) {
        ContextWrapper* wrapper = static_cast<ContextWrapper*>(reinterpret_cast<BaseContext*>(context));
        if (context == 0 || wrapper == 0) {
            return false;
        }

        if (wrapper->source) {
            wrapper->SetSize(wrapper->source->GetWidth(), wrapper->source->GetHeight());
        }
        if (!wrapper->Update()) {
            return false;
        }

        return true;
    }

    private:

    class ContextWrapper : public BaseContext{
        public:
        ContextWrapper *previous;
        ContextWrapper *next;

        const RenderTarget2D* source;

        ContextWrapper() : previous(0), next(0), source(0) {}
    };

    ContextWrapper* first;
};
} //namespace Candera

#endif //CONTEXT_2D_PROVIDER_H
