//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BlockContainer_H)
#define CANDERA_BlockContainer_H

#include <Candera/Environment.h>
#include <FeatStd/Diagnostics/Debug.h>

#include <Candera/System/MemoryManagement/CanderaHeap.h>

namespace Candera { namespace Internal {
/**
 *  @brief BlockContainer stores objects in stationary blocks so that references remain valid
 *  for the full lifetime of the object. Objects may be accessed sequentially. Inserting and
 *  removing is done only at the end, and can be done in batches of objects.
 *
 *  @param  T       Defines the type of the contained elements. The type needs to have a copy-constructor
 *                  and has to be assignable. A default constructor is not necessarily needed.
 **/
template<typename T>
class BlockContainer
{
    // forward declaration
    template <typename X> class IteratorBase;
    class Block;

    public:
        typedef IteratorBase<T> Iterator;
        typedef IteratorBase<const T> ConstIterator;

        /**
         *  Constructs a BlockContainer with no memory allocated.
         *  Use Reserve() to allocate a proper amount of memory.
         *  Initialization of the memory takes place when elements are added to the BlockContainer.
         */
        BlockContainer() :
            m_size(0),
            m_capacity(0),
            m_listHead(0),
            m_listTail(0),
            m_listEnd(0)
        {
        }

        /**
         *  Destructor calls the destructor for all Size() elements in the BlockContainer
         *  and deallocates the memory.
         *  Note: Non-Virtual, not intended as base class.
         */
        ~BlockContainer()
        {
            Clear();
            Reserve(0);
        }

        /**
         *  Copy constructor.
         *  Create a new BlockContainer with the same properties and elements as the copied object.
         *  @param other BlockContainer object that is copied.
         */
        // Copy Constructor is not supported due to undefined behavior when memory allocation fails.
        BlockContainer(const BlockContainer& other) :
            m_size(0),
            m_capacity(0),
            m_listHead(0),
            m_listTail(0),
            m_listEnd(0)
        {
            if (!Reserve(other.m_capacity)) { // Adapt capacity to the capacity of the copied Vector.
                // no other way to report memory allocation problem.
                FEATSTD_DEBUG_FAIL();
            }
            ConstIterator otherIt = other.ConstBegin();
            while (otherIt != other.ConstEnd()) {
                Add(*otherIt++);
            }
        }

        /**
         *  Assignment operator for assignments of other BlockContainer objects.
         *  Important note: Object remains unchanged if memory allocation fails.
         *  @param other BlockContainer that is assigned.
         *  @return This object with assigned properties.
         */
        BlockContainer& operator=(const BlockContainer& other) {
            if (this != &other) { // Adapt capacity to the capacity of the assigned BlockContainer
                if (!Reserve(other.m_capacity)) {
                    // xxx: Trace.
                    return *this;
                }
                Iterator it = Begin();
                ConstIterator otherIt = other.ConstBegin();
                // Overwrite the old existing elements with new ones.
                while ((it != End()) && (otherIt != other.ConstEnd())) {
                    *it++ = *otherIt++;
                }
                // Destroy any surplus elements.
                RemoveFrom(it);
                // Add new elements to the end
                while (otherIt != other.ConstEnd()) {
                    Add(*otherIt++);
                }

                // Allocator is not copied.
            }
            return *this;
        }

        /**
         *  Adds a new elements to the end of the BlockContainer and reallocates memory if necessary.
         *  @param element Element that should be attached.
         *  @return An iterator to the new element or End() if memory allocation failed.
         */
        Iterator Add(const T& element)
        {
            if (m_size == m_capacity) {
                SizeType newCapacity = (m_capacity > 0) ? m_capacity * c_MemoryExpansionRate : c_MemoryInitialCapacity;
                if (!Reserve(newCapacity)) {
                    return End();
                }
            }
            // Use placement new for adding new elements which doesn't allocate new memory.
            FEATSTD_DEBUG_ASSERT( m_listEnd != 0 );

            Iterator location = Iterator(m_listEnd, m_listEnd->m_size);
            MemoryManagement::Construct(&*location, element);

            m_listEnd->m_size ++;
            if (m_listEnd->m_size == m_listEnd->m_capacity) {
                m_listEnd = m_listEnd->m_next;
                FEATSTD_DEBUG_ASSERT((m_listEnd == 0) || (m_listEnd->m_size == 0));
            }

            m_size ++;
            return location;
        }

        /**
         *  Appends elements to the end of the container.
         *  The capacity is only increased as required.
         *  @param size Number of elements to append.
         *  @param element Value assigned to new elements.
         *  @return An iterator to the first element or End() if memory allocation failed or size is invalid.
         */
        Iterator Append(SizeType size, const T& element = T())
        {
            if (size <= 0) {
                return End();
            }
            if (m_capacity < m_size + size) {
                SizeType newCapacity = (m_capacity > 0) ? m_capacity * c_MemoryExpansionRate : c_MemoryInitialCapacity;
                newCapacity = (newCapacity > m_size + size) ? newCapacity : m_size + size;
                if (!Reserve(newCapacity)) {
                    return End();
                }
            }

            Iterator first = Add(element);
            for (SizeType i = 1; i < size; i ++) {
                Add(element);
            }

            return first;
        }

        /**
         *  Clear all elements from the BlockContainer and call their destructor.
         *  Clear doesn't deallocate any memory. After this operation
         *  the new size of the BlockContainer is 0.
         */
        void Clear()
        {
            RemoveFrom(Begin());
        }

        /**
         *  Returns the number of elements stored in the BlockContainer.
         *  @return Number of elements in the BlockContainer.
         */
        SizeType Size() const
        {
            return m_size;
        }

        /**
         *  Returns the capacity of the BlockContainer.
         *  The capacity defines the number of element for which memory is already allocated.
         *  Note: Use Reserve to change the capacity of the BlockContainer.
         *  @return Capacity of the BlockContainer.
         */
        SizeType GetCapacity() const
        {
            return m_capacity;
        }

        /**
         *  Return true if the BlockContainer is empty.
         *  @return True, if the BlockContainer doesn't contain any elements.
         */
        bool Empty() const
        {
            return (m_size == 0);
        }

        /**
         *  Changes the amount of memory reserved for the BlockContainer.
         *  New blocks or created or old blocks are destroyed as required.
         *  No elements are destructed or moved.
         *  @param capacity New capacity of the BlockContainer.
         *  @return False, if memory allocation failed or capacity parameter is invalid (negative).
         */
        bool Reserve(SizeType capacity)
        {
            SizeType availableCapacity = MemoryManagement::PlatformHeap::Available() / MemoryManagement::ArrayByteSize<T>(1);
            if (availableCapacity < capacity) {
                // xxx: Available capacity returns a value that is too small.
                // If memory allocation fails it still will be discovered when the returned pointer is checked != 0.
                //return false;
            }

            if (capacity < m_capacity) {
                // Shrink.
                SizeType dif = m_capacity - capacity;
                // Delete all empty blocks as long as they are smaller than the requested shrinkage.
                while ((m_listTail != 0) && (m_listTail->m_size == 0) && (dif >= m_listTail->m_capacity)) {
                    dif -= m_listTail->m_capacity;
                    m_capacity -= m_listTail->m_capacity;

                    Block* block = m_listTail->m_prev;
                    if (m_listEnd == m_listTail) {
                        m_listEnd = 0;
                    }
                    MemoryManagement::PlatformHeap::Free(m_listTail->m_memory);
                    FEATSTD_DELETE(m_listTail);

                    m_listTail = block;
                    if (m_listTail != 0) {
                        m_listTail->m_next = 0;
                    }
                }
                if (m_listTail == 0) {
                    m_listHead = 0;
                    m_listEnd = 0;
                }
            }
            else if (capacity > m_capacity) {
                // Increase.
                SizeType dif = capacity - m_capacity;

                // Create a new memory block of the required size.
                T* newMemory = 0;
                newMemory = reinterpret_cast<T*>(FEATSTD_ALLOC(static_cast<UInt32>(MemoryManagement::ArrayByteSize<T>(dif))));
                if (newMemory == 0) {
                    return false;
                }

                // Create a block object.
                Block* block = FEATSTD_NEW(Block);
                if (block == 0) {
                    MemoryManagement::PlatformHeap::Free(newMemory);
                    return false;
                }

                // Increase container capacity.
                m_capacity += dif;

                // Link the block object to the memory block.
                block->m_memory = newMemory;
                block->m_capacity = dif;

                // Link the block object to end of the block list.
                if (m_listTail != 0) {
                    block->m_prev = m_listTail;
                    m_listTail->m_next = block;
                    if (m_listEnd == 0) {
                        m_listEnd = block;
                    }
                }
                else {
                    FEATSTD_DEBUG_ASSERT((m_listHead == 0) && (m_listEnd == 0));
                    m_listHead = block;
                    m_listEnd = block;
                }

                m_listTail = block;
            }

            return true;
        }

        /**
         *  Changes the size of the container to the specified value.
         *  Appends elements to the end if the new size is larger,
         *  or destroys elements if it is smaller.
         *  The capacity is only increased to accommodate new element count.
         *  @param size New size of the BlockContainer.
         *  @param element Value assigned to new elements.
         *  @return False, if memory allocation failed or size parameter is invalid (negative).
         */
        bool Resize(SizeType size, const T& element = T())
        {
            bool success = true;
            if (m_capacity < size) {
                SizeType newCapacity = (m_capacity > 0) ? m_capacity * c_MemoryExpansionRate : c_MemoryInitialCapacity;
                newCapacity = (newCapacity > size) ? newCapacity : size;
                success = Reserve(newCapacity);
            }

            if (success) {
                if (m_size > size) {
                    for (Block* block = m_listHead; block != 0; block = block->m_next) {
                        if (block->m_size <= size) {
                            size -= block->m_size;
                        }
                        else {
                            RemoveFrom(Iterator(block, size));
                        }
                    }
                }
                else {
                    for (SizeType i = m_size; i < size; i++) {
                        Add(element);
                    }
                }
            }

            return success;
        }

        Iterator Begin()
        {
            return Iterator(m_listHead, 0);
        }

        Iterator End()
        {
            return Iterator(m_listEnd, m_listEnd == 0 ? 0 : m_listEnd->m_size);
        }

        ConstIterator ConstBegin() const
        {
            return ConstIterator(m_listHead, 0);
        }

        ConstIterator ConstEnd() const
        {
            return ConstIterator(m_listEnd, m_listEnd == 0 ? 0 : m_listEnd->m_size);
        }

    private:
        struct Block {
            Block* m_prev;
            Block* m_next;

            T* m_memory;
            SizeType m_size;
            SizeType m_capacity;

            Block () :
            m_prev(0),
                m_next(0),
                m_memory(0),
                m_size(0),
                m_capacity(0)
            {}
        };

        static const Int c_MemoryExpansionRate = 2;
        static const Int c_MemoryInitialCapacity = 8;

        SizeType m_size; // Number of elements in the container.
        SizeType m_capacity; // Number of elements for which memory has been allocated.

        Block* m_listHead; // Pointer to the first allocated block.
        Block* m_listTail; // Pointer to the last allocated block.
        Block* m_listEnd; // Pointer to the last used block.

        /**
         *  Remove all elements starting from a certain position.
         *  @param start Iterator to the first element to be removed.
         */
        void RemoveFrom(Iterator start)
        {
            if (start == End()) {
                return;
            }

            Block* block = start.m_block;
            SizeType item = start.m_item;

            while ((block != 0) && (block->m_size != 0)) {
                for (SizeType i = item; i < block->m_size; i++) {
                    // Destroy element at the given position.
                    MemoryManagement::Destruct(&block->m_memory[i]);

                    --m_size;
                }
                block->m_size = item;

                item = 0;
                block = block->m_next;
            }

            m_listEnd = start.m_block;
        }

        // ################# Begin: IteratorBase #################

        /**
         *  @brief Iterator and ConstIterator are derived from this base
         */
        template<typename X>
        class IteratorBase {
            public:
                IteratorBase() : m_block(0), m_item(0) {}

                IteratorBase(Block* block, SizeType item) : m_block(block), m_item(item) { }

                // Pre-increment
                IteratorBase& operator++()
                {
                    FEATSTD_DEBUG_ASSERT(m_block != 0);
                    ++m_item;
                    if (m_block->m_capacity == m_item) {
                        m_block = m_block->m_next;
                        m_item = 0;
                    }
                    return *this;
                }

                // Post-increment
                IteratorBase operator++(int)
                {
                    IteratorBase tmp = *this;
                    operator++();
                    return tmp;
                }

                bool operator!=(const IteratorBase& other) const
                {
                    return (this->m_block != other.m_block) || (this->m_item != other.m_item);
                }

                bool operator==(const IteratorBase& other) const
                {
                    return (this->m_block == other.m_block) && (this->m_item == other.m_item);
                }

                X& operator*()
                {
                    FEATSTD_DEBUG_ASSERT(m_block != 0);
                    return m_block->m_memory[m_item];
                }

                X* operator->() { return &operator*(); }

            private:
                friend class BlockContainer;

                Block* m_block;
                SizeType m_item;
        };
        // ################# End: IteratorBase #################
    };
}}
#endif //CANDERA_BlockContainer_H
