//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifdef CANDERA_2D_OVER_3D_ENABLED

#include "GlyphAtlasTextBrushCache.h"

#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine3D/Core/GlyphAtlasTextRenderContext.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DDevicePool.h>

namespace Candera
{

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine3D);

namespace Internal
{

GlyphAtlasTextBrushCache::GlyphAtlasTextBrushCache()
    :
    m_hasChanged(false)
{
}

GlyphAtlasTextBrushCache::~GlyphAtlasTextBrushCache()
{
    m_bitmapTracker.UnloadAll();
}

bool GlyphAtlasTextBrushCache::Update(const TextBrushProperties& textBrushProperties, const Rectangle& textRect)
{
    using namespace TextRendering;

    LayoutingOptions layoutingOptions = textBrushProperties.GetLayoutingOptions();
    PixelPosition textLeft = static_cast<PixelPosition>(textRect.GetLeft());
    PixelPosition textTop = static_cast<PixelPosition>(textRect.GetTop());
    layoutingOptions.SetOffset(TextCoordinate(-textLeft, -textTop));

    m_textRenderContext.Clear();
    TextRenderer textRenderer;
    m_hasChanged = textRenderer.Render(m_textRenderContext, layoutingOptions, textBrushProperties.GetShapingOptions(),
        textBrushProperties.GetTextProperties());
    return true;
}

void GlyphAtlasTextBrushCache::Render(const TextBrushProperties& textBrushProperties, const Matrix3x2& transform,
    const Rectangle& bound, ContextHandle2D output, Rectangle& outputArea)
{
    FEATSTD_UNUSED(textBrushProperties);

    if (m_hasChanged) {
        m_hasChanged = false;

        if (!m_textRenderContext.PrepareDrawCalls(bound.GetWidth(), bound.GetHeight())) {
            FEATSTD_LOG_ERROR("GlyphAtlasTextRenderContext failed to create or set vertex buffer.");
        }
    }

    if (m_textRenderContext.GetDrawCallCount() == 0) {
        return;
    }

    const GlyphAtlas& atlas = GlyphAtlas::GetInstance();
    using namespace Internal;
    Context2DOver3DDevicePool& pool = Context2DOver3DDevicePool::GetInstance();

    Matrix3x2 correctedTransform(transform);
    correctedTransform.Translate(bound.GetLeft(), bound.GetTop());
    bool result = Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, correctedTransform);
    result = result && RenderDevice2D::SetActiveArea(output, RenderDevice2D::SourceSurface, 0.0F, 0.0F, bound.GetWidth(), bound.GetHeight());

    for (SizeType i = 0; i < m_textRenderContext.GetDrawCallCount(); ++i) {
        const SizeType bitmapImageIndex = m_textRenderContext.GetBitmapImageIndex(i);
        SharedPointer<BitmapImage2D> glyphBitmapImage;
        if (atlas.GetBitmapImageCount() <= bitmapImageIndex) {
            glyphBitmapImage = m_bitmapTracker.Get(bitmapImageIndex);
        }
        else
        {
            glyphBitmapImage = atlas.GetBitmapImage(bitmapImageIndex);
            if (!glyphBitmapImage.PointsToNull())
            {
                m_bitmapTracker.Add(bitmapImageIndex, glyphBitmapImage);
            }
            else
            {
                glyphBitmapImage = m_bitmapTracker.Get(bitmapImageIndex);
            }
        }
        if (glyphBitmapImage.PointsToNull())
        {
            FEATSTD_LOG_ERROR("GlyphAtlasTextNodeRenderer cannot render the text because the atlas has been cleared or the font has been removed.");
            continue;
        }

        Rectangle rect = Rectangle(0.0F, 0.0F, bound.GetWidth(), bound.GetHeight());
        pool.SetCustomVertexBuffer(m_textRenderContext.GetVertexBuffer(i), rect);

        if (glyphBitmapImage != 0)
        {
            result = result && glyphBitmapImage->Activate(output);
            result = result && Renderer2D::Blit(output);
        }
        pool.SetCustomVertexBuffer(VertexBuffer::SharedPointer(0));
    }
    // Unload unused atlas bitmaps
    m_bitmapTracker.FinishIteration();

    if (result) {
        Float left;
        Float top;
        Float width;
        Float height;
        if (RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height)) {
            outputArea = Rectangle(left, top, width, height);
        }
    }
}

static const TextRendering::TextRenderContext& GlyphAtlasTextNodeRendererGetMeasureReferenceContext()
{
    FEATSTD_SYNCED_STATIC_OBJECT(GlyphAtlasTextRenderContext, s_glyphAtlasAccessTextRenderContext);
    return s_glyphAtlasAccessTextRenderContext;
}

const TextRendering::TextRenderContext* GlyphAtlasTextBrushCache::GetTextRenderContext() const
{
    return &GlyphAtlasTextNodeRendererGetMeasureReferenceContext();
}

void GlyphAtlasTextBrushCache::Unload()
{
    m_bitmapTracker.UnloadAll();
}

void GlyphAtlasTextBrushCache::Upload()
{
    m_bitmapTracker.UploadAll();
}

}

}

#endif
