//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TEXT_BRUSH_H)
    #define CANDERA_TEXT_BRUSH_H

#include <Candera/Environment.h>

#include <Candera/EngineBase/Common/Color.h>
#include <Candera/Engine2D/Effects/BrushEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/MemoryManagement/Disposer.h>
#include <Candera/System/GlobalizationBase/CultureDataType.h>
#include <Candera/TextEngine/TextEngineDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushProperties.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/TextBrushCache.h>

namespace Candera {

    class Image2D;

    namespace TextRendering {
        class TextRenderContext;
    }

    /**
     *  @brief Output alpha only glyph data. A color may be specified, which is applied to
     *  the color channels. The glyph data has its pixels multiplied with the color alpha.
     *  The text is transformed to the space of the current node. A font is required.
     *  Caching mode and maximum width for output image may be specified.
     *
     *  Note:
     *  No blending is done, so drawing this to an opaque surface would produce solid
     *  rectangle.
     */
    class TextBrush : public BrushEffect2D {

        FEATSTD_TYPEDEF_BASE(BrushEffect2D);

        public:

            typedef MemoryManagement::SharedPointer<TextBrush> SharedPointer;

            FEATSTD_RTTI_DECLARATION();

            /**
             *  Creates an instance of this class
             *  @return A pointer to the created object.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            /**
             *  Destructor
             */
            CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::TextBrush::Unload)
            virtual ~TextBrush();

            /**
             *  Activate the color used for text rendering.
             *  @param node
             *  @param output The color used for text rendering is activated for this 2D context.
             */
            void ActivateColor(const Node2D& node, ContextHandle2D output);

            /**
             *  Deactivate the color used for text rendering.
             *  @param output The color used for text rendering is deactivated for this 2D context.
             */
            void DeactivateColor(ContextHandle2D output) const;

            /**
             *  Activate the glyph blending used for text rendering.
             *  This is required when alpha blending in not enabled, to
             *  properly show text.
             *  @param output The glyph blending used for text rendering is activated for this 2D context.
             */
            void ActivateGlyphBlend(ContextHandle2D output);

            /**
             *  Deactivate the glyph blending used for text rendering.
             *  @param output The glyph blending used for text rendering is deactivated for this 2D context.

             */
            void DeactivateGlyphBlend(ContextHandle2D output);

            /**
             *  Render each glyph without setting the color or clearing the text area.
             *  @param transform    3x2 transform matrix
             *  @param output       The ContextHandle2D output
             *  @param outputArea   The output Area as a rectangle
             */
            void RenderText(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea);

            /**
             *  Clear the area that will be covered by the text.
             *  @param transform    3x2 transform matrix
             *  @param output       The ContextHandle2D output
             *  @param outputArea   The output Area as a rectangle
             */
            void ClearTextRectangle(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea);

            // overrides Effect2D::Upload
            virtual bool Upload();

            // overrides Effect2D::Unload
            virtual bool Unload();

            // overrides Effect2D::Update
            virtual bool Update();

            // overrides Effect2D::IsUploaded
            virtual bool IsUploaded() const { return m_isUploaded; }

            /**
             *  Access the text attached to the text brush. The text brush will process
             *  the associated text to produce glyph data.
             *  Must call Update for changes to become apparent if cache type is set to
             *  Bitmap or Surface.
             *  @return A property that can be used to modify the associated text.
             */
            TextPropertyEx& Text() { return m_properties.m_text; }
            
            /**
             *  Access the attached style. This style is used by the text brush in processing
             *  the associated text. Styles are shared objects and modifying style configuration
             *  might change the appearance of more than this TextBrush.
             *  Setting this property is mandatory. Must call Update for changes to become
             *  apparent if cache type is set to Bitmap or Surface.
             *  @return A property that can be used to modify the associated style.
             */
            StylePropertyEx& Style() { return m_properties.m_style; }

            /**
             *  Access the attached culture. The culture is used by the text renderer to define
             * language specific preferences of the rendered text.
             *  @return A property that can be used to modify the associated culture.
             */
            CulturePropertyEx& Culture() { return m_properties.m_culture; }

            /**
             *  Access the color. This represents the color of the output text.
             *  Glyph data is alpha only and this color is used as a constant color in
             *  the render device. This means that the RGB of the output data is set to
             *  the components of this color and the alpha channel is multiplied by its alpha component.
             *  @return A property that can be used to modify the color.
             */
            ColorProperty& Color() { return m_properties.m_color; }

#ifdef CANDERA_2D_OVER_3D_ENABLED
            /**
             *  Access the cache type. This allows choosing the appropriate
             *  cache to use with the text brush, for a balance in performance.
             *  Glyph level cache strategies:
             *  NoCache: small memory footprint, small update times, very large render time.
             *  GlyphCache: small memory footprint, small update times, medium render time.
             *  Whole text cache strategies:
             *  BitmapCache: very large memory footprint, very large update times, small render time.
             *  GlyphAtlasCache: small (but global) video memory footprint, small update times, small render time.
             *  SurfaceCache: large video memory footprint, large update times, small render time.
             *  Must call Update for changes to become apparent.
             *  @return A property that can be used to modify the cache type.
             */
#else
            /**
             *  Access the cache type. This allows choosing the appropriate
             *  cache to use with the text brush, for a balance in performance.
             *  Glyph level cache strategies:
             *  NoCache: small memory footprint, small update times, very large render time.
             *  GlyphCache: small memory footprint, small update times, medium render time.
             *  Whole text cache strategies:
             *  BitmapCache: very large memory footprint, very large update times, small render time.
             *  SurfaceCache: large video memory footprint, large update times, small render time.
             *  Must call Update for changes to become apparent.
             *  @return A property that can be used to modify the cache type.
             */
#endif
            TextBrushCacheTypePropertyEx& CacheType() { return m_properties.m_cacheType; }
            
            /**
             *  Access the size of the cache area. The cache area is used by
             *  certain caching mechanisms as boundary for the glyph data. This
             *  value is in pixels. If set to a negative value, the boundary
             *  becomes automatic. Example of caches that use this are 
             *  SurfaceCache and BitmapCache.
             *  Note: These caches use this value as a prediction on how large
             *  glyph data might become. Using excessively large values would
             *  waste memory. If a maximum value for the text can't be
             *  determined, better use a negative value with this property.
             *  Must call Update for changes to become apparent.
             *  @return A property that can be used to modify the cache.
             */
            Vector2PropertyEx& CacheArea() { return m_properties.m_cacheArea; }

            /**
             *  Access the size of the layout area. Alignment is done to the
             *  boundaries of this area. If either component of the specified
             *  size is negative or 0, the respective component is computed.
             *  If the text exceeds the specified area, the behavior is
             *  undefined.
             *  Must call Update for changes to become apparent.
             *  @return A property that can be used to modify the maximum width.
             */
            Vector2PropertyEx& LayoutingArea() { return m_properties.m_layoutingArea; }

            /**
             *  Access the glyph spacing. This value is used to adjust the spacing between glyphs.
             *  Must call Update for changes to become apparent.
             *  @return A property that can be used to modify the glyph spacing.
             */
            FloatPropertyEx& GlyphSpacing()
            {
                return m_properties.m_glyphSpacing;
            }

            /**
             *  Access the line spacing. Line spacing describes the distance
             *  between lines of text. It is a floating point value representing
             *  the factor by which Style::GetMetrics().lineHeight is multiplied
             *  when determining advancement to the next line.
             *  @return A property that can be used to modify the line spacing.
             */
            FloatPropertyEx& LineSpacing() { return m_properties.m_lineSpacing; }

            /**
             * Access the horizontal alignment. The horizontal alignment combined with the culture
             * text direction, is used to define how text will be horizontally aligned within its layout area.
             */
            HorizontalAlignmentPropertyEx& HorizontalAlignment() { return m_properties.m_horizontalAlignment; }

            /**
             * Access the vertical alignment. The vertical alignment is used to define how text will
             *  be vertically aligned within its layout area.
             */
            VerticalAlignmentPropertyEx& VerticalAlignment() { return m_properties.m_verticalAlignment; }

            /**
             *  Access multi line layout flag. This allows forcing a multi line text to be
             * layouted in a single line, ignoring new line markers.
             *  @return A property that can be used to modify the multi line layout flag.
             */
            BoolPropertyEx& MultiLineLayouting() { return m_properties.m_multiLineLayouting; }

            /**
             *  Access the preproecessed text attached to the text brush.
             *  When this is not 0, it will override the text given by the Text property.
             *  Must call Update for changes to become apparent if cache type is set to
             *  Bitmap or Surface.
             *  @return A property that can be used to modify the associated text.
             */
            PreprocessedTextPropertyEx& PreprocessedText() { return m_properties.m_preprocessedText; }

            /**
             *  Access word wrap flag. This allows enabling or disabling word wrap when
             *  layout text.
             *  @return A property that can be used to modify the word wrap flag.
             */
            BoolPropertyEx& WordWrap() { return m_properties.m_wordWrap; }

            /**
            *  Access the pixelwise line spacing flag. This allows switching the line spacing value interpretation.
            *  When this is set to true, line spacing will be in pixels. When this is set to false, line spacing will be 
            *  a factor by which Style::GetMetrics().lineHeight is multiplied.
            *  @return A property that can be used to modify the line spacing type.
            */
            BoolPropertyEx& PixelwiseLineSpacing() { return m_properties.m_pixelwiseLineSpacing; }

            /**
             *  Access the text layout strategy property. This property is use whenever
             *  text is drawn. See TextRendering::TextLayoutStrategy for more details.
             *  @return A property that can be used to modify the layout strategy.
             */
            TextLayoutStrategyPropertyEx& LayoutStrategy() { return m_properties.m_layoutStrategy; }

            // overrides Effect2D::GetBoundingRectangle
            virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const;

            // overrides Effect2D::GetBoundingRectangle
            virtual void GetLayoutingRectangle(Rectangle& rectangle) const;

            // overrides Effect2D::GetBasePoint
            virtual void GetBasePoint(Vector2& basePoint) const;

            // overrides Effect2D::Clone
            virtual Effect2D::SharedPointer Clone() const;

            /**
             *  The text render context used by this brush. This may not be used for rendering text, only for measurements.
             *  @return The underlying TextRenderContext.
             */
            const TextRendering::TextRenderContext* GetTextRenderContext() const { return m_cache == 0 ? 0 : m_cache->GetTextRenderContext(); }

            /// @cond excluded from doxygen
            CdaEffect2DDef(Candera::TextBrush, TextBrush, EFFECT2D_TYPE_BRUSH)
                CdaDeprecated()
                CdaEffect2DProperties()
                    CdaEffect2DImportBaseClassProperties(BrushEffect2D, Candera::BrushEffect2D)

                    CdaEffect2DDisposableProperty(Text, const Candera::TChar*, m_properties.m_text)
                        CdaDescription("The text brush will process this text to produce glyph data. "
                            "This property is mandatory.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(TextColor, Candera::Color, m_properties.m_color)
                        CdaDescription("The RGB of the glyph data is set to the components of this "
                            "color and the alpha channel is multiplied by its alpha component.")
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(Style, MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle>, m_properties.m_style)
                        CdaDescription("The style used in processing the text. This property is mandatory.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(Culture, Candera::Globalization::Culture::SharedPointer, m_properties.m_culture)
                        CdaDescription("The culture used in rendering the text.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(CacheType, Candera::TextBrushCacheType, m_properties.m_cacheType)
#ifdef CANDERA_2D_OVER_3D_ENABLED
                    CdaDescription("The caching mechanism used by this text brush."
                            "Glyph level cache strategies: "
                            "* NoCache: small memory footprint, small update times, very large render time. "
                            "* GlyphCache: small memory footprint, small update times, medium render time. "
                            "Whole text cache strategies: "
                            "* BitmapCache: very large memory footprint, very large update times, small render time. "
                            "* GlyphAtlasCache: small (but global) video memory footprint, small update times, small render time. "
                            "* SurfaceCache: large video memory footprint, large update times, small render time.")
#else
                    CdaDescription("The caching mechanism used by this text brush."
                            "Glyph level cache strategies: "
                            "* NoCache: small memory footprint, small update times, very large render time. "
                            "* GlyphCache: small memory footprint, small update times, medium render time. "
                            "Whole text cache strategies: "
                            "* BitmapCache: very large memory footprint, very large update times, small render time. "
                            "* SurfaceCache: large video memory footprint, large update times, small render time.")
#endif
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(CacheArea, Candera::Vector2, m_properties.m_cacheArea)
                        CdaDescription("Caching mechanisms that cache the whole text use this "
                            "value as size for their cache. If the text size becomes smaller then "
                            "this value, the cache is not rebuilt, saving the creation time. If the "
                            "text size become larger then this value, the text is cut to fit the cache. "
                            "If this value is 0 or negative, the cache will automatically determine the "
                            "proper size for the text.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(LayoutingArea, Candera::Vector2, m_properties.m_layoutingArea)
                        CdaDescription("This defines the size of the layout area used for text. "
                            "Alignment is done to the boundaries of this area. If either component of "
                            "the specified size is negative or 0, the respective component is computed. "
                            "If the text exceeds the specified area, the behavior is undefined.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(GlyphSpacing, Candera::Float, m_properties.m_glyphSpacing)
                        CdaDescription("This defines a value used as additional offset between glyphs.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(LineSpacing, Float, m_properties.m_lineSpacing)
                        CdaDescription("This controls the distance between lines in multi-line text. "
                            "The value is a multiple of the default line spacing of the current style.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(PixelwiseLineSpacing, bool, m_properties.m_pixelwiseLineSpacing)
                        CdaDescription("This controls the distance between lines in multi-line text. "
                        "The pixelwise line spacing allows the reinterpretation of line spacing."
                        "When this flag is set, line spacing represents an absolute pixel value for line spacing.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(HorizontalAlignment, Candera::HorizontalAlignment, m_properties.m_horizontalAlignment)
                        CdaDescription("The horizontal alignment, using the culture text direction, "
                            "is used to define how text will be horizontally aligned within its layout area.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(VerticalAlignment, Candera::VerticalAlignment, m_properties.m_verticalAlignment)
                        CdaDescription("The vertical alignment is used to define how text will be "
                            "vertically aligned within its layout area.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(MultiLineLayouting, bool, m_properties.m_multiLineLayouting)
                        CdaDescription("The multi line layout flag allows forcing a multi line text to be "
                            "layouted in a single line, ignoring new line markers.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(WordWrap, bool, m_properties.m_wordWrap)
                        CdaDescription("The word wrap flag allows forcing a multi line layout "
                            "of each paragraph of text. Each line is computed to fit into the "
                            "layout area.")
                        CdaUpdateType(OnApplyUpdateType)
                    CdaEffect2DPropertyEnd()

                CdaEffect2DPropertiesEnd()
            CdaEffect2DDefEnd()
            /// @endcond

        protected:
            // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
            TextBrush();
            explicit TextBrush(const TextBrush& rhs);
            TextBrush& operator = (const TextBrush& rhs);

            // overrides Effect2D::Render
            virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

        private:
            bool UpdateMeasurementCache() const;

            bool IsMeasurementCacheValid() const;
            void SetMeasurementCacheValid(bool valid) const;
            bool IsRenderCacheValid() const;
            void SetRenderCacheValid(bool valid) const;

            void UpdateCacheValidity() const;

            bool m_isUploaded : 1;
            //used for tracking the text cache validity.
            mutable bool m_isMeasurementCacheValid : 1;
            mutable bool m_isRenderCacheValid : 1;

            Internal::TextBrushCache* m_cache;
            Internal::TextBrushProperties m_properties;

            //used for caching the text rectangle.
            mutable Rectangle m_boundingRect;
            mutable Rectangle m_layoutingRect;
    };

}   // namespace Candera

#endif  // CANDERA_TEXT_BRUSH_H
