//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SolidColorBrushColorBlend.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(SolidColorBrushColorBlend, CombinedEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    SolidColorBrushColorBlend::SolidColorBrushColorBlend()
    {
    }

    SolidColorBrushColorBlend::SolidColorBrushColorBlend(const SolidColorBrushColorBlend& rhs) :
        Base(rhs),
        m_solidBrush(rhs.m_solidBrush),
        m_colorEffect(rhs.m_colorEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    SolidColorBrushColorBlend::~SolidColorBrushColorBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    SolidColorBrushColorBlend::SharedPointer SolidColorBrushColorBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        SolidColorBrushColorBlend* brush = FEATSTD_NEW(SolidColorBrushColorBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        SolidColorBrushColorBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void SolidColorBrushColorBlend::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                                           const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "SolidColorBrushColorBlend"));

        const Vector2& size = m_solidBrush.Size();

        if ((size.GetX() != 0.0F) && (size.GetY() != 0.0F)) {
            m_blendEffect.ActivateBlend(output);

            Color::Data& color = m_colorEffect.Color().Get();
            Color::Data colorFinal;
            colorFinal.red = m_solidBrush.Color().Get().red * color.red;
            colorFinal.green = m_solidBrush.Color().Get().green * color.green;
            colorFinal.blue = m_solidBrush.Color().Get().blue * color.blue;
            colorFinal.alpha = m_solidBrush.Color().Get().alpha * color.alpha;
            Float cameraAlpha = 1.0F;
            const Camera2D* camera2D = Renderer2D::GetActiveCamera();
            if (0 != camera2D) {
                if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                    cameraAlpha = camera2D->GetEffectiveAlphaValue();
                }
            }
            static_cast<void>(RenderDevice2D::SetSurfaceConstColor(
                output,
                RenderDevice2D::SourceSurface,
                colorFinal.red,
                colorFinal.green,
                colorFinal.blue,
                colorFinal.alpha * node.GetEffectiveAlphaValue() * cameraAlpha
                ));


            m_solidBrush.RenderSolidRectangle(transform, output, outputArea, node);

            static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));
            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer SolidColorBrushColorBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(SolidColorBrushColorBlend)(*this));
    }

}   // namespace Candera
