//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SOLID_COLOR_BRUSH_H)
    #define CANDERA_SOLID_COLOR_BRUSH_H

#include <Candera/Engine2D/Effects/BrushEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Vector2Property.h>
#include <Candera/Engine2D/Property/ColorProperty.h>
#include <Candera/System/Mathematics/MathDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>

namespace Candera {

    class Image2D;

    /**
     *  @brief Output a rectangle filled with a solid color.
     *  The rectangle is transformed to the transformation space of the current node.
     */
    class SolidColorBrush : public BrushEffect2D {

        FEATSTD_TYPEDEF_BASE(BrushEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<SolidColorBrush> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~SolidColorBrush();

        /**
         *  Activate the color used for text rendering.
         *  @param node
         *  @param output The color used for text rendering is activated for this 2D context.
         */
        void ActivateColor(const Node2D& node, ContextHandle2D output);

        /**
         *  Deactivate the color used for text rendering.
         *  @param output The color used for text rendering is deactivated for this 2D context.
         */
        void DeactivateColor(ContextHandle2D output) const;


        CANDERA_DEPRECATED_3_6_0(
            "Please use RenderSolidRectangle(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea, const Node2D& node);",
            void RenderSolidRectangle(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea));


        /**
         *  Render the rectangle without setting the color.
         *  @param transform    3x2 transform matrix.
         *  @param output       The 2D context handle.
         *  @param outputArea   The output area as a rectangle.
         *  @param node         The node that is rendered.
         */
        void RenderSolidRectangle(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea, const Node2D& node);

        /**
         *  Access the size of the fill area. This is a two component vector
         *  of which the first component represents the width and the second the height
         *  in pixels of the filled rectangle, in local space.
         *  @return A property that can be used to modify the rectangle size.
         */
        Vector2Property& Size() { return m_size; }

        /**
         *  Access the fill color. This is the color by which the rectangle is filled.
         *  @return A property that can be used to modify the rectangle color.
         */
        ColorProperty& Color() { return m_color; }

        // overrides Effect2D::GetBoundingRectangle
        virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const;

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /**
        * Mesh2D cannot be supported by all effects, as certain effects would create 2D vertex buffers, which would be overriden by the Mesh2D.
        * This flag helps to distinguish between effects that can be used with Mesh2D and effects that are not available.
        * @return Whether Mesh2D can be used with this effect (true) or not (false).
        */
        virtual bool IsMesh2DSupported() const override { return true; }

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::SolidColorBrush, SolidColorBrush, EFFECT2D_TYPE_BRUSH)
            CdaEffect2DProperties()
                CdaEffect2DImportBaseClassProperties(BrushEffect2D, Candera::BrushEffect2D)

                CdaEffect2DProperty(FillColor, Candera::Color, m_color)
                    CdaDescription("The solid color.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(Size, Vector2, m_size)
                    CdaDescription("The filled size (width and height).")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        SolidColorBrush();
        explicit SolidColorBrush(const SolidColorBrush& rhs);
        SolidColorBrush& operator = (const SolidColorBrush& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        Vector2Property m_size;
        ColorProperty m_color;
    };

}   // namespace Candera

#endif  // CANDERA_SOLID_COLOR_BRUSH_H
