//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SolidColorBrush.h"
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/Mesh2D.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(SolidColorBrush, BrushEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    SolidColorBrush::SolidColorBrush() :
        Base(),
        m_size(Vector2(50.0F, 50.0F)),
        m_color(Candera::Color(1.0F, 1.0F, 1.0F, 1.0F))
    {
    }

    SolidColorBrush::SolidColorBrush(const SolidColorBrush& rhs) :
        Base(rhs),
        m_size(rhs.m_size.Get()),
        m_color(rhs.m_color.Get())
    {

    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    SolidColorBrush::~SolidColorBrush()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    SolidColorBrush::SharedPointer SolidColorBrush::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        SolidColorBrush* brush = FEATSTD_NEW(SolidColorBrush);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        SolidColorBrush::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void SolidColorBrush::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform, const Node2D& node,
                                 ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "SolidColorBrush"));

        if ((m_size().GetX() != 0.0F) && (m_size().GetY() != 0.0F)) {
            ActivateColor(node, output);
            RenderSolidRectangle(transform, output, outputArea, node);
            DeactivateColor(output);
        }
    }

    /******************************************************************************
     *  GetBoundingRectangle
     ******************************************************************************/
    void SolidColorBrush::GetBoundingRectangle(Rectangle& boundingRectangle) const
    {
        boundingRectangle.SetPosition(0.0F, 0.0F);
        boundingRectangle.SetSize(m_size());
    }

    /******************************************************************************
     *  ActivateColor
     ******************************************************************************/
    void SolidColorBrush::ActivateColor(const Node2D& node, ContextHandle2D output)
    {

        Float cameraAlpha = 1.0F;
        const Camera2D* camera2D = Renderer2D::GetActiveCamera();
        if (0 != camera2D) {
            if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                cameraAlpha = camera2D->GetEffectiveAlphaValue();
            }
        }
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(
            output,
            RenderDevice2D::SourceSurface,
            m_color().red,
            m_color().green,
            m_color().blue,
            m_color().alpha * node.GetEffectiveAlphaValue() * cameraAlpha));
    }
    /******************************************************************************
     *  DeactivateColor
     ******************************************************************************/
    void SolidColorBrush::DeactivateColor(ContextHandle2D output) const
    {
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));
    }
    /******************************************************************************
     *  RenderSolidRectangle
     ******************************************************************************/
    void SolidColorBrush::RenderSolidRectangle(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea, const Node2D& node)
    {
        bool result = Renderer2D::SetTransformationMatrix(
            output,
            RenderDevice2D::SourceSurface,
            transform);

        result = result && RenderDevice2D::SetSurface(
            output,
            RenderDevice2D::SourceSurface,
            0);

        Rectangle rect(0.0F, 0.0F, m_size().GetX(), m_size().GetY());
        if (node.IsTypeOf(Mesh2D::GetTypeId())) {
            const Mesh2D& mesh = static_cast<const Mesh2D&>(node);
            if (!(mesh.GetVertexBuffer2D().PointsToNull())) {
                rect = mesh.GetVertexBuffer2D()->GetBoundingRectangle();
            }
        }

        result = result && RenderDevice2D::SetActiveArea(
            output,
            RenderDevice2D::SourceSurface,
            rect.GetPosition().GetX(),
            rect.GetPosition().GetY(),
            rect.GetSize().GetX(),
            rect.GetSize().GetY());

        result = result && Renderer2D::Blit(output);

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            if (RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height)) {
                outputArea = Rectangle(left, top, width, height);
            }
        }
    }


    void SolidColorBrush::RenderSolidRectangle(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea)
    {
        bool result = Renderer2D::SetTransformationMatrix(
            output,
            RenderDevice2D::SourceSurface,
            transform);

        result = result && RenderDevice2D::SetSurface(
            output,
            RenderDevice2D::SourceSurface,
            0);

        Rectangle rect(0.0F, 0.0F, m_size().GetX(), m_size().GetY());

        result = result && RenderDevice2D::SetActiveArea(
            output,
            RenderDevice2D::SourceSurface,
            rect.GetPosition().GetX(),
            rect.GetPosition().GetY(),
            rect.GetSize().GetX(),
            rect.GetSize().GetY());

        result = result && Renderer2D::Blit(output);

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            if (RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height)) {
                outputArea = Rectangle(left, top, width, height);
            }
        }
    }


    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer SolidColorBrush::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(SolidColorBrush)(*this));
    }

}   // namespace Candera
