//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(CANDERA_SHEAR_EFFECT_H)
    #define CANDERA_SHEAR_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Property.h>
#include <Candera/System/Mathematics/MathDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>

namespace Candera {

    /**
     *  @brief Shears the incoming surface horizontally and vertically using the given angles.
     */
    class ShearEffect : public InPlaceEffect2D {

        FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<ShearEffect> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class.
         *  @return Pointer to the created ShearEffect object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~ShearEffect();

        /**
         *  Retrieves the current shear angle-x, this returned property
         *  can be changed. The performed changes will affect this class.
         *  @return The current shear angle-x.
         */
        FloatProperty& ShearAngleX() { return m_shearAngleX; }

        /**
         *  Retrieves the current shear angle-y, this returned property
         *  can be changed. The performed changes will affect this class.
         *  @return The current shear angle-y.
         */
        FloatProperty& ShearAngleY() { return m_shearAngleY; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::ShearEffect, ShearEffect, EFFECT2D_TYPE_INPLACE)
            CdaEffect2DProperties()

                CdaEffect2DProperty(ShearAngleX, Float, m_shearAngleX)
                    CdaDescription("Horizontal shearing angle.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(ShearAngleY, Float, m_shearAngleY)
                    CdaDescription("Vertical shearing angle.")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        ShearEffect();
        ShearEffect(const ShearEffect& rhs);
        ShearEffect& operator = (const ShearEffect& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        FloatProperty m_shearAngleX;
        FloatProperty m_shearAngleY;
        FloatProperty m_lastShearAngleX;
        FloatProperty m_lastShearAngleY;

        Matrix3x2 m_shearMatrix;

        bool IsShearMatrixValid();
        void UpdateShearMatrix();
    };

}   // namespace Candera

#endif  // CANDERA_SHEAR_EFFECT_H
