//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ShearEffect.h"
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(ShearEffect, InPlaceEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    ShearEffect::ShearEffect() :
        m_shearAngleX(0.0F),
        m_shearAngleY(0.0F),
        m_lastShearAngleX(0.0F),
        m_lastShearAngleY(0.0F)
    {
        m_shearMatrix.SetZero();
        UpdateShearMatrix();
    }

    ShearEffect::ShearEffect(const ShearEffect& rhs) :
        Base(rhs),
        m_shearAngleX(rhs.m_shearAngleX.Get()),
        m_shearAngleY(rhs.m_shearAngleY.Get()),
        m_lastShearAngleX(0.0F),
        m_lastShearAngleY(0.0F)
    {
        m_shearMatrix.SetZero();
        UpdateShearMatrix();
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    ShearEffect::~ShearEffect()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    ShearEffect::SharedPointer ShearEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        ShearEffect* brush = FEATSTD_NEW(ShearEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        ShearEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void ShearEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& /*transform*/, const Node2D& node,
                             ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "ShearEffect"));

        if (!IsShearMatrixValid()) {
            UpdateShearMatrix();
        }

        static_cast<void>(RenderDevice2D::SetSurface(output, RenderDevice2D::SourceSurface, input));

        Camera2D* cam = Renderer2D::GetActiveCamera();
        Matrix3x2 transform = node.GetWorldTransform();
        Matrix3x2 camTransform;
        if (cam != 0) {
            camTransform = cam->GetViewMatrix();
            const Vector2& cameraTranslationOffset = Renderer2D::GetCameraTranslationOffset();
            camTransform.Translate(cameraTranslationOffset.GetX(), cameraTranslationOffset.GetY());
        }

        Matrix3x2 shear;
        shear.SetIdentity();
        FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
        shear.Translate(node.GetPivotOffset().GetX(), node.GetPivotOffset().GetY()); //Also consider offset.
        shear *= m_shearMatrix;
        transform.Translate(-node.GetPivotOffset().GetX(), -node.GetPivotOffset().GetY());
        FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
        transform = shear * transform;
        transform *= camTransform;

        outputArea = inputArea;

        static_cast<void>(RenderDevice2D::SetActiveArea(output, RenderDevice2D::SourceSurface, 0.0F, 0.0F, -1.0F, -1.0F));

        //Draw the actual sheared brush.
        static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));
        static_cast<void>(Renderer2D::Blit(output));
    }

    /******************************************************************************
     *  IsShearMatrixValid
     ******************************************************************************/
    bool ShearEffect::IsShearMatrixValid()
    {
        bool valid = true;

        if ((m_shearAngleX() != m_lastShearAngleX) || (m_shearAngleY() != m_lastShearAngleY)) {
            valid = false;
        }

        return valid;
    }

     /******************************************************************************
     *  UpdateShearMatrix
     ******************************************************************************/
    void ShearEffect::UpdateShearMatrix()
    {
        m_shearMatrix.SetIdentity();
        m_shearMatrix.Set(1,0, Math::Tangent(Math::DegreeToRadian(m_shearAngleX())));
        m_shearMatrix.Set(0,1, Math::Tangent(Math::DegreeToRadian(m_shearAngleY())));
        m_lastShearAngleX = m_shearAngleX();
        m_lastShearAngleY = m_shearAngleY();
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer ShearEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(ShearEffect)(*this));
    }

}   // namespace Candera

