//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(CANDERA_SHADOW_EFFECT_H)
    #define CANDERA_SHADOW_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Vector2Property.h>
#include <Candera/Engine2D/Property/ColorProperty.h>
#include <Candera/Engine2D/Property/Property.h>
#include <Candera/System/Mathematics/MathDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>

namespace Candera {

    /**
     *  @brief  Apply a colored shadow to the predecessor effect, Shadow is computed by it's scale and offset.
     *          Transparencies are also applied to the shadow, so transparent parts won't be drawn.
     *          Can also be used as colored glow frame effect.
     */
    class ShadowEffect : public InPlaceEffect2D {

        FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<ShadowEffect> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class.
         *  @return Pointer to the created ColorShadowEffect object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~ShadowEffect();

        /**
         *  Retrieves the current shadow color, this returned property
         *  can be changed. The performed changes will affect this class.
         *  @return The current shadow color.
         */
        ColorProperty& ShadowColor() { return m_shadowColor; }

        /**
         *  Retrieves the current shadow scale factor as a vector, this returned property
         *  can be changed. The performed changes will affect this class.
         *  @return The current shadow scale.
         */
        Vector2Property& ShadowScale() { return m_shadowScale; }

        /**
         *  Retrieves the current shadow position offset as a vector, this returned property
         *  can be changed. The performed changes will affect this class.
         *  @return The current shadow position offset.
         */
        Vector2Property& ShadowPositionOffset() { return m_shadowPositionOffset; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::ShadowEffect, ShadowEffect, EFFECT2D_TYPE_INPLACE)
            CdaEffect2DProperties()

                CdaEffect2DProperty(ShadowColor, Color, m_shadowColor)
                    CdaDescription("Color applied to the shadow. ")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(ShadowScale, Vector2, m_shadowScale)
                    CdaDescription("Determines the scale of the shadow.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(ShadowOffset, Vector2, m_shadowPositionOffset)
                    CdaDescription("Offset the shadow relative to node's position.")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond


    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        ShadowEffect();
        ShadowEffect(const ShadowEffect& rhs);
        ShadowEffect& operator = (const ShadowEffect& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        ColorProperty m_shadowColor;
        Vector2Property m_shadowScale;
        Vector2Property m_shadowPositionOffset;

    };

}   // namespace Candera

#endif  // CANDERA_SHADOW_EFFECT_H
