//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SHADOW_BITMAP_BRUSH_BLEND_H)
    #define CANDERA_SHADOW_BITMAP_BRUSH_BLEND_H

#include <Candera/Engine2D/Effects/CombinedEffect2D.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <CanderaPlatform/Device/Common/Effects/ShadowEffect.h>
#include <CanderaPlatform/Device/Common/Effects/BlendEffect.h>

namespace Candera {

    class Image2D;

    /**
     *  @brief  Output a shadowed or glowing bitmap image, both shadow/glow and image alpha blended.
     *          @see Candera::ShadowEffect
     *          @see Candera::BlendEffect
     */
    class ShadowBitmapBrushBlend : public CombinedEffect2D {

        FEATSTD_TYPEDEF_BASE(CombinedEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<ShadowBitmapBrushBlend> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class.
         *  @return Pointer to the created ShadowBitmapBrushBlend object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~ShadowBitmapBrushBlend();

        // overrides Effect2D::GetBrushEffect2D
        virtual BrushEffect2D* GetBrushEffect2D() { return &m_bitmapBrush; }

        // overrides Effect2D::GetInplaceEffect2DCount
        virtual UInt8 GetInPlaceEffect2DCount() const { return 1; }

        // overrides Effect2D::GetInPlaceEffect2D
        virtual InPlaceEffect2D* GetInPlaceEffect2D(UInt8 index) { return (index == 0) ? &m_shadowEffect: 0; }

        // overrides Effect2D::GetBlendEffect2D
        virtual BlendEffect2D* GetBlendEffect2D() { return &m_blendEffect; }

        // overrides Effect2D::Upload
        virtual bool Upload();

        // overrides Effect2D::Unload
        virtual bool Unload();

        // overrides Effect2D::Update
        virtual bool Update();

        /**
         *  Retrieves the bitmap brush.
         *  @return     The bitmap brush.
         */
        const BitmapBrush& GetBitmapBrush() const { return m_bitmapBrush; }
        BitmapBrush& GetBitmapBrush() { return m_bitmapBrush; }

        /**
         *  Retrieves the shadow effect.
         *  @return     The shadow effect.
         */
        const ShadowEffect& GetShadowEffect() const { return m_shadowEffect; }
        ShadowEffect& GetShadowEffect() { return m_shadowEffect; }

        /**
         *  Retrieves the blend effect.
         *  @return     The blend effect.
         */
        const BlendEffect& GetBlendEffect() const { return m_blendEffect; }
        BlendEffect& GetBlendEffect() { return m_blendEffect; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /**
        * Mesh2D cannot be supported by all effects, as certain effects would create 2D vertex buffers, which would be overriden by the Mesh2D.
        * This flag helps to distinguish between effects that can be used with Mesh2D and effects that are not available.
        * @return Whether Mesh2D can be used with this effect (true) or not (false).
        */
        virtual bool IsMesh2DSupported() const override { return true; }

        /// @cond excluded from doxygen
        CdaEffect2DCombinedDef(Candera::ShadowBitmapBrushBlend, ShadowBitmapBrushBlend,  EFFECT2D_TYPE_BRUSH, EFFECT2D_TYPE_BLEND)
            CdaEffect2DProperties()

                CdaEffect2DImportProperties(BitmapBrush, Candera::BitmapBrush, m_bitmapBrush)
                CdaEffect2DImportProperties(ShadowEffect, Candera::ShadowEffect, m_shadowEffect)
                CdaEffect2DImportProperties(BlendEffect, Candera::BlendEffect, m_blendEffect)

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

        friend class BitmapBrush;

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        ShadowBitmapBrushBlend();
        ShadowBitmapBrushBlend(const ShadowBitmapBrushBlend& rhs);
        ShadowBitmapBrushBlend& operator = (const ShadowBitmapBrushBlend& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        // Use class wrapper to forward protected members.
        class BitmapBrushWrapper : public BitmapBrush {} m_bitmapBrush;
        class ShadowEffectWrapper : public ShadowEffect {
            public:
                using ShadowEffect::Render;
        } m_shadowEffect;
        class BlendEffectWrapper : public BlendEffect {} m_blendEffect;
    };

}   // namespace Candera

#endif  // CANDERA_SHADOW_BITMAP_BRUSH_BLEND_H
