//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MirrorEffect.h"
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(MirrorEffect, InPlaceEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    MirrorEffect::MirrorEffect() :
        m_mirrorAxisFrom(Vector2(-1.0F, -1.0F)),
        m_mirrorAxisTo(Vector2(-1.0F, -1.0F)),
        m_alpha(1.0F)
    {
        m_mirrorMatrix.SetZero();
        UpdateMirrorMatrix();
    }

    MirrorEffect::MirrorEffect(const MirrorEffect& rhs) :
        Base(rhs),
        m_mirrorAxisFrom(rhs.m_mirrorAxisFrom.Get()),
        m_mirrorAxisTo(rhs.m_mirrorAxisTo.Get()),
        m_alpha(rhs.m_alpha.Get())
    {
        m_mirrorMatrix.SetZero();
        UpdateMirrorMatrix();
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    MirrorEffect::~MirrorEffect()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    MirrorEffect::SharedPointer MirrorEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        MirrorEffect* brush = FEATSTD_NEW(MirrorEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        MirrorEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void MirrorEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node,
                              ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "MirrorEffect"));

        if (!IsMirrorMatrixValid()) {
            UpdateMirrorMatrix();
        }

        static_cast<void>(RenderDevice2D::SetSurface(output, RenderDevice2D::SourceSurface, input));

        Camera2D* cam = Renderer2D::GetActiveCamera();
        Matrix3x2 mirrorTransform = node.GetWorldTransform();
        Matrix3x2 camTransform;
        if (cam != 0) {
            camTransform = cam->GetViewMatrix();
            const Vector2& cameraTranslationOffset = Renderer2D::GetCameraTranslationOffset();
            camTransform.Translate(cameraTranslationOffset.GetX(), cameraTranslationOffset.GetY());
        }

        outputArea = inputArea;

        static_cast<void>(RenderDevice2D::SetActiveArea(output, RenderDevice2D::SourceSurface, 0.0F, 0.0F, -1.0F, -1.0F));

        static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, mirrorTransform * m_mirrorMatrix * camTransform));
        //set alpha channel
        //xxx: gsteve Alpha value might be improved in the future by using a generated alpha gradient.
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.0F, 1.0F, 1.0F, m_alpha()));
        //blit mirrored image
        static_cast<void>(Renderer2D::Blit(output));
        //unset alpha channel
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));

        //draw the actual brush at the end, so that it is always in the foreground.
        static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));
        static_cast<void>(Renderer2D::Blit(output));
    }

    /******************************************************************************
     *  IsMirrorMatrixValid
     ******************************************************************************/
    bool MirrorEffect::IsMirrorMatrixValid()
    {
        bool valid = true;

        if ((!Math::FloatAlmostEqual(m_mirrorAxisFrom().GetX(), m_lastMirrorAxisFrom.GetX())) 
            || (!Math::FloatAlmostEqual(m_mirrorAxisFrom().GetY(), m_lastMirrorAxisFrom.GetY()))) {
            m_lastMirrorAxisFrom = m_mirrorAxisFrom;
            valid = false;
        }

        if ((!Math::FloatAlmostEqual(m_mirrorAxisTo().GetX(), m_lastMirrorAxisTo.GetX())) 
            || (!Math::FloatAlmostEqual(m_mirrorAxisTo().GetY(), m_lastMirrorAxisTo.GetY()))) {
            m_lastMirrorAxisTo = m_mirrorAxisTo;
            valid = false;
        }

        return valid;
    }

     /******************************************************************************
     *  UpdateMirrorMatrix
     ******************************************************************************/
    void MirrorEffect::UpdateMirrorMatrix()
    {
        Vector2 vec = m_mirrorAxisTo() - m_mirrorAxisFrom();
        Vector2 x(1.0F,0.0F);
        Float angle = 0.0F;
        if (vec.GetLength() != 0.0F) {
            angle = Math::RadianToDegree(Math::ACosine(vec.GetDotProduct(x) / (vec.GetLength()*x.GetLength())));
        }

        m_mirrorMatrix.SetIdentity();

        Matrix3x2 translation;
        translation.SetTranslation(-m_mirrorAxisFrom().GetX(),-m_mirrorAxisFrom().GetY());
        m_mirrorMatrix *= translation;

        Matrix3x2 rotation;
        rotation.SetRotation(-angle);
        m_mirrorMatrix *= rotation;

        Matrix3x2 mirror;
        mirror.Scale(1.0F,-1.0F);
        m_mirrorMatrix *= mirror;

        rotation.SetRotation(angle);
        m_mirrorMatrix *= rotation;

        translation.SetTranslation(m_mirrorAxisFrom().GetX(), m_mirrorAxisFrom().GetY());
        m_mirrorMatrix *= translation;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer MirrorEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(MirrorEffect)(*this));
    }

}   // namespace Candera

