//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MirrorBitmapBrushBlend.h"
#include <CanderaPlatform/Device/Common/Effects/MirrorEffect.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(MirrorBitmapBrushBlend, CombinedEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    MirrorBitmapBrushBlend::MirrorBitmapBrushBlend()
    {
    }

    MirrorBitmapBrushBlend::MirrorBitmapBrushBlend(const MirrorBitmapBrushBlend& rhs) :
        Base(rhs),
        m_bitmapBrush(rhs.m_bitmapBrush),
        m_mirrorEffect(rhs.m_mirrorEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    MirrorBitmapBrushBlend::~MirrorBitmapBrushBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    MirrorBitmapBrushBlend::SharedPointer MirrorBitmapBrushBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        MirrorBitmapBrushBlend* brush = FEATSTD_NEW(MirrorBitmapBrushBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        MirrorBitmapBrushBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void MirrorBitmapBrushBlend::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                                             const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "MirrorBitmapBrushBlend"));

        Image2D* image = m_bitmapBrush.Image();

        if (image != 0) {
            m_blendEffect.ActivateBlend(output);
            Rectangle rect;
            m_bitmapBrush.GetBoundingRectangle(rect);
            static_cast<void>(image->Activate(output));
            m_mirrorEffect.Render(image->Get2DSurfaceHandle(), rect, transform, node, output, outputArea);
            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool MirrorBitmapBrushBlend::Upload()
    {
        bool success = m_bitmapBrush.Upload();
        success = m_mirrorEffect.Upload() && success;
        success = m_blendEffect.Upload() && success;
        return success;
    }
    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool MirrorBitmapBrushBlend::Unload()
    {
        bool success = m_bitmapBrush.Unload();
        success = m_mirrorEffect.Unload() && success;
        success = m_blendEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool MirrorBitmapBrushBlend::Update()
    {
        bool success = m_bitmapBrush.Update();
        success = m_mirrorEffect.Update() && success;
        success = m_blendEffect.Update() && success;
        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer MirrorBitmapBrushBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(MirrorBitmapBrushBlend)(*this));
    }

}   // namespace Candera
