//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GL_TEXT_BRUSH_MASK_BLEND_H)
    #define CANDERA_GL_TEXT_BRUSH_MASK_BLEND_H

#include <Candera/Engine2D/Effects/CombinedEffect2D.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrush.h>
#include <CanderaPlatform/Device/Common/Effects/GlMaskEffect.h>
#include <CanderaPlatform/Device/Common/Effects/BlendEffect.h>

namespace Candera {

    /** 
     *  @brief  Output alpha only glyph data, apply an alpha mask and blend it with the store buffer.
     *          Same as chaining (TextBrush + GlMaskEffect + BlendEffect)
     *          @see Candera::TextBrush
     *          @see Candera::GlMaskEffect
     *          @see Candera::BlendEffect
     */
    class GlTextBrushMaskBlend : public CombinedEffect2D {

        FEATSTD_TYPEDEF_BASE(CombinedEffect2D);

        public:
            typedef MemoryManagement::SharedPointer<GlTextBrushMaskBlend> SharedPointer;

            FEATSTD_RTTI_DECLARATION();

            /**
             *  Creates an instance of this class
             *  @return A pointer to the created object.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            /**
             *  Destructor
             */
            virtual ~GlTextBrushMaskBlend();
           /**
             *  Retrieves the text brush.
             *  @return     The text brush.
             */
            const TextBrush& GetTextBrush() const { return m_textBrush; }
            TextBrush& GetTextBrush() { return m_textBrush; }


           /**
             *  Retrieves the mask effect.
             *  @return     The mask effect.
             */
            const GlMaskEffect& GetMaskEffect() const { return m_maskEffect; }
            GlMaskEffect& GetMaskEffect() { return m_maskEffect; } 

            /**
             *  Retrieves the blend effect.
             *  @return     The blend effect.
             */
            const BlendEffect& GetBlendEffect() const { return m_blendEffect; }
            BlendEffect& GetBlendEffect() { return m_blendEffect; }

            /**
            *  Gets a valid BrushEffect2D pointer if the effect is a brush effect or
            *  if it is a combined effect and has a brush effect, otherwise 0.
            *  @return     A valid BrushEffect2D pointer if the effect is a brush effect or
            *              if it is a combined effect and has a brush effect, otherwise 0.
            */
            virtual BrushEffect2D* GetBrushEffect2D() { return &m_textBrush; }

            // overrides Effect2D::GetInplaceEffect2DCount
            virtual UInt8 GetInPlaceEffect2DCount() const { return 1; }

            // overrides Effect2D::GetInPlaceEffect2D
            virtual InPlaceEffect2D* GetInPlaceEffect2D(UInt8 index) { return (index == 0)? &m_maskEffect : 0; }

            /**
             *  Gets a valid BlendEffect2D pointer if the effect is a blend effect or
             *  if it is a combined effect and has a blend effect, otherwise 0.
             *  @return     A valid BlendEffect2D pointer if the effect is a blend effect or
             *              if it is a combined effect and has a blend effect, otherwise 0.
             */
            virtual BlendEffect2D* GetBlendEffect2D() { return &m_blendEffect; }

            /**
            *  Retrieves the current bounding rectangle.
            *  @param boundingRectangle This value will hold the current bounding rectangle after function-call.
            */
            virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const { m_textBrush.GetBoundingRectangle(boundingRectangle); };

            /**
             *  Uploads all associated resources.
             *  @return true if successful.
             */
            virtual bool Upload();

            /**
             *   Check whether all the resources are uploaded.
             *   @return true if all the resources are uploaded, false otherwise.
             */
            virtual bool IsUploaded() const;

            /**
             *  Unloads all associated resources.
             *  @return true if successful.
             */
            virtual bool Unload();

            /**
             *  Uploads all resources that changed since last Upload/Update.
             *  Unloads resources removed from the effect, that were previously uploaded
             *  by an Upload/Update.
             *  This should only be called on Uploaded effects. (otherwise, call Upload.)
             *  @return true if successful.
             */
            virtual bool Update();

            // overrides Effect2D::Clone
            virtual Effect2D::SharedPointer Clone() const;

            /// @cond excluded from doxygen
            CdaEffect2DCombinedDef(Candera::GlTextBrushMaskBlend, GlTextBrushMaskBlend, EFFECT2D_TYPE_BRUSH, EFFECT2D_TYPE_BLEND)
                CdaDeprecated()
                CdaEffect2DProperties()

                    CdaEffect2DImportProperties(TextBrush, Candera::TextBrush, m_textBrush)
                    CdaEffect2DImportProperties(GlMaskEffect, Candera::GlMaskEffect, m_maskEffect)
                    CdaEffect2DImportProperties(BlendEffect, Candera::BlendEffect, m_blendEffect)

                CdaEffect2DPropertiesEnd()
            CdaEffect2DDefEnd()
            /// @endcond

        protected:
            // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
            GlTextBrushMaskBlend();
            GlTextBrushMaskBlend(const GlTextBrushMaskBlend& rhs);
            GlTextBrushMaskBlend& operator = (const GlTextBrushMaskBlend& rhs);

            /**
             *  Triggers the rendering for the effect.
             *  @param  input       Defines the input for the effect. The first effect inside the effect list NO input. The input for the second
             *                      effect in the list must be set to the output of the first effect and so on.
             *  @param  inputArea   Area of the input that should be used as input. This should by supplied through the ouputArea parameter of the last effect.
             *  @param  transform   Defines the transformation matrix for the effect.
             *  @param  node        Node this effect is referenced to, containing transformation matrix for the effect.
             *  @param  output      Defines the output for the effect. The last effect inside the effect list must be the context of the render target.
             *  @param  outputArea  Output parameter through which the effect supplies the area of the output that was modified.
             */
            virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

        private:
            /*
             *  Use class wrapper to forward protected members.
             */
            class TextBrushWrapper : public TextBrush {} m_textBrush;
            class MaskEffectWrapper : public GlMaskEffect {} m_maskEffect;
            class BlendEffectWrapper : public BlendEffect {} m_blendEffect;
    };


}   // namespace Candera

#endif  // CANDERA_GL_TEXT_BRUSH_MASK_BLEND_H
