//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlTextBrushMaskBlend.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(GlTextBrushMaskBlend, CombinedEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GlTextBrushMaskBlend::GlTextBrushMaskBlend():
        Base()
    {
    }

    GlTextBrushMaskBlend::GlTextBrushMaskBlend(const GlTextBrushMaskBlend& rhs) :
        Base(rhs),
        m_textBrush(rhs.m_textBrush),
        m_maskEffect(rhs.m_maskEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlTextBrushMaskBlend::~GlTextBrushMaskBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlTextBrushMaskBlend::SharedPointer GlTextBrushMaskBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlTextBrushMaskBlend* brush = FEATSTD_NEW(GlTextBrushMaskBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlTextBrushMaskBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlTextBrushMaskBlend::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform,
                                     const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "GlTextBrushMaskBlend"));
        FEATSTD_UNUSED(input);
        FEATSTD_UNUSED(inputArea);

        if (m_textBrush.Text() != 0) {
            m_blendEffect.ActivateBlend(output);
            m_textBrush.ActivateColor(node, output);
            m_maskEffect.ActivateMask(node, output);
            m_textBrush.RenderText(transform, output, outputArea);
            m_maskEffect.DeactivateMask(output);
            m_textBrush.DeactivateColor(output);
            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool GlTextBrushMaskBlend::Upload()
    {
        bool success = m_textBrush.Upload();
        success = m_maskEffect.Upload() && success;
        success = m_blendEffect.Upload() && success;
        return success;
    }

    /******************************************************************************
     *  IsUploaded
     ******************************************************************************/
    bool GlTextBrushMaskBlend::IsUploaded() const
    {
        return m_textBrush.IsUploaded() && m_maskEffect.IsUploaded() && m_blendEffect.IsUploaded();
    }

    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool GlTextBrushMaskBlend::Unload()
    {
        bool success = m_textBrush.Unload();
        success = m_maskEffect.Unload() && success;
        success = m_blendEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool GlTextBrushMaskBlend::Update()
    {
        bool success = m_textBrush.Update();
        success = m_maskEffect.Update() && success;
        success = m_blendEffect.Update() && success;
        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlTextBrushMaskBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlTextBrushMaskBlend)(*this));
    }


}   // namespace Candera
